#ifndef FDOSMLPDATAPROPERTYDEFINITION_H
#define FDOSMLPDATAPROPERTYDEFINITION_H		1
/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <Sm/Lp/SimplePropertyDefinition.h>
#include <Sm/Ph/Column.h>

// DataPropertyDefinition derives from PropertyDefinition and represents simple
// values or collections of simple values. DataPropertyDefinitions can take on
// any of the data types listed in the DataType enumeration.
class FdoSmLpDataPropertyDefinition : public FdoSmLpSimplePropertyDefinition
{
public:
    
    /// Gets the DataType of this property.
    FdoDataType GetDataType() const;

    /// Gets the length of a String, BLOB, or CLOB data property. This value
    /// does not apply to any other DataTypes.
    int GetLength() const;

    /// Gets the precision (total number of digits) of a Decimal data property.
    /// This value does not apply to any other DataTypes.
    int GetPrecision() const;

    /// Gets the scale (number of digits to the right of the decimal point) of a
    /// Decimal data property. This value does not apply to any other DataTypes.
    int GetScale() const;

    /// If the property is an identity property, this function returns its position
    /// within the class's identity property collection.
    /// Otherwise, it returns 0.
	int GetIdPosition() const;

    /// Gets the default value for this property. The default value is used when
    /// an instance of the containing class is created without specifing a value
    /// for this property.
    FdoPtr<FdoDataValue> GetDefaultValue() const;

    // Gets default value as a string (in FDO Expression format).
    FdoStringP GetDefaultValueString() const;

    /// Gets whether values for this property are autogenerated.
    bool GetIsAutoGenerated() const;

    /// Gets Revision Number column
    bool GetIsRevisionNumber() const;

    /// Gets the name of sequence to allocate a value for this property
    FdoString* GetSequenceName() const;

    /// Returns the data property types for which value autogeneration is
    /// supported by the RDBMS Provider.
    virtual FdoDataType* GetSupportedAutoGeneratedTypes(FdoInt32& length);

    /// Get the type of property ( object, data, geometric, etc. ) that
    /// this object represents.
	virtual FdoPropertyType GetPropertyType() const;

    /// Get the Schema Mappings for this data property
	//
    /// Parameters:
    /// 	bIncludeDefaults - 
    ///          true: include all schema mappings.
    ///          false: include only non-default mappings.
    /// Returns:
    ///      The Schema Mappings.
    ///      NULL if there are no mappings.
    virtual FdoPhysicalPropertyMappingP GetSchemaMappings( bool bIncludeDefaults ) const
    {
        return (FdoPhysicalPropertyMapping*) NULL;
    }

    /// Makes this property an inherited property.
	//
    /// Parameters:
    /// 	pBaseProp: the property to inherit from.
	virtual void SetInherited( const FdoSmLpPropertyDefinition* pBaseProp );
/*
    /// Inherit this property to a subclass. Creates a new instance of this 
    /// property, which can be attached to the subclass.
	//
    /// Parameters:
    /// 	pSubClass: inherit to this subclass. Must be a subclass of this
    /// 		property's defining class.
	virtual FdoSmLpPropertyDefinition* CreateInherited( FdoSmLpClassDefinition* pSubClass ) const;

    /// Create a non-inherited copy of this property. 
	//
    /// Parameters:
    /// 	pTargetClass: The class that will contain the new property.
    /// 	logicalName: The name for the new property.
    /// 		If L"" then the new property has the same name as this
    /// 		property.
    /// 	physicalName: the name of the new property's column.
    /// 		If L"" then the column is the same as the property name.
    ///      pPropOverrides: Fdo to physical mapping overrides for created property.
	virtual FdoSmLpPropertyDefinition* CreateCopy( 
        FdoSmLpClassDefinition* pTargetClass, 
        FdoStringP logicalName, 
        FdoStringP physicalName,
        FdoOracleOvPropertyDefinition* pPropOverrides
    ) const;
*/
    /// Convenience function for casting a property to a data property.
    /// Returns NULL if the property is not a data property.
	static const FdoSmLpDataPropertyDefinition* Cast( const FdoSmLpPropertyDefinition* src )
	{
		return( src ? ( src->GetPropertyType() == FdoPropertyType_DataProperty ? 
						(const FdoSmLpDataPropertyDefinition*) src : NULL
					  ) : NULL );
	}

    /// When this is a ClassName or SchemaName base property and the corresponding
    /// column can't be found ( happens for non-feature classes ). This function
    /// sets a literal value for the property by setting the default value.
	//
    /// When it is a ClassName property then defaultValue is set to the name of 
    /// the containing class ( this works as long as multiple classes are never
    /// kept in the same table ). 
	//
    /// When it is a SchemaName property then defaultValue is set to the name of 
    /// the containing schema.
	//
    /// Parameters:
    /// 	bOverrideColumn:
    /// 		true: set default value even when this property has a column.
    /// 		false: don't set it if it has a column.
	void SetDefaultValue(bool bOverrideColumn = false );

	/// Set the property default value
	void SetDefaultValue(FdoPtr<FdoDataValue> value);

    /// Set the 1-based position of this property within the identity properties of its class.
    /// If position is 0 then this is not an identity property.
	void SetIdPosition(int idPosition);

    /// Change the autogenerated setting.
    void SetIsAutoGenerated( bool isAutoGenerated )
    {
        mIsAutoGenerated = isAutoGenerated;
    }

    void SetIsRevisionNumber( bool isRevisionNumber )
    {
        mIsRevisionNumber = isRevisionNumber;
    }

    /// Copy updates from corresponding FDO property. 
    //
    /// Parameters
    ///      pFdoProp: the FDO element.
    ///      elementState: the modification state for this data property.
    ///      pPropOverrides: Fdo to physical mapping overrides.
    ///      bIgnoreStates: true if the element state on the FDO element is 
    ///          to be ignored.
	virtual void Update(
        FdoPropertyDefinition* pFdoProp,
        FdoSchemaElementState elementState,
        FdoPhysicalPropertyMapping* pPropOverrides,
        bool bIgnoreStates
    ); 

    /// Synchronize this property and its column. Creates a column if the property does
    /// not have one. 
    //
    /// Parameters:
    //
    ///      bRollbackOnly - 
    ///          true - synchronize only if a change to this property has been rolled back since
    ///              the previous synchronization.
    ///          false - always synchronize this property.
	virtual void SynchPhysical(bool bRollbackOnly = true);

    /// Post outstanding modifications to the current database.
	virtual void Commit(bool fromParent = false );

    /// Name of ClassName base property
	static const FdoStringP mClassNamePropName;

    /// Name of SchemaName base property
	static const FdoStringP mSchemaNamePropName;

    /// Serialize this property to an XML file.
    /// Mainly for unit testing.
	virtual void XMLSerialize( FILE* xmlFp, int ref ) const;


    virtual FdoSchemaExceptionP Errors2Exception( FdoSchemaException* pFirstException = NULL ) const;

    /// Constructs a new column for this data property.
    //
    /// Parameters:
    /// 	dbObject: put the column in this database object.
    /// 	columnName: the name for the new column.
	virtual FdoSmPhColumnP NewColumn( FdoSmPhDbObjectP dbObject, FdoStringP columnName );

    virtual FdoSmPhColumnP NewColumn( 
        FdoSmPhDbObjectP dbObject, 
        FdoStringP columnName,
        bool nullable,
        FdoStringP rootColumnName
    );

protected:
    /// unused constructor needed only to build on Linux
    FdoSmLpDataPropertyDefinition() {}

    FdoSmLpDataPropertyDefinition(FdoSmPhClassPropertyReaderP propReader, FdoSmLpClassDefinition* parent);

    FdoSmLpDataPropertyDefinition(
        FdoDataPropertyDefinition* pFdoProp, 
        bool bIgnoreStates,
        FdoSmLpClassDefinition* parent
    );

    /// Constructs an instance of a DataPropertyDefinition from the given parameters.
    /// Used mainly by CreateInherited() and CreateCopy().
    //
    /// Parameters:
    /// 	pBaseProperty: initialize the new property from this base property.
    /// 	pTargetClass: the class that will contain the new property.
    /// 	logicalName: The name for the new property.
    /// 		If L"" then the new property has the same name as this
    /// 		property.
    /// 	physicalName: the name of the new property's column.
    /// 		If L"" then the column is the same as the base property column name.
    /// 	bInherit:
    /// 		True: the new property inherits from the base property.
    /// 		False: the new property is just a copy of the base property.
    ///      pPropOverrides: Fdo to physical mapping overrides for created property.
	FdoSmLpDataPropertyDefinition(
		FdoPtr<FdoSmLpDataPropertyDefinition> pBaseProperty, 
		FdoSmLpClassDefinition* pTargetClass, 
		FdoStringP logicalName, 
		FdoStringP physicalName, 
		bool bInherit,
        FdoPhysicalPropertyMapping* pPropOverrides = NULL
	);

    FdoStringP Get_DefaultValueString() const;

    void SetSequenceName( FdoString* sequenceName )
    {
        mSequenceName = sequenceName;
    }

    /// Finalize this property by generating its column.
    virtual void Finalize();

    /// Check if autogenerated values are supported for this type of data property.
    /// Generate an error if not supported.
    virtual void VldAutoGenerated();

	void AddColNameChangeError( FdoStringP newColName );

    virtual void XMLSerializeSequence( FILE* xmlFp, int ref ) const;

    FdoPtr<FdoDataValue> ParseDefaultValue( FdoStringP defaultStr );

private:
	void AddDataTypeChangeError( FdoDataType newType );
	void AddNullableChangeError();
	void AddLengthChangeError( int newLength );
	void AddPrecisionChangeError( int newPrecision );
	void AddScaleChangeError( int newScale );
	void AddDefaultValueError( FdoString* defaultValue );
	void AddDefaultChangeError( FdoString* defaultValue );
    void AddNotNullBaseMappingError();
    void AddAutoGeneratedChangeError();
    void AddAutoGenDataTypeError();
    void AddRedefinedAutoGenError( const FdoSmLpDataPropertyDefinition* pBaseProp );

    int mLength;

    int mPrecision;

    int mScale;

	int mIdPosition;

    FdoPtr<FdoDataValue> mDefaultValue;

    bool mIsAutoGenerated;

    bool mIsRevisionNumber;

    FdoStringP mSequenceName;

    FdoDataType mDataType;
};

typedef FdoPtr<FdoSmLpDataPropertyDefinition> FdoSmLpDataPropertyP;

#endif


