#ifndef FDOSMERROR_H
#define FDOSMERROR_H		1
//
// Copyright (C) 2004-2006  Autodesk, Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of version 2.1 of the GNU Lesser
// General Public License as published by the Free Software Foundation.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#ifdef _WIN32
#pragma once
#endif

#include <Sm/SchemaElement.h>
#include <Sm/Collection.h>

enum FdoSmErrorType
{
    /// Tried to re-entrantly finalize a class
    FdoSmErrorType_ClassLoop,

    /// Class has a loop in its base class ancestry
	FdoSmErrorType_BaseClassLoop,

    /// Base class is not in the current database.
    FdoSmErrorType_BaseClassMissing,

    /// Missing required column
    FdoSmErrorType_ColumnMissing,

    /// Error generating the table mapping for an object property
    FdoSmErrorType_ObjectPropertyMapping,

    /// The type class for an object property also includes
    /// this object property.
    FdoSmErrorType_ObjectPropertyLoop,

    FdoSmErrorType_NoJoinPath,

    /// Errors involving the Data Property IsAutoGenerated setting.
    FdoSmErrorType_AutoGen,

    /// Errors involving a spatial context mismatch.
    FdoSmErrorType_SpatialMismatch,

    /// Other errors.
    FdoSmErrorType_Other
};

class FdoSmError : public FdoSmSchemaElement
{
public:
    /// Create an error object.
	//
    /// Parameters:
    /// 	errorType: the general type of error
    /// 	description: detailed description of the error
	FdoSmError(FdoSmErrorType errorType, FdoSchemaExceptionP error);

	~FdoSmError(void);

    /// Gets the general type of error
	FdoSmErrorType GetType(void) const;

    /// Convert this error into the corresponding exception. Returns the 
    /// resulting exception
    //
    /// Parameters:
    /// 	pPrevException: tack this exception onto the resulting exception,
    ///      as the cause.
    FdoSchemaExceptionP CreateException( FdoSchemaException* pPrevException ) const;

    /// Mainly for Unit Testing.
    /// Serializes this error element to the given XML file.
	virtual void XMLSerialize( FILE* xmlFp, int ref ) const;

    /// \brief
    /// Gets the exception message from the Schema Manager message catalogue.
    /// 
    /// \remarks
    /// In addition to the listed parameters, a flexible number of additional variables, as required, 
    /// can be used as input to format the exception message.
    /// 
    /// \param msgNum 
    /// Input the error message number
    /// \param defMsg 
    /// Input a text error message to use if the message number is not found
    /// \param file 
    /// Input file name where exception occurred
    /// \param line 
    /// Input the line number where exception occurred
    /// 
    /// \return
    /// Returns the localized message text
    /// 
    static FdoString* NLSGetMessage(FdoInt32 msgNum, const char* defMsg, const char* file, int line, ...);

    /// \brief
    /// Gets the exception message from the Schema Manager message catalogue.
    /// 
    /// \remarks
    /// In addition to the listed parameters, a flexible number of additional variables, as required, 
    /// can be used as input to format the exception message.
    /// 
    /// \param msgNum 
    /// Input the error message number
    /// \param defMsg 
    /// Input a text error message to use if the message number is not found
    /// 
    /// \return
    /// Returns the localized message text
    /// 
    static FdoString* NLSGetMessage(FdoInt32 msgNum, const char* defMsg, ...);

protected:
    /// unused constructor needed only to build Linux
    FdoSmError();

private:
	FdoSmErrorType      mType;
    FdoSchemaExceptionP mError;
};

/// \brief
/// FdoSmErrorP is a FdoPtr on FdoSmError, provided for convenience.
typedef FdoPtr<FdoSmError> FdoSmErrorP;

// Represents a collection of Errors.
class FdoSmErrorCollection : public FdoSmCollection<FdoSmError> 
{
public:

    FdoSmErrorCollection()
        : FdoSmCollection<FdoSmError>()
    {
    }

    /// Overload of GetItem causes GetItem on base class
    /// to disappear, so added wrapper around it.
	const FdoSmError* RefItem(FdoInt32 index) const
	{
		return FdoSmCollection<FdoSmError>::RefItem(index);
	}

    /// Get the first error of the given type
	const FdoSmError* RefItem( FdoSmErrorType errorType ) const
	{
		for ( int i = 0; i < GetCount(); i++ ) {
			const FdoSmError* pError = RefItem(i);

			if ( errorType ==  pError->GetType() )
				return(pError);
		}

		return NULL;
	}

    /// Add an error element
	int Add( FdoSmError* element )
	{
		return( FdoSmCollection<FdoSmError>::Add(element) );
	}

    /// Add a new error element from the given error type and 
    /// description.
	int Add( FdoSmErrorType errorType, FdoSchemaExceptionP error )
	{
		FdoSmErrorP pError = new FdoSmError( 
            errorType, 
            error
        ); 

		int ret = Add( pError );
		
		return( ret );
	}
};

/// \brief
/// FdoSmErrorsP is a FdoPtr on FdoSmErrorCollection, provided for convenience.
typedef FdoPtr<FdoSmErrorCollection> FdoSmErrorsP;

#endif


