/*
 * 
* Copyright (C) 2004-2006  Autodesk, Inc.
* 
* This library is free software; you can redistribute it and/or
* modify it under the terms of version 2.1 of the GNU Lesser
* General Public License as published by the Free Software Foundation.
* 
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
* 
 */

#include "stdafx.h"

ShpSchemaCapabilities::ShpSchemaCapabilities ()
{
}

ShpSchemaCapabilities::~ShpSchemaCapabilities ()
{
}

void ShpSchemaCapabilities::Dispose ()
{
    delete this;
}

/// <summary>Returns an array of the FdoClassType objects the feature provider supports.</summary>
/// <param name="length">Output the number of class types.</param> 
/// <returns>Returns the list of class types.</returns> 
FdoClassType* ShpSchemaCapabilities::GetClassTypes (FdoInt32& length)
{
    static FdoClassType classes[] =
    {
        FdoClassType_FeatureClass,
        FdoClassType_Class
    };

    length = sizeof (classes) / sizeof (FdoClassType);

    return (classes);
}

/// <summary>Returns an array of the FdoDataType objects the feature provider supports.</summary>
/// <param name="length">Output the number of data types.</param> 
/// <returns>Returns the list of data types</returns> 
FdoDataType* ShpSchemaCapabilities::GetDataTypes (FdoInt32& length)
{
    static FdoDataType types[] =
    {
        FdoDataType_Decimal,
		FdoDataType_Int16,
		FdoDataType_Int32,
        FdoDataType_Int64,
        FdoDataType_String,
        FdoDataType_DateTime,
        FdoDataType_Boolean
    };

    length = sizeof (types) / sizeof (FdoDataType);

    return (types);
}


/// <summary>Determines if the feature provider supports inheritance.</summary>
/// <returns>Returns true if the feature provider supports inheritance</returns> 
bool ShpSchemaCapabilities::SupportsInheritance ()
{
    return (false);
}


/// <summary>Determines if the provider supports multiple schemas. Returns false if the provider supports only a single schema.</summary>
/// <returns>Returns true if the provider supports multiple schemas.</returns> 
bool ShpSchemaCapabilities::SupportsMultipleSchemas ()
{
    return (false);
}


/// <summary>Determines if the provider supports object properties. Returns false if the provider supports only data and geometric properties.</summary>
/// <returns>Returns true if the provider supports object properties.</returns> 
bool ShpSchemaCapabilities::SupportsObjectProperties ()
{
    return (false);
}


/// <summary>Determines if the provider supports Schema Mapping Overrides 
/// (overrides to the default rules for mapping Feature Schemas to provider-specific
/// physical schemas)</summary>
/// <returns>Returns true if the provider supports Schema Overrides</returns> 
bool ShpSchemaCapabilities::SupportsSchemaOverrides ()
{
    return (true);
}


/// <summary>Determines if the provider supports association property.</summary>
/// <returns>Returns true if the provider support the association property.</returns> 
bool ShpSchemaCapabilities::SupportsAssociationProperties (void)
{
    return (false);
}

/// <summary>Determines if the provider supports the network model.</summary>
/// <returns>Returns true if the provider supports the network model.</returns> 
bool ShpSchemaCapabilities::SupportsNetworkModel (void)
{
    return (false);
}

/// <summary>Determines if the provider can auto-generate unique ID values for a class.</summary>
/// <returns>Returns true if the provider supports id autogeneration.</returns> 
bool ShpSchemaCapabilities::SupportsAutoIdGeneration ()
{
    return (true);
}

/// <summary>Determines if the provider can auto-generate unique ID values that are unique for the entire datastore, rather than just for a particular class.</summary>
/// <returns>Returns true if the provider supports id autogeneration unique across an datastore.</returns> 
bool ShpSchemaCapabilities::SupportsDataStoreScopeUniqueIdGeneration ()
{
    return (false);
}

/// <summary>Returns an array of the FdoDataType objects the feature provider supports for auto-generation.</summary>
/// <param name="length">Output the number of data types.</param> 
/// <returns>Returns the list of data types</returns> 
FdoDataType* ShpSchemaCapabilities::GetSupportedAutoGeneratedTypes (FdoInt32& length)
{
    static FdoDataType autogen_types[] =
    {
        FdoDataType_Int32,
    };

    length = sizeof (autogen_types) / sizeof (FdoDataType);

    return (autogen_types);
}

/// <summary>Determines if the provider support update a schema through the ApplySchema command.
/// If False, then an application can only create a schema once for the provider.
/// They can use the ApplySchema command once for a given datastore.
/// Once the schema is defined, it cannot be changed.
/// If True, then the provider supports the ability to modify or add to the schema after it is defined initially.</summary>
/// <returns>Returns true if the provider supports the update of the schema.</returns>
bool ShpSchemaCapabilities::SupportsSchemaModification ()
{
    return (true);
}

/// <summary>Determines if the provider supports inclusive value range constraints.</summary>
/// <returns>Returns true if the provider supports inclusive value range constraints.</returns> 
bool ShpSchemaCapabilities::SupportsInclusiveValueRangeConstraints()
{
    return false;
}

/// <summary>Determines if the provider supports exclusive value range constraints.</summary>
/// <returns>Returns true if the provider supports exclusive value range constraints.</returns>
bool ShpSchemaCapabilities::SupportsExclusiveValueRangeConstraints()
{
    return false;
}
/// <summary>Determines if the provider supports value constraints list.</summary>
/// <returns>Returns true if the provider supports value constraints list.</returns>
bool ShpSchemaCapabilities::SupportsValueConstraintsList()
{
    return false;
}

/// <summary>Determines if the provider supports nullable value constraint.</summary>
/// <returns>Returns true if the provider supports nullable value constraint.</returns>
bool ShpSchemaCapabilities::SupportsNullValueConstraints()
{
    return true;
}

/// <summary>Determines if the provider supports unique value constraint.</summary>
/// <returns>Returns true if the provider supports unique value constraint.</returns>
bool ShpSchemaCapabilities::SupportsUniqueValueConstraints()
{
    return false;
}

/// <summary>Determines if the provider supports composite unique value constraint.</summary>
/// <returns>Returns true if the provider supports composite unique value constraint.</returns>
bool ShpSchemaCapabilities::SupportsCompositeUniqueValueConstraints()
{
    return false;
}

/// <summary>Determines if the provider supports multiple identity properties per class.</summary>
/// <returns>Returns true if the provider supports multiple identity properties per class.</returns>
bool ShpSchemaCapabilities::SupportsCompositeId()
{
    return false;
}

FdoInt32 ShpSchemaCapabilities::GetNameSizeLimit (FdoSchemaElementNameType nameType)
{
    FdoInt32 classLimit;

#ifdef _WIN32
    classLimit = 162; // 166 - 4 for the file extension.
#else
    classLimit = 251; // 255 - 4 for the file extension.
#endif

    switch (nameType)
    {
        case FdoSchemaElementNameType_Datastore:   return -1;           // SHP does not generate data stores
        case FdoSchemaElementNameType_Schema:      return 7;            // == wcslen("default")
        case FdoSchemaElementNameType_Class:       return classLimit;
        case FdoSchemaElementNameType_Property:    return 11;           // == max dbf column name size length (see DBF spec)
        case FdoSchemaElementNameType_Description: return 0;            // DBF doesn't store column descriptions
    }
    return -1;
}

FdoString* ShpSchemaCapabilities::GetReservedCharactersForName()
{
    return L".:";  // NOTE: DBF column names are always 100% ascii, but it's difficult to list all non-ascii characters
}

FdoInt64 ShpSchemaCapabilities::GetMaximumDataValueLength (FdoDataType dataType)
{
    switch (dataType)
    {
        case FdoDataType_String:   return (FdoInt64)255;
        case FdoDataType_BLOB:     return (FdoInt64)-1;
        case FdoDataType_CLOB:     return (FdoInt64)-1;
        case FdoDataType_Decimal:  return (FdoInt64)255;
        case FdoDataType_Boolean:  return (FdoInt64)sizeof(FdoBoolean);
        case FdoDataType_Byte:     return (FdoInt64)sizeof(FdoByte);
        case FdoDataType_DateTime: return (FdoInt64)sizeof(FdoDateTime);
        case FdoDataType_Double:   return (FdoInt64)sizeof(FdoDouble);
        case FdoDataType_Int16:    return (FdoInt64)sizeof(FdoInt16);
        case FdoDataType_Int32:    return (FdoInt64)sizeof(FdoInt32);
        case FdoDataType_Int64:    return (FdoInt64)sizeof(FdoInt64);
        case FdoDataType_Single:   return (FdoInt64)sizeof(FdoFloat);
    }
    return (FdoInt64)-1;
}

FdoInt32 ShpSchemaCapabilities::GetMaximumDecimalPrecision()
{
    return 255;
}

FdoInt32 ShpSchemaCapabilities::GetMaximumDecimalScale()
{
    return 255;
}

FdoDataType* ShpSchemaCapabilities::GetSupportedIdentityPropertyTypes(FdoInt32& length)
{
    length = 1;
    static FdoDataType supportedIdentityTypes[1];
    supportedIdentityTypes[0] = FdoDataType_Int32;
    return supportedIdentityTypes;
}

bool ShpSchemaCapabilities::SupportsDefaultValue()
{
    return false;
}
