/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "Pch.h"
#include "MySqlFdoApplySchemaTest.h"
#include "ConnectionUtil.h"
#include "UnitTestUtil.h"
#include "../SchemaMgr/Ph/Owner.h"
#include "../MySQL/SchemaMgr/Ph/Owner.h"
#include "../MySQL/SchemaMgr/Ph/CharacterSet.h"

CPPUNIT_TEST_SUITE_REGISTRATION( MySqlFdoApplySchemaTest );
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION( MySqlFdoApplySchemaTest, "FdoApplySchemaTest");

FdoString*    MySqlFdoApplySchemaTest::DB_NAME_CHARSIZE_SUFFIX = L"_apply_charsize";


void MySqlFdoApplySchemaTest::set_provider()
{
	UnitTestUtil::SetProvider( "MySql" );
	mIsLowerDatastoreName = true;
}

void MySqlFdoApplySchemaTest::TestCharSize ()
{
	FdoPtr<FdoIConnection> connection;
    StaticConnection* staticConn = NULL;

    try {
        // delete, re-create and open the datastore
		printf( "Initializing Connection ... \n" );
		connection = UnitTestUtil::CreateConnection(
			true,
			true,
            DB_NAME_CHARSIZE_SUFFIX,
            0,
            NULL,
            0
		);

	    FdoFeatureSchemaP schema = FdoFeatureSchema::Create( L"AllDefaultOver", L"" );
        FdoClassesP classes = schema->GetClasses();

        FdoFeatureClassP classDef = FdoFeatureClass::Create( L"AllDefaultOver", L"" );
    	classDef->SetIsAbstract(false);
        classes->Add(classDef);
        FdoDataPropertiesP idProps = classDef->GetIdentityProperties();
        FdoPropertiesP props = classDef->GetProperties();

    	FdoDataPropertyP dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	dataProp->SetDataType( FdoDataType_Int64 );
	    dataProp->SetNullable(false);
        dataProp->SetIsAutoGenerated(true);
	    props->Add(dataProp);
        idProps->Add(dataProp);

    	dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(65509);
        dataProp->SetDefaultValue(L"ABCD");
        props->Add(dataProp);

    	FdoPtr<FdoIApplySchema>  pCmd = (FdoIApplySchema*) connection->CreateCommand(FdoCommandType_ApplySchema);
        pCmd->SetFeatureSchema(schema);

        bool failed = false;
        try 
        {
            pCmd->Execute();
        }
        catch ( FdoException* ex ) 
        {
            failed = true;
            UnitTestUtil::PrintException(ex);
            FDO_SAFE_RELEASE(ex);
        }

        CPPUNIT_ASSERT(failed);

	    schema = FdoFeatureSchema::Create( L"SomeDefaultOver", L"" );
        classes = schema->GetClasses();

        classDef = FdoFeatureClass::Create( L"SomeDefaultOver", L"" );
    	classDef->SetIsAbstract(false);
        classes->Add(classDef);
        idProps = classDef->GetIdentityProperties();
        props = classDef->GetProperties();

    	dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	dataProp->SetDataType( FdoDataType_Int64 );
	    dataProp->SetNullable(false);
        dataProp->SetIsAutoGenerated(true);
	    props->Add(dataProp);
        idProps->Add(dataProp);

    	dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(10000);
        dataProp->SetDefaultValue(L"ABCD");
        props->Add(dataProp);

    	dataProp = FdoDataPropertyDefinition::Create( L"Col2", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(35505);
        dataProp->SetDefaultValue(L"EFGH");
        props->Add(dataProp);

    	dataProp = FdoDataPropertyDefinition::Create( L"Col3", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(20000);
        dataProp->SetDefaultValue(L"IJKL");
        props->Add(dataProp);

    	pCmd = (FdoIApplySchema*) connection->CreateCommand(FdoCommandType_ApplySchema);
        pCmd->SetFeatureSchema(schema);

        failed = false;
        try 
        {
            pCmd->Execute();
        }
        catch ( FdoException* ex ) 
        {
            failed = true;
            UnitTestUtil::PrintException(ex);
            FDO_SAFE_RELEASE(ex);
        }

        CPPUNIT_ASSERT(failed);

        FdoDictionaryP expected = FdoDictionary::Create();

        staticConn = UnitTestUtil::NewStaticConnection();
        staticConn->connect();
        staticConn->SetSchema( DB_NAME_CHARSIZE_SUFFIX );

        FdoSchemaManagerP mgr = staticConn->CreateSchemaManager();
        FdoSmPhMgrP ph = mgr->GetPhysicalSchema();
        FdoSmPhOwnerP owner = ph->GetOwner();
        FdoInt32 charLen = (FdoInt32)(owner->SmartCast<FdoSmPhMySqlOwner>()->GetCharacterSet()->SmartCast<FdoSmPhMySqlCharacterSet>()->GetCharLen());

	    schema = FdoFeatureSchema::Create( L"CharSize", L"" );
        classes = schema->GetClasses();
        FdoGeometricPropertyP geomProp;

        if ( charLen == 1 ) 
        {
            classDef = FdoFeatureClass::Create( L"AllOverWDefault", L"" );
    	    classDef->SetIsAbstract(false);
            classes->Add(classDef);
            idProps = classDef->GetIdentityProperties();
            props = classDef->GetProperties();

    	    dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	    dataProp->SetDataType( FdoDataType_Int64 );
	        dataProp->SetNullable(false);
            dataProp->SetIsAutoGenerated(true);
	        props->Add(dataProp);
            idProps->Add(dataProp);

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(1000);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"AllOverWDefault:Col1", L"1000:TEXT");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col2", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(35469);
            dataProp->SetDefaultValue(L"EFGH");
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"AllOverWDefault:Col2", L"35469:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col3", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(1);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"AllOverWDefault:Col3", L"1:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col4", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(30000);
            dataProp->SetDefaultValue(L"EFGH");
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"AllOverWDefault:Col4", L"30000:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col5", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(11);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"AllOverWDefault:Col5", L"11:TEXT");

            FdoGeometricPropertyP geomProp = FdoGeometricPropertyDefinition::Create( L"Geometry", L"" );
            props->Add(geomProp);
        }

        classDef = FdoFeatureClass::Create( L"AllOver", L"" );
    	classDef->SetIsAbstract(false);
        classes->Add(classDef);
        idProps = classDef->GetIdentityProperties();
        props = classDef->GetProperties();

    	dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	dataProp->SetDataType( FdoDataType_Int64 );
	    dataProp->SetNullable(false);
        dataProp->SetIsAutoGenerated(true);
	    props->Add(dataProp);
        idProps->Add(dataProp);

    	dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(49161);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllOver:Col1", L"49161:TEXT");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col2", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(65532);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllOver:Col2", L"65532:TEXT");

        geomProp = FdoGeometricPropertyDefinition::Create( L"Geometry", L"" );
        props->Add(geomProp);

        if ( charLen == 1 ) 
        {
            classDef = FdoFeatureClass::Create( L"SomeOverWDefault1", L"" );
    	    classDef->SetIsAbstract(false);
            classes->Add(classDef);
            idProps = classDef->GetIdentityProperties();
            props = classDef->GetProperties();

    	    dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	    dataProp->SetDataType( FdoDataType_Int64 );
	        dataProp->SetNullable(false);
            dataProp->SetIsAutoGenerated(true);
	        props->Add(dataProp);
            idProps->Add(dataProp);

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(5000);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault1:Col1", L"5000:TEXT");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col2", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(30000);
            dataProp->SetDefaultValue(L"EFGH");
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault1:Col2", L"30000:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col3", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(1000);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault1:Col3", L"1000:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col4", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(30000);
            dataProp->SetDefaultValue(L"EFGH");
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault1:Col4", L"30000:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col5", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(8000);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault1:Col5", L"8000:TEXT");

            geomProp = FdoGeometricPropertyDefinition::Create( L"Geometry", L"" );
            props->Add(geomProp);

            classDef = FdoFeatureClass::Create( L"SomeOverWDefault2", L"" );
    	    classDef->SetIsAbstract(false);
            classes->Add(classDef);
            idProps = classDef->GetIdentityProperties();
            props = classDef->GetProperties();

    	    dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	    dataProp->SetDataType( FdoDataType_Int64 );
	        dataProp->SetNullable(false);
            dataProp->SetIsAutoGenerated(true);
	        props->Add(dataProp);
            idProps->Add(dataProp);

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(500);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault2:Col1", L"500:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col2", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(30000);
            dataProp->SetDefaultValue(L"EFGH");
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault2:Col2", L"30000:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col3", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(1000);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault2:Col3", L"1000:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col4", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(500);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault2:Col4", L"500:VARCHAR");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col5", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(40000);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault2:Col5", L"40000:TEXT");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col6", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(28000);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault2:Col6", L"28000:TEXT");

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col7", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(5000);
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"SomeOverWDefault2:Col7", L"5000:VARCHAR");

            geomProp = FdoGeometricPropertyDefinition::Create( L"Geometry", L"" );
            props->Add(geomProp);
        }

        classDef = FdoFeatureClass::Create( L"SomeOver", L"" );
    	classDef->SetIsAbstract(false);
        classes->Add(classDef);
        idProps = classDef->GetIdentityProperties();
        props = classDef->GetProperties();

    	dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	dataProp->SetDataType( FdoDataType_Int64 );
	    dataProp->SetNullable(false);
        dataProp->SetIsAutoGenerated(true);
	    props->Add(dataProp);
        idProps->Add(dataProp);

    	dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(16000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col1", L"16000:TEXT");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col2", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(104);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col2", L"104:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col3", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(15000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col3", L"15000:TEXT");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col4", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(200);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col4", L"200:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col5", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(12000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col5", L"12000:TEXT");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col6", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(500);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col6", L"500:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col7", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(10000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col7", L"10000:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col8", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(7000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col8", L"7000:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col9", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(10000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col9", L"10000:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col10", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(9000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col10", L"9000:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col11", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(10000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col11", L"10000:VARCHAR");

        geomProp = FdoGeometricPropertyDefinition::Create( L"Geometry", L"" );
        props->Add(geomProp);

        if ( charLen == 1 ) 
        {
            classDef = FdoFeatureClass::Create( L"AllUnderDefaultOnly", L"" );
    	    classDef->SetIsAbstract(false);
            classes->Add(classDef);
            idProps = classDef->GetIdentityProperties();
            props = classDef->GetProperties();

    	    dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	    dataProp->SetDataType( FdoDataType_Int64 );
	        dataProp->SetNullable(false);
            dataProp->SetIsAutoGenerated(true);
	        props->Add(dataProp);
            idProps->Add(dataProp);

    	    dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	    dataProp->SetDataType( FdoDataType_String );
    	    dataProp->SetLength(65508);
            dataProp->SetDefaultValue(L"ABCD");
            props->Add(dataProp);
            TestCharSize_AddToExpected(expected, L"AllUnderDefaultOnly:Col1", L"65508:VARCHAR");
        }

        classDef = FdoFeatureClass::Create( L"AllUnder1", L"" );
    	classDef->SetIsAbstract(false);
        classes->Add(classDef);
        idProps = classDef->GetIdentityProperties();
        props = classDef->GetProperties();

    	dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	dataProp->SetDataType( FdoDataType_Int64 );
	    dataProp->SetNullable(false);
        dataProp->SetIsAutoGenerated(true);
	    props->Add(dataProp);
        idProps->Add(dataProp);

    	dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(750);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col1", L"750:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col2", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(375);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col2", L"375:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col3", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(1500);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col3", L"1500:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col4", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(182);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col4", L"182:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col5", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(3000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col5", L"3000:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col6", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(91);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col6", L"91:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col7", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(6000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col7", L"6000:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col8", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(45);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col8", L"45:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col9", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(12000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col9", L"12000:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col10", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(22);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col10", L"22:VARCHAR");

    	dataProp = FdoDataPropertyDefinition::Create( L"Col11", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(24000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder1:Col11", L"24000:VARCHAR");

        geomProp = FdoGeometricPropertyDefinition::Create( L"Geometry", L"" );
        props->Add(geomProp);

        classDef = FdoFeatureClass::Create( L"AllUnder2", L"" );
    	classDef->SetIsAbstract(false);
        classes->Add(classDef);
        idProps = classDef->GetIdentityProperties();
        props = classDef->GetProperties();

    	dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	dataProp->SetDataType( FdoDataType_Int64 );
	    dataProp->SetNullable(false);
        dataProp->SetIsAutoGenerated(true);
	    props->Add(dataProp);
        idProps->Add(dataProp);

    	dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(48700);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder2:Col1", L"48700:VARCHAR");

        geomProp = FdoGeometricPropertyDefinition::Create( L"Geometry", L"" );
        props->Add(geomProp);

        classDef = FdoFeatureClass::Create( L"AllUnder3", L"" );
    	classDef->SetIsAbstract(false);
        classes->Add(classDef);
        idProps = classDef->GetIdentityProperties();
        props = classDef->GetProperties();

    	dataProp = FdoDataPropertyDefinition::Create( L"FeatureId", L"" );
    	dataProp->SetDataType( FdoDataType_Int64 );
	    dataProp->SetNullable(false);
        dataProp->SetIsAutoGenerated(true);
	    props->Add(dataProp);
        idProps->Add(dataProp);

    	dataProp = FdoDataPropertyDefinition::Create( L"Col1", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(40000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder3:Col1", L"40000:VARCHAR");

        geomProp = FdoGeometricPropertyDefinition::Create( L"Geometry", L"" );
        props->Add(geomProp);

    	pCmd = (FdoIApplySchema*) connection->CreateCommand(FdoCommandType_ApplySchema);
        pCmd->SetFeatureSchema(schema);
        pCmd->Execute();

     	FdoPtr<FdoIDescribeSchema> pDescCmd = (FdoIDescribeSchema*) connection->CreateCommand(FdoCommandType_DescribeSchema);
    	pDescCmd->SetSchemaName( L"CharSize" );

    	FdoFeatureSchemasP schemas = pDescCmd->Execute();
        CPPUNIT_ASSERT(schemas->GetCount() == 1);
        FdoFeatureSchemaP schema2 = schemas->GetItem(0);

        mgr = staticConn->CreateSchemaManager();
        ph = mgr->GetPhysicalSchema();
        owner = ph->GetOwner();

        if ( charLen == 1 ) 
            TestCharSize_Validate(expected, schema2, owner);

        owner = NULL;
        ph = NULL;
        mgr = NULL;

        classes = schema2->GetClasses();

        FdoClassDefinitionP classDef2 = classes->FindItem( L"AllUnder3" );
        CPPUNIT_ASSERT(classDef2);
        props = classDef2->GetProperties();

        dataProp = FdoDataPropertyDefinition::Create( L"Col2", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(8500);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder3:Col2", L"8500:VARCHAR");

        dataProp = FdoDataPropertyDefinition::Create( L"Col3", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(9500);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"AllUnder3:Col3", L"9500:VARCHAR");

        classDef2 = classes->FindItem( L"SomeOver" );
        CPPUNIT_ASSERT(classDef2);
        props = classDef2->GetProperties();

        dataProp = FdoDataPropertyDefinition::Create( L"Col12", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(16000);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col12", L"16000:TEXT");

        dataProp = FdoDataPropertyDefinition::Create( L"Col13", L"" );
    	dataProp->SetDataType( FdoDataType_String );
    	dataProp->SetLength(13900);
        props->Add(dataProp);
        TestCharSize_AddToExpected(expected, L"SomeOver:Col13", L"13900:VARCHAR");

    	pCmd = (FdoIApplySchema*) connection->CreateCommand(FdoCommandType_ApplySchema);
        pCmd->SetFeatureSchema(schema2);
        pCmd->Execute();

     	pDescCmd = (FdoIDescribeSchema*) connection->CreateCommand(FdoCommandType_DescribeSchema);
    	pDescCmd->SetSchemaName( L"CharSize" );

    	schemas = pDescCmd->Execute();
        CPPUNIT_ASSERT(schemas->GetCount() == 1);
        FdoFeatureSchemaP schema3 = schemas->GetItem(0);

        mgr = staticConn->CreateSchemaManager();
        ph = mgr->GetPhysicalSchema();
        owner = ph->GetOwner();

        if ( charLen == 1 ) 
            TestCharSize_Validate(expected, schema3, owner);

        owner = NULL;
        ph = NULL;
        mgr = NULL;

        if ( charLen != 1 ) 
            CPPUNIT_FAIL ("Test appears to be Ok; However database character set is not Latin1 so validation was skipped");
    }
	catch ( FdoException* e ) 
	{
		try {
			if ( connection) connection->Close(); 
            if ( staticConn ) delete staticConn;
		}
		catch ( ... ) 
		{
		}
		UnitTestUtil::FailOnException( e );
	}
	catch ( CppUnit::Exception e ) 
	{
        try {
		    if (connection) connection->Close(); 
            if ( staticConn ) delete staticConn;
        }
        catch ( ... ) 
		{
		}
		throw;
	}
   	catch (...)
   	{
        try {
		    if (connection) connection->Close(); 
            if ( staticConn ) delete staticConn;
        }
        catch ( ... ) 
		{
		}
   		CPPUNIT_FAIL ("caught unexpected exception");
   	}
		
    if ( staticConn ) delete staticConn;

	printf( "Done\n" );
}

void MySqlFdoApplySchemaTest::TestCharSize_AddToExpected(
    FdoDictionaryP expected, 
    FdoString* key,
    FdoString* value
)
{
    FdoDictionaryElementP elem = FdoDictionaryElement::Create(key, value);
    expected->Add(elem);
}


void MySqlFdoApplySchemaTest::TestCharSize_Validate(
    FdoDictionaryP expected,
    FdoFeatureSchemaP schema, 
    FdoSmPhOwnerP owner
)
{
    int validated = 0;

    FdoClassesP classes = schema->GetClasses();

    for ( int i = 0; i < classes->GetCount(); i++ ) 
    {
        FdoClassDefinitionP classDef = classes->GetItem(i);
        FdoStringP className = classDef->GetName();

        FdoSmPhDbObjectP dbObject = owner->FindDbObject(className.Lower());
        CPPUNIT_ASSERT(dbObject);

        FdoPropertiesP props = classDef->GetProperties();
        FdoSmPhColumnsP columns = dbObject->GetColumns();

        for ( int j = 0; j < props->GetCount(); j++ ) 
        {
            FdoPropertyP prop = props->GetItem(j);
            if ( prop->GetPropertyType() == FdoPropertyType_DataProperty )
            {
                FdoDataPropertyDefinition *pDataProp = (FdoDataPropertyDefinition*) prop.p;

                if ( pDataProp->GetDataType() == FdoDataType_String ) 
                {
                    FdoStringP propName = pDataProp->GetName();
                    int propLen = pDataProp->GetLength();

                    FdoSmPhColumnP column = columns->FindItem(propName.Lower());
                    CPPUNIT_ASSERT( column );
                    FdoStringP colType = column->GetTypeName().Upper();

                    FdoStringP key = className + L":" + propName;
                    FdoStringP value = FdoStringP::Format(L"%d", propLen) + L":" + colType;

                    FdoDictionaryElementP elem = expected->FindItem(key);
                    CPPUNIT_ASSERT(elem);
                    if ( value != elem->GetValue() ) 
                    {
                        char buffer[5000];

                        sprintf( 
                            buffer,
                            "Key: '%ls', Value: '%ls', Expected '%ls'\n",
                            (FdoString*) key,
                            (FdoString*) value,
                            elem->GetValue()
                        );
                        CPPUNIT_FAIL( buffer );
                    }

                    validated++;
                }
            }
        }
    }

    CPPUNIT_ASSERT( validated == expected->GetCount() );
}

FdoRdbmsOvPhysicalSchemaMapping* MySqlFdoApplySchemaTest::CreateOverrideDefaults( FdoIConnection* connection, int passNum )
{ 
	FdoRdbmsOvPhysicalSchemaMapping* pOverrides = FdoApplySchemaTest::CreateOverrideDefaults( connection, passNum );

	FdoMySQLOvPhysicalSchemaMapping* pMySqlOverrides = static_cast<FdoMySQLOvPhysicalSchemaMapping*>(pOverrides);
	pMySqlOverrides->SetDataDirectory( L"c:\\\\" );  //TODO: remove doubled backslashes once MySQL Provider properly handles literal strings containing backslashes.
	pMySqlOverrides->SetIndexDirectory( L"c:\\\\" );  //TODO: remove doubled backslashes once MySQL Provider properly handles literal strings containing backslashes.
	pMySqlOverrides->SetStorageEngine( MySQLOvStorageEngineType_MyISAM );

    return pOverrides;
}

void MySqlFdoApplySchemaTest::CreateRdbmsSpecificElements(FdoIConnection* connection, FdoString* wDatastore)
{
    // nothing here yet
}

FdoRdbmsOvClassDefinition* MySqlFdoApplySchemaTest::CreateOvClassDefinition(FdoString *name)
{
    if (name == NULL)
        return FdoMySQLOvClassDefinition::Create();
    else
    {
        FdoPtr<FdoMySQLOvClassDefinition> classOv = FdoMySQLOvClassDefinition::Create(name);

        if (0==wcscmp(name, L"OvClassA"))
        {
            classOv->SetAutoIncrementPropertyName(L"FeatId");
            classOv->SetAutoIncrementSeed(100);
        }

        return FDO_SAFE_ADDREF(classOv.p);
    }
}

FdoRdbmsOvTable* MySqlFdoApplySchemaTest::CreateOvTable(FdoString* name)
{
    FdoMySQLOvTable* table = FdoMySQLOvTable::Create(name);

    // Set RDBMS-specific overrides, for some of the tables:
    if (0==wcscmp(name, L"ov_table_c"))
    {
// Assume the client and server are running on the same machine.
#ifdef _WIN32
        table->SetDataDirectory(L"d:\\\\");  //NOTE: need to double-up the back-slashes to please MySQL
        table->SetIndexDirectory(L"d:\\\\");  //NOTE: need to double-up the back-slashes to please MySQL
#else
        char mysqlDir[200];
        sprintf( mysqlDir, "/opt/mysql_data/%ls/", (FdoString*)  mDatastore );
        mkdir( mysqlDir, ACCESSPERMS | S_IWGRP | S_IWOTH );
        // mkdir does not set group and other write permissions so need
        // the following chmod.
        chmod( mysqlDir, ACCESSPERMS );
        table->SetDataDirectory(FdoStringP(L"/opt/mysql_data/") + mDatastore + L"/");
        table->SetIndexDirectory(FdoStringP(L"/opt/mysql_data/") + mDatastore + L"/");
#endif
        //TODO: most (all?) tables in the test have geometry columns, which only work on MyISAM engine,
        // so overriding the engine will cause errors during table creation:
        //table->SetStorageEngine(MySQLOvStorageEngineType_InnoDB);
    }

    if (0==wcscmp(name, L"oneforeign"))
    {
        table->SetDatabase( UnitTestUtil::GetEnviron("datastore", DB_NAME_OVERRIDE_SUFFIX) );
    }

    return table;
}

void MySqlFdoApplySchemaTest::VldClassCapabilities( int ltMode, int lckMode, FdoClassDefinition* pClass )
{
    // The following verifies that the AcDbEntity class capabilities are valid.
    FdoClassCapabilitiesP cc = pClass->GetCapabilities();
    CPPUNIT_ASSERT( cc != NULL );
    CPPUNIT_ASSERT( !(cc->SupportsLocking()) );
    CPPUNIT_ASSERT( !(cc->SupportsLongTransactions()) );

    FdoInt32 lockTypeCount;
    FdoLockType* lockTypes = cc->GetLockTypes( lockTypeCount );

    CPPUNIT_ASSERT( lockTypeCount == 0 );

    FdoPtr<FdoPropertyDefinitionCollection> props = pClass->GetProperties();
    for ( FdoInt32 i = 0; i < props->GetCount(); i++ )
    {
        FdoPtr<FdoPropertyDefinition> prop = props->GetItem(i);
        if (prop->GetPropertyType() == FdoPropertyType_GeometricProperty)
        {
            CPPUNIT_ASSERT( cc->GetPolygonVertexOrderRule(prop->GetName()) == FdoPolygonVertexOrderRule_CCW );
            CPPUNIT_ASSERT( !cc->GetPolygonVertexOrderStrictness(prop->GetName()) );
        }
    }
}

FdoStringP MySqlFdoApplySchemaTest::LogicalPhysicalBend( FdoString* inFile, int hybridLevel )
{
    FdoStringP outFile = UnitTestUtil::GetOutputFileName( inFile );

    _logicalPhysicalBend( inFile, outFile, L"MySql", hybridLevel );

    return outFile;
}

FdoStringP MySqlFdoApplySchemaTest::LogicalPhysicalFormat( FdoString* inFile )
{
    FdoStringP outFile = FdoStringP(inFile).Replace( L".", L"_LPhF." );

    _logicalPhysicalFormat( inFile, outFile, "MySql" );

    return outFile;
}

FdoStringP MySqlFdoApplySchemaTest::SchemaTestErrFile( int fileNum, bool isMaster ) 
{
    if ( isMaster ) {
        if ( (fileNum == 3) || (fileNum == 6) || (fileNum == 9) ) 
            return L"";
		return FdoStringP::Format( L"apply_schema_err%d%ls%ls.txt", fileNum,
			(fileNum == 2 || fileNum == 5 || fileNum == 8) ? L"M" : L"", L"_master");
    }
    return UnitTestUtil::GetOutputFileName( FdoStringP::Format( L"apply_schema_err%d%ls.txt",fileNum,
        (fileNum == 2 || fileNum == 3 || fileNum == 5 || fileNum == 6 || fileNum == 8 || fileNum == 9) ? L"M" : L"") );
}

FdoStringP MySqlFdoApplySchemaTest::GetParcelFirstName()
{
	return L"first_name";
}

FdoStringP MySqlFdoApplySchemaTest::GetParcelLastName()
{
	return L"last_name";
}

