/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */


#include "stdafx.h"
#include <Sm/Lp/Schema.h>
#include <Sm/Lp/PropertyMappingSingle.h>
#include "ObjectPropertyClass.h"
#include "DataPropertyDefinition.h"

FdoSmLpGrdObjectPropertyClass::FdoSmLpGrdObjectPropertyClass(
	FdoSmLpObjectPropertyDefinition* pParent,
	FdoSmLpClassDefinition* pParentType,
	FdoSmLpPropertyMappingDefinition* pMapping,
	FdoSmLpPropertyMappingType mappingType,
    FdoPhysicalClassMapping* pOverrides
) :
	FdoSmLpObjectPropertyClass(pParent, pParentType, pMapping, mappingType, pOverrides),
	FdoSmLpClassBase(	MakeName(pParent),
						L"Autogenerated Object Property Class",
						pMapping,
						(FdoSmLpClassDefinition*) NULL,
						FdoSmOvTableMappingType_ConcreteTable,
						pParent->GetContainingDbObjectName(),
						pParent->GetContainingDbObject(),
						pParent->GetElementState(),
                        pParent->GetIsFromFdo()
	)
{
    FdoRdbmsOvClassDefinition* pClassOverrides = (FdoRdbmsOvClassDefinition*) pOverrides;

    // Initialize the properties for this class.
	InitNestedProperties(pParent, pParentType, mappingType);
	InitProperties(pParent, pParentType, pMapping, mappingType, pClassOverrides);

    // There are no identity properties when object property is embedded in containing
    // class table.
	if ( mappingType != FdoSmLpPropertyMappingType_Single ) {
    	InitLocalIdProperty(pParent, pParentType, mappingType);
	    InitIdProperties(pParent, pParentType, pMapping, mappingType);
    }
}

FdoSmLpGrdObjectPropertyClass::FdoSmLpGrdObjectPropertyClass(
	FdoSmLpObjectPropertyClassP pBase,
	FdoSmLpObjectPropertyDefinition* pParent,
	FdoSmLpClassDefinition* pParentType,
	FdoSmLpPropertyMappingDefinition* pMapping,
	FdoSmLpPropertyMappingType mappingType,
    FdoPhysicalClassMapping* pOverrides
) :
	FdoSmLpObjectPropertyClass(pBase, pParent, pParentType, pMapping, mappingType, pOverrides),
	FdoSmLpClassBase(	MakeName(pParent),
						L"Autogenerated Object Property Class",
						pMapping,
						pBase.p->SmartCast<FdoSmLpClassDefinition>(),
						FdoSmOvTableMappingType_ConcreteTable,
						pParent->GetContainingDbObjectName(),
						pParent->GetContainingDbObject(),
						pParent->GetElementState(),
                        pParent->GetIsFromFdo()
	)
{
    FdoRdbmsOvClassDefinition* pClassOverrides = (FdoRdbmsOvClassDefinition*) pOverrides;

	// Initialize the properties for this class.
	InitNestedProperties(pBase);
	InitProperties(pBase, pParent, pMapping, mappingType, pClassOverrides);
	
    // There are no identity properties when object property is embedded in containing
    // class table.
    if ( mappingType != FdoSmLpPropertyMappingType_Single ) {
	    InitLocalIdProperty(pBase);
	    InitIdProperties(pBase);
    }
}

FdoSmLpGrdObjectPropertyClass::~FdoSmLpGrdObjectPropertyClass(void)
{
}

void FdoSmLpGrdObjectPropertyClass::Update(
    FdoClassDefinition* pFdoClass,
    FdoSchemaElementState elementState,
    FdoPhysicalSchemaMapping* pOverrides,
    bool bIgnoreStates
)
{
    //TODO: call base class's Update() method here?
}


void FdoSmLpGrdObjectPropertyClass::InitProperties (
    FdoSmLpObjectPropertyDefinition* pParent, 
	FdoSmLpClassDefinition* pParentType,
    FdoSmLpPropertyMappingDefinition* pMapping,
	FdoSmLpPropertyMappingType mappingType,
    FdoRdbmsOvClassDefinition* pOverrides
)
{
	FdoSmPhMgrP       pPhysical = GetLogicalPhysicalSchema()->GetPhysicalSchema();

    // Set up column prefix for when object property is embedded in containing class table.
    FdoStringP prefix;

    if ( mappingType == FdoSmLpPropertyMappingType_Single ) {
        prefix = ((FdoSmLpPropertyMappingSingle*) pMapping)->GetPrefix();

        // Chop prefix if too long; reserve last 1/4 for column name.
        FdoSize prefixMaxLen = (FdoSize)(pPhysical->DbObjectNameMaxLen() * 3 / 4);

        if ( prefix.GetLength() > prefixMaxLen ) {
            // Use last part of prefix if too long.
            prefix = prefix.Mid( prefix.GetLength() - prefixMaxLen, prefixMaxLen );
        }
    }

	// Ref the class defining the object property
	const FdoSmLpClassDefinition* pDefiningClass = pParent->RefDefiningClass();

	// Can't initialize properties if none of these classes can be found.
	if ( !pDefiningClass || !RefContainingClass() || !pParentType ) 
		return;

	// Set up the prefix for finding the nested properties that are properties
	// of this class.
	// Nested property names are not qualified by the outermost containing
	// class so strip out the outermost containing class name.
	FdoStringP propPrefix = FdoStringP( GetName() ).Right(L"." );		

	FdoSmLpPropertiesP pProps =
		GetProperties();

    FdoSmLpPropertiesP pNestedProps = 
		GetNestedProperties();

    FdoSmPhDbObjectP pPhDbObject = pPhysical->FindDbObject(GetDbObjectName());
    FdoSmPhTableP pPhTable = pPhDbObject.p->SmartCast<FdoSmPhTable>();

    // Treat created columns has having fixed names when that associated database object
    // is in the current datastore but is not a table. This prevents name adjustment for 
    // columns in local views. 
    bool isFixedColumn = pPhDbObject && (!pPhTable) && (wcslen(this->GetOwner()) == 0);


	/* Add all Object property class properties if mapping is "concrete".
	   When mapping is "class" then this class merely represents a 
	   relation table between the containing class table and the 
	   referenced class table. 
	 */

	if ( mappingType != FdoSmLpPropertyMappingType_Class ) {
		FdoSmLpPropertiesP pRefProps = 
			pParentType->GetProperties();

		for ( int i = 0; i < pRefProps->GetCount(); i++ ) {
			FdoSmLpPropertyP pRefProp = pRefProps->GetItem(i);

            // Skip system properties since the object property class will also
            // inherit them.
            if ( !pRefProp->GetIsSystem() ) {
			    const FdoSmLpDataPropertyDefinition* pRefDataProp =
                    FdoSmLpDataPropertyDefinition::Cast( pRefProp );

                FdoRdbmsOvPropertyP pPropOverride;
                // Ref the Schema Overrides for the sub-property
                if ( pOverrides ) 
                    pPropOverride = FdoRdbmsOvReadOnlyPropertiesP(pOverrides->GetProperties())->FindItem( pRefProp->GetName() );

    	        FdoSmLpPropertyP pProp;

                // Generate column name, with prefix if specified
                FdoStringP colName;
                if ( prefix.GetLength() > 0 ) 
                    colName = UniqueColumnName( pPhDbObject, NULL, (prefix + L"_" + pRefProp->GetName()), isFixedColumn );

	            // For each ref class property, find the corresponding
		        // nested property.
		        const FdoSmLpPropertyDefinition* pNestedProp =
			        pNestedProps->RefItem( propPrefix + L"." + pRefProp->GetName() );

		        if ( pNestedProp ) {
                    // The source property for this nested property is the property
                    // from the referenced class.
    	            if ( !pNestedProp->RefPrevProperty() )
                        ((FdoSmLpPropertyDefinition*) pNestedProp)->SetSrcProperty( pRefProp );
				        
                    // Nested property present, create property from nested property
	    	        pProp = pNestedProp->CreateCopy( this, pRefProp->GetName(), L"" );
                }
		        else {
			        // Nested property not present, generate property 
			        // from property on referenced class.
			        pProp = pRefProp->CreateCopy( this, L"", colName, pPropOverride );
		        }

                // Keep track of the top property in the property nesting.
		        pProp->SetTopProperty( pParent->GetTopProperty() );

		        pProps->Add(pProp);
            }
		}
	}

    // Skip foreign key properties when Single table mapping. These join properties are
    // not needed when the object property is in the same table as the containing class.
    if ( mappingType != FdoSmLpPropertyMappingType_Single ) {

	    /* Add foreign key properties to reference containing class */

        FdoSmLpClassDefinitionP pPkClass = GetObjectProperty()->GetPkClass();

	    FdoSmLpDataPropertiesP pContainingIdProps = 
		    pPkClass->GetIdentityProperties();

	    const FdoSmLpPropertyDefinitionCollection* pContainingProps = 
		    pPkClass->RefProperties();

    	
	    // Also calculate the source and target columns for joining this class's
	    // table to the containing class table:
	    // The source columns are the columns in the containing class table.
	    // The target columns are the corresponding columns in this class's table.

	    // First, check the Metadata (F_AttributeDependency row) for the 
	    // source and target columns.

	    FdoSmPhColumnListP pkColumnNames = 
            RefDependency() ? 
            RefDependency()->GetPkColumnNames() : 
            FdoSmPhColumnListP(FdoSmPhColumnList::Create(GetLogicalPhysicalSchema()->GetPhysicalSchema()));
   
	    FdoSmPhColumnListP fkColumnNames =
		    RefDependency() ? 
            RefDependency()->GetFkColumnNames() : 
            FdoSmPhColumnListP(FdoSmPhColumnList::Create(GetLogicalPhysicalSchema()->GetPhysicalSchema()));

        FdoSmPhColumnListP sourceColumnNames = FdoSmPhColumnList::Create( GetLogicalPhysicalSchema()->GetPhysicalSchema() );
	    FdoSmPhColumnListP targetColumnNames = FdoSmPhColumnList::Create( GetLogicalPhysicalSchema()->GetPhysicalSchema() );

	    if ( pkColumnNames && (pkColumnNames->GetCount() > 0) ) {
		    /* pkey columns for joining object property table with containing class table
		    * are defined in metadata. use these column names.
		    */

		    for ( int i = 0; i < pkColumnNames->GetCount(); i++ ) {
			    sourceColumnNames->Add( pkColumnNames->GetString(i) );

			    // Find corresponding property in containing table.

			    FdoSmLpDataPropertyP pMatchedProp = 
                    FDO_SAFE_ADDREF(
				        (FdoSmLpDataPropertyDefinition*) FdoSmLpDataPropertyDefinitionCollection::ColName2Property(
					        pContainingProps,	
					        sourceColumnNames->GetString(i)
				        )
                    );

			    if ( pMatchedProp ) {
				    GetSourceProperties()->Add(pMatchedProp);
			    }
			    else {
				    if ( GetElementState() != FdoSchemaElementState_Deleted ) 
					    AddSourcePropertyMissingError(pParent, sourceColumnNames->GetString(i));
			    }
    //			else
    // Todo: Autocreate a property for source column that is not a property
		    }
	    }
	    else {
		    /* Not in metadata, auto-generate from containing class's id properties */

		    for ( int i = 0; i < pContainingIdProps->GetCount(); i++ ) {
			    sourceColumnNames->Add( pContainingIdProps->RefItem(i)->GetColumnName() );
			    GetSourceProperties()->Add( FdoSmLpDataPropertyP(pContainingIdProps->GetItem(i)) );
		    }
	    }

	    if ( fkColumnNames && (fkColumnNames->GetCount() > 0) ) {
		    /* fkey columns for joining object property table with containing class table
		    * are defined in metadata. use these column names.
		    */

		    for ( int i = 0; i < fkColumnNames->GetCount(); i++ ) {
			    targetColumnNames->Add( fkColumnNames->GetString(i) );
		    }
	    }
	    else {
		    /* Not in metadata, auto-generate from containing class's id properties */

            // column name is prefixed by the containing' class's table name.
            // Use the name of the foreign table if class is mapped to a foreign schema.
            FdoStringP substSrcTableName = 
                (wcslen(pDefiningClass->GetRootDbObjectName()) > 0) ?
                    pDefiningClass->GetSubstRootDbObjectName() : 
                    pDefiningClass->GetSubstDbObjectName();

		    for ( int i = 0; i < pContainingIdProps->GetCount(); i++ ) {
                FdoStringP targetColName = substSrcTableName + L"_" + sourceColumnNames->GetString(i);
                      
                // Adjust target column name to be unique only if this object did 
                // not originate from an FDO config doc. In this case, the target
                // property must attach to an already existing column.
                if ( !(GetIsFromFdo() && (GetElementState() == FdoSchemaElementState_Unchanged)) )
                    targetColName = UniqueColumnName(
                        pPhDbObject, 
                        NULL, 
                        targetColName,
                        isFixedColumn
                    );
                targetColumnNames->Add( targetColName );
		    }
	    }

	    if ( sourceColumnNames->GetCount() != targetColumnNames->GetCount() ) {
		    if ( GetElementState() != FdoSchemaElementState_Deleted )
			    AddJoinMismatchError(pParent);
		    return;
	    }

	    // Add the foreign key properties to this object property class.

	    for ( int i = 0; i < targetColumnNames->GetCount(); i++ ) {

		    // Find corresponding property in containing table.
		    FdoSmLpDataPropertyP pMatchedProp = 
                FDO_SAFE_ADDREF(
			        (FdoSmLpDataPropertyDefinition*) FdoSmLpDataPropertyDefinitionCollection::ColName2Property(
				        pContainingProps,	
				        sourceColumnNames->GetString(i)
			        )
                );

		    if ( pMatchedProp ) {
			    // Property Found, create a copy of it. 
			    // The name of the copy is a concatenation of the original <class_name>_<property_name>
			    //  "_" is used to concatenate instead of "." so that FdoIdentifier will still
			    //  properly parse the property name.
			    // attach the copy to the target column name.

			    const FdoSmLpPropertyDefinition* pSrcProp = pMatchedProp->RefSrcProperty() ? pMatchedProp->RefSrcProperty() : (const FdoSmLpDataPropertyDefinition*) pMatchedProp;
                FdoStringP pTargPropName = FdoStringP(pSrcProp->GetParent()->GetName()) + L"_" + pSrcProp->GetName();

                // Get the Schema Overrides for the target property
                FdoRdbmsOvPropertyP pPropOverride;
                if ( pOverrides ) 
                    pPropOverride = FdoRdbmsOvReadOnlyPropertiesP(pOverrides->GetProperties())->FindItem( pTargPropName );

		        // Check if the target property has a nested property
		        const FdoSmLpDataPropertyDefinition* pNestedProp =
                    FdoSmLpDataPropertyDefinition::Cast( pNestedProps->RefItem(propPrefix + L"." + pTargPropName) );

                // Create target property from nested property, if found.
                // Otherwise, create from source property
		        const FdoSmLpPropertyDefinition* pCopyProp;
                
                if ( pNestedProp ) {
                    pCopyProp = pNestedProp;

                    // The source property for this nested property is the property
                    // from the referenced class.
                    if ( !pNestedProp->RefPrevProperty() ) 
                        ((FdoSmLpDataPropertyDefinition*) pNestedProp)->SetSrcProperty( pMatchedProp.p->SmartCast<FdoSmLpPropertyDefinition>() );
                }
                else {
                    pCopyProp = pMatchedProp;  
                }

                FdoSmLpDataPropertyP pProp =
				    pCopyProp->CreateCopy( 
					    this, 
					    pTargPropName,
					    targetColumnNames->GetString(i),
                        pPropOverride
				    ).p->SmartCast<FdoSmLpDataPropertyDefinition>();
    			
                // Keep track of the top property in the property nesting.
    	        pProp->SetTopProperty( pParent->GetTopProperty() );

			    // The foreign key columns must always be settable. Otherwise 
			    // it is impossible to populate object properties. 
			    pProp->SetReadOnly(false);
                // The foreign key values would be set to the values of
                // the properties they reference. Therefore, they are 
                // never Autogenerated.

                pProp->SetIsAutoGenerated(false);

			    // Check if there isn't already a property of the same name
			    if ( pProps->RefItem(pProp->GetName()) != NULL ) {
				    // This should happen very rarely but trap just in case.
				    AddTargetConflictError(pParent, pProp, pParentType);
			    }
			    else {
				    // add the new property to this class. 
				    pProps->Add( pProp );
				    GetTargetProperties()->Add( (FdoSmLpDataPropertyDefinition*) pProp );
 /* ?   			
				    if ( (pProp->RefColumn() == NULL) &&
					    (GetElementState() != FdoSchemaElementState_Deleted) ) 
					    // Error, target property has no column.
					    AddTargetColumnMissingError( pParent, pProp );
*/
			    }
		    }
		    else {
			    // Error, target column has no corresponding source property.
			    if ( GetElementState() != FdoSchemaElementState_Deleted ) 
				    AddTargetPropertyMissingError( pParent, targetColumnNames->GetString(i) );
		    }
	    }
    }
}

bool FdoSmLpGrdObjectPropertyClass::CanInherit( const FdoSmLpPropertyDefinition* pProp )
{
    bool canInherit = FdoSmLpClassBase::CanInherit( pProp );

    // Filter out any system properties related to FDO locking.
    // Only the topmost containing object of an object property can be locked.
    if ( canInherit && pProp->GetIsSystem() ) {
        FdoStringP propName = pProp->GetName();

        if ( (propName == FdoSmLpGrdDataPropertyDefinition::LockIdName) ||
             (propName == FdoSmLpGrdDataPropertyDefinition::LockTypeName)
        )
            canInherit = false;
    }

    return canInherit;
}

void FdoSmLpGrdObjectPropertyClass::InitProperties (
   	FdoSmLpObjectPropertyClassP pBase,
	FdoSmLpObjectPropertyDefinition* pParent,
    FdoSmLpPropertyMappingDefinition* pMapping,
    FdoSmLpPropertyMappingType mappingType,
    FdoRdbmsOvClassDefinition* pOverrides
)
{
	int i;
	FdoSmLpPropertiesP pProps =
		GetProperties();

	// Copy properties from base class.
	for ( i = 0; i < pBase->RefProperties()->GetCount(); i++ ) {
		FdoSmLpPropertyP pProp = 
			pBase->RefProperties()->RefItem(i)->CreateInherited( this );

        // Keep track of the top property in the property nesting.
    	pProp->SetTopProperty( pParent->GetTopProperty() );

		pProps->Add( pProp );
	}

    FdoSmLpClassDefinitionP pPkClass = GetObjectProperty()->GetPkClass();

	// Copy source properties from base class
	for ( i = 0; i < pBase->RefSourceProperties()->GetCount(); i++ ) {
		const FdoSmLpDataPropertyDefinition* pBaseProp = 
			pBase->RefSourceProperties()->RefItem(i);

		// Source property is in containing class of this class's object property.
		FdoSmLpDataPropertyP pProp =
			pPkClass->GetProperties()->FindItem(pBaseProp->GetName())->SmartCast<FdoSmLpDataPropertyDefinition>(true);

		if ( pProp ) 
			GetSourceProperties()->Add( pProp );
	}

	// Copy Target properties from base class
	for ( i = 0; i < pBase->RefTargetProperties()->GetCount(); i++ ) {
		const FdoSmLpDataPropertyDefinition* pBaseProp = 
			pBase->RefTargetProperties()->RefItem(i);

		FdoSmLpDataPropertyP pProp =
			pProps->GetItem(pBaseProp->GetName())->SmartCast<FdoSmLpDataPropertyDefinition>(true);

		if ( pProp ) 
			GetTargetProperties()->Add( pProp );
	}
}

