/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
#include "stdafx.h"
#include "Fdo/Pvc/FdoRdbmsPvcInsertHandler.h"
#include <Sm/Lp/ObjectPropertyDefinition.h>
#include <Sm/Lp/ObjectPropertyClass.h>
#include <Sm/Lp/PropertyMappingSingle.h>
#include "FdoRdbmsSchemaUtil.h"
#include "FdoRdbmsLobUtility.h"

FdoRdbmsPvcInsertHandler::FdoRdbmsPvcInsertHandler( ) :
    mNextQidToFree(0),
	mFdoConnection( NULL ),
	mInsertAutoIncrementProperties( false ),
	mBindUnsetValues( false )
{
}

FdoRdbmsPvcInsertHandler::FdoRdbmsPvcInsertHandler( FdoRdbmsConnection *connection ) :
    mNextQidToFree(0),
	mFdoConnection( connection ), // no addref as it is owned by a component that is already holding a reference on the connection. 
	mInsertAutoIncrementProperties( false ),
	mBindUnsetValues( false )
{
    mLastTableName[0] = '\0';
    for( int i = 0; i<QUERY_CACHE_SIZE; i++ )
    {
        mInsertQueryCache[i].qid = -1;
        mInsertQueryCache[i].tableName[0] = '\0';
        mInsertQueryCache[i].bindHelper = new FdoRdbmsPropBindHelper(mFdoConnection);
        mInsertQueryCache[i].bindProps = new std::vector< std::pair< FdoLiteralValue*, FdoInt64 > >();
        mInsertQueryCache[i].bindPropNames = FdoStringCollection::Create();
        mInsertQueryCache[i].specialValues = FdoDataValueCollection::Create();
    }
}

FdoRdbmsPvcInsertHandler::~FdoRdbmsPvcInsertHandler()
{
	DbiConnection *mConnection = mFdoConnection->GetDbiConnection();
	for( int i = 0; i<QUERY_CACHE_SIZE; i++ )
    {
        if (mInsertQueryCache[i].qid != -1 && mFdoConnection->GetConnectionState() == FdoConnectionState_Open )
		{
            mConnection->GetGdbiCommands()->free_cursor(mInsertQueryCache[i].qid);
			mInsertQueryCache[i].qid = -1;
            delete mInsertQueryCache[i].bindHelper;
            delete mInsertQueryCache[i].bindProps;
            mInsertQueryCache[i].bindHelper = NULL;
        }
    }
}

long FdoRdbmsPvcInsertHandler::Execute( const FdoSmLpClassDefinition *classDefinition, FdoPropertyValueCollection  *propValCollection, bool revisionNumUpdate, bool handleForeignAutoincrementedId )
{
	int                 i;
    int                 gid = -1;
    int                 numberOfProperties;
    InsertQueryDef      *insertQuery = NULL;
    bool                needBind = false;

	DbiConnection *mConnection = mFdoConnection->GetDbiConnection();

    try
    {
        if (propValCollection == NULL)
            return 0;

        numberOfProperties = (int) propValCollection->GetCount();
        if (numberOfProperties == 0)
            return 0;

        // Only the main class may have a list that ONLY contain system or autogenerated properties. In that case
        // and when all other properties are nullable properties, we are creating an empty object.
        // The object property classes attached to the main object cannot contain only system and autogen properties;
        // it means the user did not provide any values for those sub-object and we will not add any empty objects.
		// Unless we are overrriding autoincrement property, then all bets are off.
		if( dynamic_cast<const FdoSmLpObjectPropertyClass*>(classDefinition) != NULL && ! mInsertAutoIncrementProperties )
        {
            const FdoSmLpPropertyDefinitionCollection *properties = classDefinition->RefProperties();
            bool emptyPropList = true;
            for( i=0; i<numberOfProperties; i++ )
            {
                FdoPtr<FdoPropertyValue> prop = propValCollection->GetItem(i);
                FdoPtr<FdoIdentifier>nameId = prop->GetName();
                const FdoSmLpPropertyDefinition *phprop = properties->RefItem( nameId->GetName() );
                if( phprop && phprop->GetPropertyType() == FdoPropertyType_DataProperty )
                {
                    const FdoSmLpDataPropertyDefinition *dataProp = (FdoSmLpDataPropertyDefinition *)phprop;
                    if( ! dataProp->GetIsAutoGenerated() &&
                        ! dataProp->GetIsSystem() &&
                        ! dataProp->GetIsFeatId() )
                    {
                        // Check if it's an autogenerated property; the autogenerated property is the property added by schema
                        // manager for Object properties
                        const FdoSmLpObjectPropertyClass* cls = dynamic_cast<const FdoSmLpObjectPropertyClass*>(classDefinition);
                        if( cls )
                        {
                            const FdoSmLpDataPropertyDefinitionCollection *targProps = cls->RefTargetProperties();
                            const FdoSmLpDataPropertyDefinition *targProp = targProps->RefItem( nameId->GetName() );
                            if( targProp == NULL )
                            {
                                emptyPropList = false;
                                break;
                            }
                        }
                    }
                }
            }
            if( emptyPropList )
                return 0;
        }
        const FdoSmLpDbObject* table = classDefinition->RefDbObject();
        FdoStringP tableName = table->RefDbObject()->GetDbQName();

        FdoStringP name = classDefinition->GetQName();
		if( mInsertAutoIncrementProperties )
			insertQuery = GetInsertQuery( (const wchar_t*)(name+FdoStringP(L"WithId")), true );
		else
			insertQuery = GetInsertQuery(  name, false );

		FdoStringP valuesString(L"");
        FdoStringP colSpecString(L"");
		FdoStringP insertStartString(L"");
		FdoStringP whereString(L"");
        int        bindCount = 0;
        // Reparse each time in the LOBs are involved
        // (the insert stm. may be different depending on the current values)
        if(insertQuery->qid != -1 && !FdoRdbmsLobUtility::ContainsLobs( classDefinition ) )
        {
            // CreateInsertString may modify the propValCollection in case of adding association.
            CreateInsertString( classDefinition, propValCollection, colSpecString, valuesString, bindCount, true);
        }
        else
        {			
            CreateInsertString( classDefinition, propValCollection, colSpecString, valuesString, bindCount, false);

            bool useLocalInsertString = true;
            const FdoSmLpDataPropertyDefinitionCollection *propDefCol = classDefinition->RefIdentityProperties();
            for( int i = 0; i < propDefCol->GetCount(); i++ )
            {
                if( propDefCol->RefItem(i)->GetIsAutoGenerated() )
                {
                    useLocalInsertString = false;
                    break;
                }
            }

            if( useLocalInsertString && ( classDefinition->GetClassType() == FdoClassType_FeatureClass ) )
            {
                FdoPtr<FdoIdentifier> currPropName;
	            const FdoSmLpPropertyDefinitionCollection *allProps = classDefinition->RefProperties();

	            for( int i = 0; i < propValCollection->GetCount(); i++ )
	            {
                    FdoPtr<FdoPropertyValue> pPropVal = propValCollection->GetItem(i);
                    currPropName = pPropVal->GetName();
		            const FdoSmLpDataPropertyDefinition *aProp =
                        FdoSmLpDataPropertyDefinition::Cast(allProps->RefItem(currPropName->GetName()));
                    if ( aProp != NULL )
                    {
		                if( aProp->GetIsAutoGenerated() )
		                {
			                useLocalInsertString = false;
		                }
                    }
                    if( !useLocalInsertString )
                        break;
                }
            }

            if( useLocalInsertString )
                FdoRdbmsPvcInsertHandler::GetStartInsertString( insertStartString, tableName );
            else
                GetStartInsertString( insertStartString, tableName );

			CreateInsertStringWhereCriteria( classDefinition, propValCollection, whereString, bindCount );

            FdoStringP tmp = FdoStringP::Format(L"%ls %ls %ls %ls",
													(const wchar_t * ) insertStartString,
                                                    (const wchar_t * ) colSpecString,
                                                    (const wchar_t * ) valuesString,
													(const wchar_t * ) whereString);

            mConnection->GetGdbiCommands()->sql( (wchar_t *)(const wchar_t *) tmp, &gid );

            insertQuery->qid = gid;
            needBind = true;
        }

        SetBindVariables(classDefinition, L"", propValCollection, insertQuery);
			
    	SetAditionalBindVariables(classDefinition, L"", propValCollection, insertQuery, gid);

        if ( needBind ) 
            insertQuery->bindHelper->BindParameters(mConnection->GetGdbiCommands(), insertQuery->qid, insertQuery->bindProps);
        else
            insertQuery->bindHelper->BindValues(mConnection->GetGdbiCommands(), insertQuery->qid, insertQuery->bindProps);

        mConnection->GetGdbiCommands()->execute( insertQuery->qid );

        // Insert LOB values Stream Reader based
//        FdoRdbmsLobUtility::InsertStreamedLobs( mConnection, classDefinition, propValCollection, insertQuery->bind, insertQuery->count );

    }
    catch( FdoException *ex )
    {
        if( insertQuery != NULL && insertQuery->qid != -1 )
        {
            mConnection->GetGdbiCommands()->free_cursor( insertQuery->qid );
            insertQuery->qid = -1;
        }

        throw ex;
    }
    catch (...)
    {
        throw FdoException::Create(L"FdoRdbmsPvcInsertHandler::Execute() failed");
    }

	return 1;
}



void FdoRdbmsPvcInsertHandler::CreateInsertString(const FdoSmLpClassDefinition *currentClass,
                                               FdoPropertyValueCollection  *propValCollection,
                                               FdoStringP& insertString,
                                               FdoStringP& valuesString,
                                               int& bindCount,
                                               bool scanOnly )
{
    // TODO: The logic and some of the checks in CreateInsertString must line up with those in 
    // SetBindVariables, or the bound columns will not match those referenced in the SQL strings
    // generated in this function. CreateInsertString() and SetBindVariables() should be merged
    // to ensure that they do not diverge in the future.

    FdoPtr<FdoSmLpPropertyDefinitionCollection> propertyDefinitions = ((FdoSmLpClassDefinition*)currentClass)->GetProperties();
    int     pass;
    int     i;
    bool    emptyBlobAdded;

    // Make a copy of the properties with the geometries at the end of the collection
    if ( mFdoConnection->BindGeometriesLast() )
        propertyDefinitions = MoveGeometryProperties( currentClass );
        
    FdoStringP comma(L",");

	DbiConnection *mConnection = mFdoConnection->GetDbiConnection();

    const FdoSmPhDbObject* classTab = currentClass->RefDbObject()->RefDbObject();

    // In Most cases we only bind in columns for properties for which a property value was specified.
    // This is done by pass 0. 
    // However, if no property values were specified, then the generated SQL is invalid since the 
    // insert statement must have a column list. 
    // If no columns were bound in pass 0, do a pass 1, which just binds in the one column from the
    // class table. 

    for (pass=0; pass < 2; pass++)
    {
        // Bind only 1 column in pass 1. Also quits here if pass 0 bound columns.
        if ( (pass == 1) && (bindCount > 0) )
            break;

        for (i=0; i<propertyDefinitions->GetCount(); i++)
        {
            // Bind only 1 column in pass 1.
            if ( (pass == 1) && (bindCount > 0) )
                break;

            const FdoSmLpPropertyDefinition *propertyDefinition = propertyDefinitions->RefItem(i);
            FdoPropertyType  propType = propertyDefinition->GetPropertyType();
            emptyBlobAdded = false;
            const wchar_t* tmpName = propertyDefinition->GetName();

            FdoPtr<FdoValueExpression> exp;

            switch ( propType )
            {
                case FdoPropertyType_DataProperty:
                    {
                        // Skip property if not in property value collection and we're on pass 0.
                        if ( !BindThisValue(tmpName, (pass == 0) ? propValCollection : (FdoPropertyValueCollection*) NULL, exp) )
                            break;

                        if( scanOnly )
                            break;

                        const FdoSmLpDataPropertyDefinition* dataProp =
                            static_cast<const FdoSmLpDataPropertyDefinition*>(propertyDefinition);

                        bool autoGenerated = dataProp && dataProp->GetIsAutoGenerated();

                        if( dataProp->RefContainingDbObject() != classTab )
                            continue;

                        const FdoSmPhColumn *column = dataProp->RefColumn();
                        if( column == NULL || ( ! mInsertAutoIncrementProperties && column->GetAutoincrement() ) )
                            continue;

                        CreateInsertStringForColumn(
                            column,
                            propertyDefinition,
                            propValCollection,
                            insertString, valuesString,
                            bindCount, emptyBlobAdded );
                    }
                    break;

                case FdoPropertyType_GeometricProperty:
                    {
                        if( scanOnly )
                            break;

                        // Skip property if not in property value collection and we're on pass 0.
                        if ( !BindThisValue(tmpName, (pass == 0) ? propValCollection : (FdoPropertyValueCollection*) NULL, exp) )
                            break;

                        const FdoSmLpGeometricPropertyDefinition* geomProp =
                            static_cast<const FdoSmLpGeometricPropertyDefinition*>(propertyDefinition);

                        if( geomProp->RefContainingDbObject() != classTab )
                            continue;

                        FdoSmOvGeometricColumnType columnType = geomProp->GetGeometricColumnType();

                        switch (columnType)
                        {
                        case FdoSmOvGeometricColumnType_Default:
                        case FdoSmOvGeometricColumnType_BuiltIn:
                        case FdoSmOvGeometricColumnType_Blob:
                        case FdoSmOvGeometricColumnType_Clob:
                        case FdoSmOvGeometricColumnType_String:
                            {
                                const FdoSmPhColumn *column = geomProp->RefColumn();
                                if (NULL != column)
                                {
                                    CreateInsertStringForColumn(
                                        column,
                                        propertyDefinition,
                                        propValCollection,
                                        insertString, valuesString,
                                        bindCount, emptyBlobAdded );
                                }
                            }
                            break;
                        case FdoSmOvGeometricColumnType_Double:
                            {
                                const FdoSmPhColumn *columnX = geomProp->RefColumnX();
                                const FdoSmPhColumn *columnY = geomProp->RefColumnY();
                                const FdoSmPhColumn *columnZ = geomProp->RefColumnZ();

                                if (NULL != columnX && NULL != columnY)
                                {
                                    // Support ordinate column storage format.
                                    CreateInsertStringForColumn(
                                        columnX,
                                        propertyDefinition,
                                        propValCollection,
                                        insertString, valuesString,
                                        bindCount, emptyBlobAdded );

                                    CreateInsertStringForColumn(
                                        columnY,
                                        propertyDefinition,
                                        propValCollection,
                                        insertString, valuesString,
                                        bindCount, emptyBlobAdded );

                                    if (NULL != columnZ)
                                    {
                                        CreateInsertStringForColumn(
                                            columnZ,
                                            propertyDefinition,
                                            propValCollection,
                                            insertString, valuesString,
                                            bindCount, emptyBlobAdded );
                                    }
                                }
                            }
                            break;
                        default:
                            // Nothing useful to do here.
                            continue;
                        }   // end of switch (columnType)

                        // Support spatial index columns, if present.
                        const FdoSmPhColumn *columnSi1 = geomProp->RefColumnSi1();
                        const FdoSmPhColumn *columnSi2 = geomProp->RefColumnSi2();
                        if (NULL != columnSi1 && NULL != columnSi2)
                        {
                            CreateInsertStringForColumn(
                                columnSi1,
                                propertyDefinition,
                                propValCollection,
                                insertString, valuesString,
                                bindCount, emptyBlobAdded );

                            CreateInsertStringForColumn(
                                columnSi2,
                                propertyDefinition,
                                propValCollection,
                                insertString, valuesString,
                                bindCount, emptyBlobAdded );
                        }
                    }
                    break;

                case FdoPropertyType_ObjectProperty:
                    {
                        const FdoSmLpObjectPropertyDefinition* objProp = static_cast<const FdoSmLpObjectPropertyDefinition*>(propertyDefinition);
                        const FdoSmLpPropertyMappingDefinition* mappping = objProp->RefMappingDefinition();
                        if ( mappping->GetType() != FdoSmLpPropertyMappingType_Single )
                            break; // We only handle the single mapping

                        const FdoSmLpPropertyMappingSingle * singleMapping = static_cast<const FdoSmLpPropertyMappingSingle*>( mappping );
                        CreateInsertString( objProp->RefTargetClass(), propValCollection, insertString, valuesString, bindCount, scanOnly );
                    }
                    break;

                case FdoPropertyType_AssociationProperty:
                    {
                        const FdoSmLpAssociationPropertyDefinition* associationPropertyDefinition = static_cast<const FdoSmLpAssociationPropertyDefinition*>(propertyDefinition);
                        if( associationPropertyDefinition->GetReadOnly() )
                        {
                            // TODO: need to check the property value collection and not the class properties.

                            // Should not happen as we don't add the read-only association property.
                            // The caller must have added it to the collection.
                            //throw FdoCommandException::Create(NlsMsgGet(FDORDBMS_290, "Cannot set a read only association property"));
                            break;
                        }


                        //
                        // We only add the column if the identity properties are not set; translation: we added axtra column
                        // to link to the associated class. If the identity properties are set, it mean we are using existing
                        // properties/column and there is no need to add any column as those column will be added a data properties.
                        const FdoStringsP   identCollection = associationPropertyDefinition->GetIdentityProperties();
                        if( identCollection->GetCount() == 0  )
                        {
                            const FdoSmPhColumnListP identCols  = associationPropertyDefinition->GetReverseIdentityColumns();
                            const FdoSmLpDataPropertyDefinitionCollection *identPropCol =  associationPropertyDefinition->RefAssociatedClass()->RefIdentityProperties();
                            for(int i=0; i<identCols->GetCount() && ! scanOnly ; i++ )
                            {
                                const FdoSmLpDataPropertyDefinition *identProp = identPropCol->RefItem( i );
                                FdoStringP qPropName = FdoStringP(tmpName, true) + L"." + identProp->GetName(); 
                                if ( !BindThisValue(qPropName, (pass == 0) ? propValCollection : (FdoPropertyValueCollection*) NULL, exp) )
                                    continue;

							    CreateInsertStringForColumn(
								    (const wchar_t * ) identCols->GetDbString(i),
								    insertString, valuesString,
								    bindCount );
                            }
                        }
                        else
                        {
                            //
                            // We may have duplicate entry for the same column. The <asso>.<ident> or the rever_ident property may both be initialized.
                            // If both are initialized, then both values should be the same.
                            // If only the <asso>.<ident> is set, then we (add) and/or set the rever_ident property.
                            // If the rever_ident is set, then we do nothing
                            const FdoStringsP   revIdentCollection = associationPropertyDefinition->GetReverseIdentityProperties();
                            for( int i=0; i<identCollection->GetCount(); i++ )
                            {
                                FdoStringP assoPropName = FdoStringP::Format(L"%ls.%ls",propertyDefinition->GetName(),(const wchar_t*)(identCollection->GetString(i)));
                                FdoPtr<FdoPropertyValue> assocPropertyValue;
                                FdoPtr<FdoPropertyValue> identPropertyValue;
                                FdoPtr<FdoValueExpression> assoVal;
                                FdoPtr<FdoValueExpression> identVal;
                                assocPropertyValue = propValCollection->FindItem( (const wchar_t*)assoPropName );
                                assoVal = assocPropertyValue ? assocPropertyValue->GetValue() : NULL;

                                FdoStringP  revIdenName = revIdentCollection->GetString(i);
                                identPropertyValue = propValCollection->FindItem( (const wchar_t*)(revIdenName) );
                                identVal = identPropertyValue ? identPropertyValue->GetValue() : NULL;
                                if( assocPropertyValue != NULL && identPropertyValue != NULL &&  assoVal != NULL && identVal != NULL )
                                {
                                    if( FdoRdbmsUtil::StrCmp(assoVal->ToString(), identVal->ToString() ) )
                                        throw FdoCommandException::Create(NlsMsgGet2(FDORDBMS_291, "Association property '%1$ls' and property '%2$ls' must have the same value or only one should be set",
                                        (const wchar_t*)assoPropName, (const wchar_t*)revIdenName ));

                                }
                                else if( assoVal != NULL )
                                {
                                    if( identPropertyValue != NULL )
                                    {
                                        // rever ident found but not set; set its value from the <asso>.<ident> property value.
                                        // Note that this may not get executed as this property will be removed in FdoRdbmsSchemaUtil::GroupByClass
                                        identPropertyValue->SetValue( assoVal );
                                    }
                                    else
                                    {
                                        // The <asso>.<ident> is set and the rever_ident does not exist. In this case we just
                                        // rename the <asso>.<ident> name to the rever_ident name. The correponding column is already
                                        // by the data property case above. We only needed to hook the value of the property. That is
                                        // easily done by renaming the property that contain the value to the name of the property
                                        // that need the value.
                                        assocPropertyValue->SetName( (const wchar_t*)revIdenName );
                                    }
                                }
                            }
                        }
                        //
                        // Perform a constrain check before goind any further. The constrain check should be done at this point as it
                        // depend on the processing that may happen above.
                        AssociationConstrainCheck( associationPropertyDefinition, propValCollection );
                    }
                    break;
                default:
                    break;
            }
        }
    }
}

void FdoRdbmsPvcInsertHandler::GetStartInsertString( FdoStringP& insertStartString, const wchar_t* tableName )
{
	const wchar_t* frmStr = L"insert into %ls ";
	insertStartString = FdoStringP::Format(frmStr, tableName );
}

void FdoRdbmsPvcInsertHandler::CreateInsertStringWhereCriteria(
        const FdoSmLpClassDefinition *classDefinition,
        FdoPropertyValueCollection  *propValCollection,
        FdoStringP &whereString,
        int& bindCount )
{
		whereString = ")"; // Needed to close the values spec
}

void FdoRdbmsPvcInsertHandler::CreateInsertStringForColumn(
        const wchar_t *columnName,
        FdoStringP &insertString,
        FdoStringP &valuesString,
        int& bindCount )
{
	FdoStringP comma(L",");

	if( ((const wchar_t*)valuesString)[0] == '\0' )
		valuesString += L") values ("; // End of the column spec and start of values spec

	if( ((const wchar_t*)insertString)[0] == '\0' )
		insertString += L" (";  // Start of the columns spec

    if( bindCount != 0 )
        insertString += (const wchar_t * )comma;
    insertString += (const wchar_t * ) columnName;

    if( bindCount != 0 )
        valuesString += (const wchar_t * )comma;

    valuesString += mFdoConnection->GetBindString( bindCount+1 );
    
	bindCount++;
}

void FdoRdbmsPvcInsertHandler::CreateInsertStringForColumn(
    const FdoSmPhColumn *column,
    const FdoSmLpPropertyDefinition *propertyDefinition,
    FdoPropertyValueCollection  *propValCollection,
    FdoStringP &insertString,
    FdoStringP &valuesString,
    int& bindCount,
    bool& emptyBlobAdded )
{
    FdoStringP comma(L",");
    FdoStringP empty_blob(L"EMPTY_BLOB()");
    FdoStringP null_blob(L"NULL");

	if( ((const wchar_t*)valuesString)[0] == '\0' )
		valuesString += L") values ("; // End of the column spec and start of values spec

	if( ((const wchar_t*)insertString)[0] == '\0' )
		insertString += L" (";  // Start of the columns spec

    FdoStringP colName = column->GetDbName();
    if( bindCount != 0 )
        insertString += (const wchar_t * )comma;
    insertString += (const wchar_t * ) colName;

    if( bindCount != 0 )
        valuesString += (const wchar_t * )comma;

    // We'll bind LOB references in separate SELECT operation.
    // Insert an empty lob if inserting is streamed.
    FdoPtr<FdoPropertyValue> propertyValue;
    FdoPtr<FdoIStreamReader> streamReader;

    propertyValue = propValCollection->FindItem( propertyDefinition->GetName() );
    if ( propertyValue )
        streamReader = propertyValue->GetStreamReader();

    FdoPropertyType  propType = propertyDefinition->GetPropertyType();

    if (propType == FdoPropertyType_DataProperty)
    {
        const FdoSmLpDataPropertyDefinition* dataProp =
        static_cast<const FdoSmLpDataPropertyDefinition*>(propertyDefinition);

        if ( dataProp->GetDataType() == FdoDataType_BLOB )
        {
            emptyBlobAdded = true;
            if ( streamReader != NULL )
                valuesString += (const wchar_t * )empty_blob;
            else if ( propertyValue == NULL || propertyValue->GetValue() == NULL )
                valuesString += (const wchar_t * )null_blob;
            else
                emptyBlobAdded = false;
        }
    }
    if ( !emptyBlobAdded )
    {
        valuesString += mFdoConnection->GetBindString( bindCount+1, propertyDefinition );
    }
    bindCount++;
}

void FdoRdbmsPvcInsertHandler::SetAditionalBindVariables(const FdoSmLpClassDefinition *currentClass, 
						  const wchar_t *scope,
						  FdoPropertyValueCollection  *propValCollection, 
						  InsertQueryDef *insertQuery, int gid)
{
	// Nothing to add
}

void FdoRdbmsPvcInsertHandler::SetBindVariables(const FdoSmLpClassDefinition *currentClass, 
												const wchar_t *scope, FdoPropertyValueCollection  *propValCollection, InsertQueryDef *queryDef)
{
    FdoStringCollection* bindPropNames = queryDef->bindPropNames;
    std::vector< std::pair< FdoLiteralValue*, FdoInt64 > >* bindProps = queryDef->bindProps;
    FdoDataValueCollection* specialValues = queryDef->specialValues;


    if ( (*scope) == 0 ) 
    {
        bindPropNames->Clear();
        bindProps->clear();
        specialValues->Clear();
    }

    FdoPtr<FdoSmLpPropertyDefinitionCollection> propertyDefinitions = ((FdoSmLpClassDefinition*)currentClass)->GetProperties();

    // Make a copy of the properties with the geometries at the end of the collection
    if ( mFdoConnection->BindGeometriesLast() )
        propertyDefinitions = MoveGeometryProperties( currentClass );

    int pass;
    int i;

    // In Most cases we only bind in columns for properties for which a property value was specified.
    // This is done by pass 0. 
    // However, if no property values were specified, then the generated SQL is invalid since the 
    // insert statement must have a column list. 
    // If no columns were bound in pass 0, do a pass 1, which just binds in the one column from the
    // class table. 

    for (pass=0; pass < 2; pass++ ) 
    {
        // Quit if on pass 1 and at least one column was bound. This also quits if pass 0 found 
        // columns to bind.
        if ( (pass == 1) && (bindPropNames->GetCount() > 0) )
            break;

        for (i=0; i<propertyDefinitions->GetCount(); i++)
        {
           // Quit if on pass 1 and at least one column was bound. 
            if ( (pass == 1) && (bindPropNames->GetCount() > 0) )
                break;

            const FdoSmLpPropertyDefinition *propertyDefinition = propertyDefinitions->RefItem(i);

            SetBindVariable(
                currentClass, 
                scope, 
                (pass == 0) ? propValCollection : (FdoPropertyValueCollection*) NULL, 
                queryDef,
                propertyDefinition
            );
        }
    }
}

void FdoRdbmsPvcInsertHandler::SetBindVariable(const FdoSmLpClassDefinition *currentClass, 
												const wchar_t *scope, FdoPropertyValueCollection  *propValCollection, InsertQueryDef *queryDef,
                                                const FdoSmLpPropertyDefinition *propertyDefinition, FdoString* columnName)
{
    FdoStringCollection* bindPropNames = queryDef->bindPropNames;
    std::vector< std::pair< FdoLiteralValue*, FdoInt64 > >* bindProps = queryDef->bindProps;
    FdoDataValueCollection* specialValues = queryDef->specialValues;

    FdoRdbmsSpatialManagerP spatialManager = mFdoConnection->GetSpatialManager();

    const FdoSmPhDbObject* classTab = currentClass->RefDbObject()->RefDbObject();
	DbiConnection *mConnection = mFdoConnection->GetDbiConnection();

    FdoPropertyType  propType = propertyDefinition->GetPropertyType();

    FdoPtr<FdoValueExpression> exp;

    FdoStringP qPropName = propertyDefinition->GetName();
    if( scope[0] != '\0' )
        qPropName = FdoStringP(scope) + L"." + propertyDefinition->GetName();

    switch ( propType )
    {
        case FdoPropertyType_DataProperty:
        {
            if ( !BindThisValue(qPropName, propValCollection, exp) )
                break;

            const FdoSmLpDataPropertyDefinition* dataProp =
                static_cast<const FdoSmLpDataPropertyDefinition*>(propertyDefinition);

            if( (!columnName) && (dataProp->RefContainingDbObject() != classTab) )
                break;

            const FdoSmPhColumn *column = NULL;
                
            if ( columnName ) 
                column = ((FdoSmPhDbObject*) classTab)->GetColumns()->FindItem(columnName);
            else
                column = dataProp->RefColumn();

            if (NULL == column)
            {
                if ( dataProp->GetIsSystem() ) 
                {
                    // It is possible for externally defined tables to not have
                    // columns for certain system properties. In this case, 
                    // skip these properties.
                    break;
                }
                else 
                {
                    throw FdoRdbmsException::Create(NlsMsgGet1(
                            FDORDBMS_485,
                            "No column for property '%1$ls'.",
                            dataProp->GetName()));
                }
            }
            const wchar_t *colName = column->GetName();

            if ( ! mInsertAutoIncrementProperties && column->GetAutoincrement() )
                break;

            FdoDataType dataType = dataProp->GetDataType();
            // Initialize the bind item

            bindPropNames->Add(qPropName);

            if ( !exp ) 
            {
                FdoDataValue* dv = FdoDataValue::Create(dataType);
                dv->SetNull();
                exp = dv;
                specialValues->Add(dv);
            }

            bindProps->push_back(std::make_pair((FdoLiteralValue*)exp.p, 0));
               
            break;
        }

        case FdoPropertyType_GeometricProperty:
        {
            if ( wcscmp(propertyDefinition->GetName(), L"Bounds") == 0 )
                break;

            if ( !BindThisValue(qPropName, propValCollection, exp) )
                break;

            const FdoSmLpGeometricPropertyDefinition* geomProp =
                static_cast<const FdoSmLpGeometricPropertyDefinition*>(propertyDefinition);

            const wchar_t *colName = NULL;

            FdoSmOvGeometricColumnType columnType = geomProp->GetGeometricColumnType();

            FdoPtr<FdoFgfGeometryFactory> gf = FdoFgfGeometryFactory::GetInstance();
            FdoGeometryValue *geomValue = NULL;
            FdoPtr<FdoIGeometry> geom;

            if (exp != NULL)
            {
                geomValue = dynamic_cast<FdoGeometryValue*>(exp.p);
                if ( geomValue ) 
                {
                    FdoPtr<FdoByteArray> ba = geomValue->GetGeometry();
                    if ( ba )
                    {
                        geom = gf->CreateGeometryFromFgf(ba);

                        if ( geom )
                        {
                            mConnection->GetSchemaUtil()->CheckGeomPropOrdDimensionality( currentClass, propertyDefinition->GetName(), geom );
                            mConnection->GetSchemaUtil()->CheckGeomPropShapeType( currentClass, propertyDefinition->GetName(), geom );
                            mConnection->GetSchemaUtil()->CheckGeomPropValidity( currentClass, propertyDefinition->GetName(), geom );
                        }
                    }
                }
            }

            switch (columnType)
            {
            case FdoSmOvGeometricColumnType_Default:
            case FdoSmOvGeometricColumnType_BuiltIn:
            case FdoSmOvGeometricColumnType_Blob:
            case FdoSmOvGeometricColumnType_Clob:
            case FdoSmOvGeometricColumnType_String:
                {
                    const FdoSmPhColumn *column = geomProp->RefColumn();
                    colName = column->GetName();
                    FdoInt64 srid = 0;
               
                    const FdoSmPhColumnP gColumn = ((FdoSmLpSimplePropertyDefinition*)geomProp)->GetColumn();
					FdoSmPhColumnGeomP geomCol;
                    if ( gColumn ) 
                        geomCol = gColumn.p->SmartCast<FdoSmPhColumnGeom>();
					if (geomCol)
                        srid = mFdoConnection->GetProcessedSRID(gColumn->GetTypeName(), geomCol->GetSRID());

                    bindPropNames->Add(qPropName);

                    if ( !exp )
                        exp = FdoGeometryValue::Create();

                    bindProps->push_back(std::make_pair((FdoLiteralValue*)exp.p, srid));
                }
                break;
            case FdoSmOvGeometricColumnType_Double:
                {
                    const FdoSmPhColumn *columnX = geomProp->RefColumnX();
                    const FdoSmPhColumn *columnY = geomProp->RefColumnY();
                    const FdoSmPhColumn *columnZ = geomProp->RefColumnZ();

                    FdoPtr<FdoDataValue> xValue;
                    FdoPtr<FdoDataValue> yValue;
                    FdoPtr<FdoDataValue> zValue;

                    if ( geom ) 
                    {
                        FdoGeometryType         geomType = geom->GetDerivedType();
                        if (FdoGeometryType_Point != geomType)
                            break;
                        FdoIPoint * pointValue = static_cast<FdoIPoint *>(geom.p);
                                
                        double x, y, z, m;
                        FdoInt32 dimensionality;
                        double doubleValue = 0.0;
 
                        pointValue->GetPositionByMembers(&x, &y, &z, &m, &dimensionality);
                        xValue = GetGeomOrdinateBindValue(x, columnX);
                        yValue = GetGeomOrdinateBindValue(y, columnY);
                        zValue = GetGeomOrdinateBindValue(z, columnZ);
                    }
                    else
                    {
                        xValue = FdoDoubleValue::Create();
                        yValue = FdoDoubleValue::Create();
                        zValue = FdoDoubleValue::Create();
                    }

                    if (NULL != columnX)
                    {
                        specialValues->Add(xValue);
                        bindProps->push_back(std::make_pair((FdoLiteralValue*)xValue.p, 0));
                    }

                    if (NULL != columnY)
                    {
                        specialValues->Add(yValue);
                        bindProps->push_back(std::make_pair((FdoLiteralValue*)yValue.p, 0));
                    }
                            
                    if (NULL != columnZ)
                    {
                        specialValues->Add(zValue);
                        bindProps->push_back(std::make_pair((FdoLiteralValue*)zValue.p, 0));
                    }
                }
                break;
            default:
                // Nothing useful to do here.
                break;
            }   // end of switch (columnType)

            // Set up spatial index columns.
            const FdoSmPhColumn *columnSi1 = geomProp->RefColumnSi1();
            const FdoSmPhColumn *columnSi2 = geomProp->RefColumnSi2();
            if (NULL != columnSi1 && NULL != columnSi2  && spatialManager != NULL)
            {
                FdoStringsP geomSiKeys;
                spatialManager->InsertGeometryInLine(geomProp, geomValue, geomSiKeys);

                if ( !geomSiKeys )
                    geomSiKeys = FdoStringCollection::Create();

                // If not all 2 si keys set, the just set extra ones to NULL.
                while ( geomSiKeys->GetCount() < 2 ) 
                    geomSiKeys->Add(L"");

                for ( int k = 0; k < geomSiKeys->GetCount(); k++ ) 
                {
                    FdoPtr<FdoStringValue> siValue = FdoStringValue::Create(geomSiKeys->GetString(k));
                    specialValues->Add(siValue);
                    bindPropNames->Add(qPropName);
                    bindProps->push_back(std::make_pair((FdoLiteralValue*)siValue.p, 0));
                }
            }
            break;
        }

        case FdoPropertyType_AssociationProperty:
        {
            const FdoSmLpAssociationPropertyDefinition* associationPropertyDefinition = static_cast<const FdoSmLpAssociationPropertyDefinition*>(propertyDefinition);
            if( associationPropertyDefinition->GetReadOnly() )
                break;
            //
            // We only add the column if the identity properties are not set; translation: we added axtra column
            // to link to the associated class. And if the identity properties are set, it mean we are using existing
            // properties/columns and there is no need to add any column as those column will be added as data properties.
            // For the latest case, we do some sanity check to make sure the duplicate entry(if any) are the same.
            const FdoStringsP   identCollection = associationPropertyDefinition->GetIdentityProperties();
            if( identCollection->GetCount() == 0 )
            {
                const FdoSmLpDataPropertyDefinitionCollection *identPropCol =  associationPropertyDefinition->RefAssociatedClass()->RefIdentityProperties();
                const FdoSmPhColumnListP identCols  = associationPropertyDefinition->GetReverseIdentityColumns();
                if( identCols->GetCount() != identPropCol->GetCount() )
                    throw FdoCommandException::Create(NlsMsgGet(FDORDBMS_292, "Association identity properties and identity columns mismatch"));

                for(int i=0; i<identCols->GetCount(); i++ )
                {
                    FdoString* revIdentColName = identCols->GetString(i);
                    const FdoSmLpDataPropertyDefinition *prop = identPropCol->RefItem( i );
                    SetBindVariable(currentClass, qPropName, propValCollection, queryDef, prop, revIdentColName);
                }
            }
        }
            break;

        case FdoPropertyType_ObjectProperty:
        {
            const FdoSmLpObjectPropertyDefinition* objProp = static_cast<const FdoSmLpObjectPropertyDefinition*>(propertyDefinition);
            const FdoSmLpPropertyMappingDefinition* mappping = objProp->RefMappingDefinition();
            if ( mappping->GetType() != FdoSmLpPropertyMappingType_Single )
                break; // We only handle the single mapping

            const FdoSmLpPropertyMappingSingle * singleMapping = static_cast<const FdoSmLpPropertyMappingSingle*>( mappping );
            FdoStringP newScope = propertyDefinition->GetName();
            if( scope[0] != '\0' )
                newScope = FdoStringP(scope) + L"." + propertyDefinition->GetName();
            SetBindVariables(objProp->RefTargetClass(),(const wchar_t*)newScope,  propValCollection, queryDef);
        }
            break;
        default:
            break;
    }
}

bool FdoRdbmsPvcInsertHandler::BindThisValue( FdoString* propName, FdoPropertyValueCollection  *propValCollection,FdoPtr<FdoValueExpression>& exp )
{
    bool ret = false;

    if ( propValCollection )
    {
        // Property values supplied, bind property only if it has a property value
        for (int j=0; j<propValCollection->GetCount(); j++)
        {
            FdoPtr<FdoPropertyValue> propertyValue = propValCollection->GetItem(j);
            if (!propertyValue)
                throw FdoCommandException::Create(NlsMsgGet(FDORDBMS_39, "Property value is NULL"));

            FdoPtr<FdoIdentifier> identifier = propertyValue->GetName();
            if (wcscmp(propName, identifier->GetText()) == 0)
            {
                ret = true;
                exp = propertyValue->GetValue();
            }
        }
    }
    else
    {
        // No property values supplied for checking, assume always binding property.
        exp = NULL;
        ret = true;
    }

    return ret;
}

InsertQueryDef *FdoRdbmsPvcInsertHandler::GetInsertQuery( const wchar_t *tableName, bool alloc_new )
{
    int         i;

	DbiConnection *mConnection = mFdoConnection->GetDbiConnection();

	if( ! alloc_new )
	{
		if( wcscmp( mLastTableName, tableName ) == 0 )
			return mLastInsertQuery;
		else
			wcscpy( mLastTableName, tableName );

		for( i=0; i<QUERY_CACHE_SIZE; i++ )
		{
			if( wcscmp( tableName, mInsertQueryCache[i].tableName ) == 0 )
			{
				mLastInsertQuery = &mInsertQueryCache[i];
				return &mInsertQueryCache[i];
			}
		}
	}
    // Find a free slot
    for( i=0; i<QUERY_CACHE_SIZE; i++ )
    {
        if( mInsertQueryCache[i].qid == -1 )
        {
            wcsncpy( mInsertQueryCache[i].tableName, tableName, GDBI_SCHEMA_ELEMENT_NAME_SIZE );
            mInsertQueryCache[i].tableName[GDBI_SCHEMA_ELEMENT_NAME_SIZE - 1] = '\0';
            mLastInsertQuery = &mInsertQueryCache[i];
            return &mInsertQueryCache[i];
        }
    }

    // We didn't find one free: Let's reuse an existing one.
    int    nextIdx = mNextQidToFree%QUERY_CACHE_SIZE;
    mNextQidToFree ++;

    // Free the resources allocated by the previous query
    if( mInsertQueryCache[nextIdx].qid != -1 )
        mConnection->GetGdbiCommands()->free_cursor(mInsertQueryCache[nextIdx].qid);

    mInsertQueryCache[nextIdx].qid = -1;

    // Mark it for the new table
    wcsncpy( mInsertQueryCache[nextIdx].tableName, tableName, GDBI_SCHEMA_ELEMENT_NAME_SIZE );
    mInsertQueryCache[nextIdx].tableName[GDBI_SCHEMA_ELEMENT_NAME_SIZE - 1] = '\0';
    mLastInsertQuery = &mInsertQueryCache[nextIdx];
    return &mInsertQueryCache[nextIdx];
}

void FdoRdbmsPvcInsertHandler::AssociationConstrainCheck( const FdoSmLpAssociationPropertyDefinition* propertyDefinition,
                                    FdoPropertyValueCollection  *propValCollection )
{
    const wchar_t* multiplicity = propertyDefinition->GetMultiplicity();
    const wchar_t* revMultiplicity = propertyDefinition->GetReverseMultiplicity();

    // The association property should be initialized if the multiplicity is "1"; We need to insert the asociation with the object.
    if( FdoRdbmsUtil::StrCmp( revMultiplicity, L"1") == 0 )
    {
        FdoPtr<FdoValueExpression> identVal;
        const FdoStringsP   identCollection = propertyDefinition->GetIdentityProperties();
        if( identCollection->GetCount() == 0 )
        {
            const FdoSmLpDataPropertyDefinitionCollection *identPropCol =  propertyDefinition->RefAssociatedClass()->RefIdentityProperties();
            for(int i=0; i<identPropCol->GetCount(); i++ )
            {
                const FdoSmLpDataPropertyDefinition *prop = identPropCol->RefItem( i );
                FdoStringP idenName = FdoStringP::Format(L"%ls.%ls",propertyDefinition->GetName(), prop->GetName());
                try
                {
                    FdoPtr<FdoPropertyValue> identPropertyValue = propValCollection->GetItem( (const wchar_t*)(idenName) );
                    identVal = identPropertyValue->GetValue();
                }
                catch(FdoException *exp )
                {
                    exp->Release();
                    identVal = NULL;
                    break;
                }
            }
        }
        else
        {
            const FdoStringsP   revIdentCollection = propertyDefinition->GetReverseIdentityProperties();
            for( int i=0; i<revIdentCollection->GetCount(); i++ )
            {
                try
                {
                    FdoPtr<FdoPropertyValue>identPropertyValue = propValCollection->GetItem( (const wchar_t*)(revIdentCollection->GetString(i)) );
                    identVal = identPropertyValue->GetValue();
                }
                catch(FdoException *exp )
                {
                    exp->Release();
                    identVal = NULL;
                    break;
                }
            }
        }
        if( identVal == NULL )
        {
            throw FdoCommandException::Create(NlsMsgGet(FDORDBMS_289, "Association property is required" ) );
        }
    }

    // Only one-to-one association exists; make sure we don't already have an association to the same instance.
    if( FdoRdbmsUtil::StrCmp( multiplicity, L"1") == 0 )
    {
        // We need to make a query to see if an association between the associated instance and an instance of this
        // class exists already.
        // This is potentially a performance killer as a select would trigger the flush of the cach.
        // Also, this is an issue if subclassing is involved as any number of subclasses can be
        // Associsted to this instance. That means we need to search every subclass instances for refrence
        // to the associated instance.
    }

    //TODO We need to make sure that the associated instance exists. This is a performace issue as we are forcing a cach flush.
    // May be we should ignore the existance of the associated instance is the multiplicity is set to "m".

}

FdoSmLpPropertyDefinitionCollection *FdoRdbmsPvcInsertHandler::MoveGeometryProperties( const FdoSmLpClassDefinition *currentClass )
{
    const   FdoSmLpPropertyDefinitionCollection *propertyDefinitions = currentClass->RefProperties();

    FdoSmLpPropertyDefinitionCollection *propertyDefinitions2 = new FdoSmLpPropertyDefinitionCollection();
    
    for ( int i = 0; i < propertyDefinitions->GetCount(); i++ )
    {            
        if ( propertyDefinitions->RefItem(i)->GetPropertyType() != FdoPropertyType_GeometricProperty )
            propertyDefinitions2->Add((FdoSmLpPropertyDefinition *)propertyDefinitions->RefItem(i));
    }
    for ( int i = 0; i < propertyDefinitions->GetCount(); i++ )
    {           
        if ( propertyDefinitions->RefItem(i)->GetPropertyType() == FdoPropertyType_GeometricProperty )
            propertyDefinitions2->Add((FdoSmLpPropertyDefinition *)propertyDefinitions->RefItem(i));
    }
    // Caller should free the collection
    return propertyDefinitions2;
}

