// Copyright (C) 2004-2006  Autodesk, Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of version 2.1 of the GNU Lesser
// General Public License as published by the Free Software Foundation.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;

using OSGeo.FDO.Connections;
using OSGeo.FDO.Commands;
using OSGeo.FDO.Commands.Schema;
using OSGeo.FDO.Commands.Feature;
using OSGeo.FDO.Schema;
using OSGeo.FDO.Expression;
using OSGeo.Geometry;

using unit_test.Framework;

namespace unit_test.ProviderTests
{
    class InsertTests : BaseTestWithConnection
    {
        public InsertTests(ShpTestProvider connectionProvider)
            : base(connectionProvider)
        { }

        public override void SetUp()
        {
            //
            if(!Directory.Exists(GetLocation()))
            {
                Directory.CreateDirectory(GetLocation());
            }

            base.SetUp() ;
        }

        public override void TearDown()
        {
            // Clean up leftover class, if any:
            ShpTests.CleanUpClass(base.ConnectionInitialized, null, className);

            base.TearDown() ;

            //
            if (File.Exists(GetLocation() + "schema.xml"))
            {
                File.Delete(GetLocation() + "schema.xml");
            }
            if (Directory.Exists(GetLocation()))
            {
                Directory.Delete(GetLocation());
            }
        }

        public enum GeometricType
        {
            GeometricType_Point = 0x01,
            GeometricType_Curve = 0x02,
            GeometricType_Surface = 0x04,
            GeometricType_Solid = 0x08
        };

        const string schemaName = "TheSchema";
        const string className = "Test";

        private void create_schema (GeometricType type, bool elevation, bool measure)
        {
            // Delete old class, if its still there:
            ShpTests.CleanUpClass(base.ConnectionInitialized, null, className);

                
            DataPropertyDefinition FeatId = new DataPropertyDefinition("FeatId", "integer");
            FeatId.DataType = DataType.DataType_Int32;
            FeatId.IsAutoGenerated = true ;
            FeatId.Nullable = false ;

            DataPropertyDefinition Id = new DataPropertyDefinition("Id", "decimal");
            Id.DataType = DataType.DataType_Decimal;
            Id.Precision = 10 ;
            Id.Scale = 0 ;
        
            DataPropertyDefinition street = new DataPropertyDefinition("Street", "text");
            street.DataType = DataType.DataType_String;
            street.Length = 64;

            DataPropertyDefinition area = new DataPropertyDefinition("Area", "double");
            area.DataType = DataType.DataType_Decimal;
            area.Precision = 20;
            area.Scale = 8;
       
            DataPropertyDefinition vacant = new DataPropertyDefinition("Vacant", "boolean");
            vacant.DataType = DataType.DataType_Boolean;

            DataPropertyDefinition birthday = new DataPropertyDefinition("Birthday", "date");
            birthday.DataType = DataType.DataType_DateTime;

            // build a location geometry property
            GeometricPropertyDefinition location = new GeometricPropertyDefinition ("Geometry", "geometry");
            location.GeometryTypes = (int)type;
            location.HasElevation = true;
            location.HasMeasure = true;

            // assemble the feature class
            FeatureClass feature = new FeatureClass (className, "test class created with apply schema");
            PropertyDefinitionCollection properties = feature.Properties;
            properties.Add (FeatId);
            properties.Add (Id);
            properties.Add (street);
            properties.Add (area);
            properties.Add (vacant);
            properties.Add (birthday);
            properties.Add (location);
            feature.GeometryProperty = location;
            DataPropertyDefinitionCollection identities = feature.IdentityProperties;
            identities.Add (FeatId);
            
            FeatureSchema schema = new FeatureSchema (schemaName, "");
            ClassCollection classes = schema.Classes;
            classes.Add(feature) ;

            // submit the new schema
            
            IApplySchema apply = (IApplySchema)base.ConnectionInitialized.CreateCommand (CommandType.CommandType_ApplySchema);
            apply.FeatureSchema = schema;
            apply.Execute ();
        }

        public void Test_insert_geometry_multipolygon_xyzm () //fail
        {
            try
            {
                create_schema (GeometricType.GeometricType_Surface, true, true);

                IInsert insert = (IInsert)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_Insert);
                insert.SetFeatureClassName (className);

                PropertyValueCollection values = insert.PropertyValues ;

                ValueExpression expression = (ValueExpression)ShpTests.ParseByDataType ("7", DataType.DataType_Decimal);
                PropertyValue value = new PropertyValue("Id", expression);
                values.Add (value);

                expression = (ValueExpression)Expression.Parse ("'Linux Loop'");
                value = new PropertyValue("Street", expression);
                values.Add (value);

                expression = new DecimalValue(99.9999);
                value = new PropertyValue("Area", expression);
                values.Add (value);

                expression = (ValueExpression)Expression.Parse ("true");
                value = new PropertyValue("Vacant", expression);
                values.Add (value);

                expression = (ValueExpression)Expression.Parse ("DATE'2005-03-22'");
                value = new PropertyValue("Birthday", expression);
                values.Add (value);

                // add geometry value:
                GeometryValue geometry = (GeometryValue)Expression.Parse ("GeomFromText('MULTIPOLYGON XYZM (((5100 5230 20 1, 5100 5030 20 2, 5300 5030 20 3, 5300 5230 20 1, 5100 5230 20 1), (5239 5119 20 1, 5239 5141 20 2, 5261 5141 20 3, 5261 5119 20 1, 5239 5119 20 1), (5141 5135 20 1, 5141 5153 20 2, 5159 5153 20 3, 5159 5135 20 1, 5141 5135 20 1)), ((6100 5230 20 1, 6100 5030 20 2, 6300 5030 20 3, 6300 5230 20 1, 6100 5230 20 1), (6239 5119 20 1, 6239 5141 20 2, 6261 5141 20 3, 6261 5119 20 1, 6239 5119 20 1), (6141 5135 20 1, 6141 5153 20 2, 6159 5153 20 3, 6159 5135 20 1, 6141 5135 20 1)))')");
                value = new PropertyValue("Geometry", geometry);
                values.Add (value);

                IFeatureReader reader = insert.Execute ();
                int featid;
                featid = -1;
                while (reader.ReadNext ())
                {
                    if (-1 != featid)
                        Debug.Fail("too many features inserted");
                    featid = reader.GetInt32 ("FeatId");
                }
                reader.Close ();
                if (-1 == featid)
                    Debug.Fail("too few features inserted");

                // check by doing a select

                ISelect select = (ISelect)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_Select);
                select.SetFeatureClassName (className);
                reader = select.Execute ();
                while (reader.ReadNext ())
                {
                    unit_test_assert ("incorrect featid value", featid == reader.GetInt32 ("FeatId"));
                    unit_test_assert ("incorrect id value", 7 == reader.GetDouble ("Id"));
                    unit_test_assert ("incorrect street value", 0 == string.Compare("Linux Loop", reader.GetString ("Street")));
                    unit_test_assert ("incorrect area value", 99.9999 == reader.GetDouble ("Area"));
                    unit_test_assert ("incorrect vacant value", reader.GetBoolean ("Vacant"));
                    unit_test_assert ("incorrect birthday value", 2005 == reader.GetDateTime ("Birthday").Year);
                    unit_test_assert ("incorrect birthday value", 3 == reader.GetDateTime ("Birthday").Month);
                    unit_test_assert ("incorrect birthday value", 22 == reader.GetDateTime ("Birthday").Day);
                    unit_test_assert ("geometry is null", !reader.IsNull ("Geometry"));

                    //test geometry data
                    int beforeLen = geometry.Geometry.Length ;
                    Byte [] before = new Byte[beforeLen] ;
                    before = geometry.Geometry ;

                    int afterLen = reader.GetGeometry ("Geometry").Length ;
                    Byte [] after = new Byte[afterLen] ;
                    after = reader.GetGeometry ("Geometry") ;

                    unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                    for (int i = 0; i < beforeLen; i++)
                    {
                        Byte left = before[i];
                        Byte right = after[i];
                        unit_test_assert ("incorrect geometry content", left == right);
                    }
                }
                reader.Close ();

                // close and reopen the connection
                base.ConnectionInitialized.Close();
                unit_test_assert("connection state not open", ConnectionState.ConnectionState_Open == base.ConnectionInitialized.Open());

                // check by doing a select
                select = (ISelect)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_Select);
                select.SetFeatureClassName (className);
                reader = select.Execute ();
                while (reader.ReadNext ())
                {
                    unit_test_assert ("incorrect featid value", featid == reader.GetInt32 ("FeatId"));
                    unit_test_assert ("incorrect id value", 7 == reader.GetDouble ("Id"));
                    unit_test_assert ("incorrect street value", 0 == string.Compare("Linux Loop", reader.GetString ("Street")));
                    unit_test_assert ("incorrect area value", 99.9999 == reader.GetDouble ("Area"));
                    unit_test_assert ("incorrect vacant value", reader.GetBoolean ("Vacant"));
                    unit_test_assert ("incorrect birthday value", 2005 == reader.GetDateTime ("Birthday").Year);
                    unit_test_assert ("incorrect birthday value", 3 == reader.GetDateTime ("Birthday").Month);
                    unit_test_assert ("incorrect birthday value", 22 == reader.GetDateTime ("Birthday").Day);
                    unit_test_assert ("geometry is null", !reader.IsNull ("Geometry"));

                    //test geometry data
                    int beforeLen = geometry.Geometry.Length ;
                    Byte [] before = new Byte[beforeLen] ;
                    before = geometry.Geometry ;

                    int afterLen = reader.GetGeometry ("Geometry").Length ;
                    Byte [] after = new Byte[afterLen] ;
                    after = reader.GetGeometry ("Geometry") ;

                    unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                    for (int i = 0; i < beforeLen; i++)
                    {
                        Byte left = before[i];
                        Byte right = after[i];
                        unit_test_assert("incorrect geometry content", left == right);
                    }
                }

                reader.Close ();
            }
            catch (OSGeo.Common.Exception e) 
            {
                ShpTests.Fail (e);
            }
        }

        public void Test_batch_insert () //fail
        {
            try
            {
                create_schema (GeometricType.GeometricType_Surface, true, true);

                IInsert insert = (IInsert)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_Insert);
                insert.SetFeatureClassName (className);

                PropertyValueCollection values = insert.PropertyValues ;

                //Set up the property value collection:
                Parameter parameter;
                PropertyValue value;
                
                parameter = new Parameter("Param0");
                value = new PropertyValue("Id", parameter);
                values.Add (value);

                parameter = new Parameter("Param1");
                value = new PropertyValue("Street", parameter);
                values.Add(value);

                parameter = new Parameter("Param2");
                value = new PropertyValue("Area", parameter);
                values.Add(value);
                
                parameter = new Parameter("Param3");
                value = new PropertyValue("Vacant", parameter);
                values.Add (value);
                
                parameter = new Parameter("Param4");
                value = new PropertyValue("Birthday", parameter);
                values.Add (value);
                
                parameter = new Parameter("Param5");
                value = new PropertyValue("Geometry", parameter);
                values.Add (value);
                
                // fill the batch parameter value collection with multiple (2) rows of data
                BatchParameterValueCollection parameters = insert.BatchParameterValues ;
                ParameterValue parameterValue;
                LiteralValue valueExpression ;
                
                // row 1
                ParameterValueCollection collection = new ParameterValueCollection() ;

                //valueExpression = (LiteralValue)ShpTests.ParseByDataType ("7", DataType.DataType_Decimal);
                valueExpression = new DecimalValue(7) ;
                parameterValue = new ParameterValue("Param0", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("'4 Linux Loop'");
                parameterValue = new ParameterValue("Param1", valueExpression);
                collection.Add (parameterValue);
                
                valueExpression = new DecimalValue(256.00);
                parameterValue = new ParameterValue("Param2", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("true");
                parameterValue = new ParameterValue("Param3", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("DATE'2005-04-18'");
                parameterValue = new ParameterValue("Param4", valueExpression);
                collection.Add (parameterValue);

                GeometryValue geometry1 = (GeometryValue)Expression.Parse("GEOMFROMTEXT ('LINESTRING XY ( 7171.723 8282.99, 6824.82 6545.87, 8920.5 9929.77)')");
                parameterValue = new ParameterValue("Param5", geometry1);
                collection.Add (parameterValue);

                parameters.Add (collection);

                // row 2
                //valueExpression = (LiteralValue)ShpTests.ParseByDataType ("7", DataType.DataType_Decimal);
                valueExpression = new DecimalValue(11) ;
                parameterValue = new ParameterValue("Param0", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("'84 Windows Way'");
                parameterValue = new ParameterValue("Param1", valueExpression);
                collection.Add (parameterValue);
                
                valueExpression = new DecimalValue(128.00);
                parameterValue = new ParameterValue("Param2", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("false");
                parameterValue = new ParameterValue("Param3", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("DATE'2005-04-16'");
                parameterValue = new ParameterValue("Param4", valueExpression);
                collection.Add (parameterValue);

                GeometryValue geometry2 = (GeometryValue)Expression.Parse ("GEOMFROMTEXT ('LINESTRING XY ( 6677.99 7820.89, 6272.90 7020.20, 7012.82 7516.91)')");
                parameterValue = new ParameterValue("Param5", geometry2);
                collection.Add (parameterValue);

                parameters.Add (collection);

                // perform the batch insert
                IFeatureReader reader = insert.Execute ();

                // check the id's
                int featid1 = -1;
                int featid2 = -1;
                while (reader.ReadNext ())
                {
                    if (-1 != featid1)
                        if (-1 != featid2)
                            Debug.Fail("too many features inserted");
                        else 
                            featid2 = reader.GetInt32 ("FeatId");
                    else
                        featid1 = reader.GetInt32 ("FeatId");
                }
                reader.Close ();
                if (-1 == featid2)
                    Debug.Fail("too few features inserted");

                // check by doing a select

                ISelect select = (ISelect)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_Select);
                select.SetFeatureClassName (className);
                reader = select.Execute ();
                while (reader.ReadNext ())
                {
                    if (featid1 == reader.GetInt32 ("FeatId"))
                    {
                        unit_test_assert ("incorrect id value", 7 == reader.GetDouble ("Id"));
                        unit_test_assert("incorrect street value", 0 == string.Compare("4 Linux Loop", reader.GetString("Street")));
                        unit_test_assert ("incorrect area value", 256.00 == reader.GetDouble ("Area"));
                        unit_test_assert ("incorrect vacant value", reader.GetBoolean ("Vacant"));
                        unit_test_assert ("incorrect birthday value", 2005 == reader.GetDateTime ("Birthday").Year);
                        unit_test_assert ("incorrect birthday value", 4 == reader.GetDateTime ("Birthday").Month);
                        unit_test_assert ("incorrect birthday value", 18 == reader.GetDateTime ("Birthday").Day);
                        unit_test_assert ("geometry is null", !reader.IsNull ("Geometry"));

                        //test geometry data
                        int beforeLen = geometry1.Geometry.Length ;
                        Byte [] before = new Byte[beforeLen] ;
                        before = geometry1.Geometry ;

                        int afterLen = reader.GetGeometry ("Geometry").Length ;
                        Byte [] after = new Byte[afterLen] ;
                        after = reader.GetGeometry ("Geometry") ;

                        unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                        for (int i = 0; i < beforeLen; i++)
                        {
                            Byte left = before[i];
                            Byte right = after[i];
                            unit_test_assert ("incorrect geometry content", left == right);
                        }
                    }
                    else if (featid2 == reader.GetInt32 ("FeatId"))
                    {
                        unit_test_assert ("incorrect id value", 11 == reader.GetDouble ("Id"));
                        unit_test_assert("incorrect street value", 0 == string.Compare("84 Windows Way", reader.GetString("Street")));
                        unit_test_assert ("incorrect area value", 128.00 == reader.GetDouble ("Area"));
                        unit_test_assert ("incorrect vacant value", !reader.GetBoolean ("Vacant"));
                        unit_test_assert ("incorrect birthday value", 2005 == reader.GetDateTime ("Birthday").Year);
                        unit_test_assert ("incorrect birthday value", 4 == reader.GetDateTime ("Birthday").Month);
                        unit_test_assert ("incorrect birthday value", 16 == reader.GetDateTime ("Birthday").Day);
                        unit_test_assert ("geometry is null", !reader.IsNull ("Geometry"));

                        //test geometry data
                        int beforeLen = geometry2.Geometry.Length ;
                        Byte [] before = new Byte[beforeLen] ;
                        before = geometry2.Geometry ;

                        int afterLen = reader.GetGeometry ("Geometry").Length ;
                        Byte [] after = new Byte[afterLen] ;
                        after = reader.GetGeometry ("Geometry") ;

                        unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                        for (int i = 0; i < beforeLen; i++)
                        {
                            Byte left = before[i];
                            Byte right = after[i];
                            unit_test_assert ("incorrect geometry content", left == right);
                        }
                    }
                }
                reader.Close ();
         
                // close and reopen the connection
                // close and reopen the connection
                base.ConnectionInitialized.Close();
                unit_test_assert("connection state not open", ConnectionState.ConnectionState_Open == base.ConnectionInitialized.Open());

                // check by doing a select
                select = (ISelect)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_Select);
                select.SetFeatureClassName (className);
                reader = select.Execute ();
                while (reader.ReadNext ())
                {
                    if (featid1 == reader.GetInt32 ("FeatId"))
                    {
                        unit_test_assert ("incorrect id value", 7 == reader.GetDouble ("Id"));
                        unit_test_assert ("incorrect street value", 0 == string.Compare("4 Linux Loop", reader.GetString ("Street")));
                        unit_test_assert ("incorrect area value", 256.00 == reader.GetDouble ("Area"));
                        unit_test_assert ("incorrect vacant value", reader.GetBoolean ("Vacant"));
                        unit_test_assert ("incorrect birthday value", 2005 == reader.GetDateTime ("Birthday").Year);
                        unit_test_assert ("incorrect birthday value", 4 == reader.GetDateTime ("Birthday").Month);
                        unit_test_assert ("incorrect birthday value", 18 == reader.GetDateTime ("Birthday").Day);
                        unit_test_assert ("geometry is null", !reader.IsNull ("Geometry"));

                        //test geometry data
                        int beforeLen = geometry1.Geometry.Length ;
                        Byte [] before = new Byte[beforeLen] ;
                        before = geometry1.Geometry ;

                        int afterLen = reader.GetGeometry ("Geometry").Length ;
                        Byte [] after = new Byte[afterLen] ;
                        after = reader.GetGeometry ("Geometry") ;

                        unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                        for (int i = 0; i < beforeLen; i++)
                        {
                            Byte left = before[i];
                            Byte right = after[i];
                            unit_test_assert ("incorrect geometry content", left == right);
                        }
                    }
                    else if (featid2 == reader.GetInt32 ("FeatId"))
                    {
                        unit_test_assert ("incorrect id value", 11 == reader.GetDouble ("Id"));
                        unit_test_assert("incorrect street value", 0 == string.Compare("84 Windows Way", reader.GetString("Street")));
                        unit_test_assert ("incorrect area value", 128.00 == reader.GetDouble ("Area"));
                        unit_test_assert ("incorrect vacant value", !reader.GetBoolean ("Vacant"));
                        unit_test_assert ("incorrect birthday value", 2005 == reader.GetDateTime ("Birthday").Year);
                        unit_test_assert ("incorrect birthday value", 4 == reader.GetDateTime ("Birthday").Month);
                        unit_test_assert ("incorrect birthday value", 16 == reader.GetDateTime ("Birthday").Day);
                        unit_test_assert ("geometry is null", !reader.IsNull ("Geometry"));

                        //test geometry data
                        int beforeLen = geometry2.Geometry.Length ;
                        Byte [] before = new Byte[beforeLen] ;
                        before = geometry2.Geometry ;

                        int afterLen = reader.GetGeometry ("Geometry").Length ;
                        Byte [] after = new Byte[afterLen] ;
                        after = reader.GetGeometry ("Geometry") ;

                        unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                        for (int i = 0; i < beforeLen; i++)
                        {
                            Byte left = before[i];
                            Byte right = after[i];
                            unit_test_assert ("incorrect geometry content", left == right);
                        }
                    }
                }
                reader.Close ();         
            }
            catch (OSGeo.Common.Exception e) 
            {
                ShpTests.Fail (e);
            }
        }

        public void Test_null_geometry_insert() //fail
        {
            try
            {
                create_schema (GeometricType.GeometricType_Surface, true, true);

                IInsert insert = (IInsert)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_Insert);
                insert.SetFeatureClassName (className);

                PropertyValueCollection values = insert.PropertyValues ;

                //Set up the property value collection:
                Parameter parameter;
                PropertyValue value;
                
                parameter = new Parameter("Param0");
                value = new PropertyValue("Id", parameter);
                values.Add (value);

                parameter = new Parameter("Param1");
                value = new PropertyValue("Street", parameter);
                values.Add(value);

                parameter = new Parameter("Param2");
                value = new PropertyValue("Area", parameter);
                values.Add(value);
                
                parameter = new Parameter("Param3");
                value = new PropertyValue("Vacant", parameter);
                values.Add (value);
                
                parameter = new Parameter("Param4");
                value = new PropertyValue("Birthday", parameter);
                values.Add (value);
                
                parameter = new Parameter("Param5");
                value = new PropertyValue("Geometry", parameter);
                values.Add (value);
                
                // fill the batch parameter value collection with multiple (2) rows of data
                BatchParameterValueCollection parameters = insert.BatchParameterValues ;
                ParameterValue parameterValue;
                LiteralValue valueExpression ;
                
                // row 1
                ParameterValueCollection collection = new ParameterValueCollection() ;

                //valueExpression = (LiteralValue)ShpTests.ParseByDataType ("7", DataType.DataType_Decimal);
                valueExpression = new DecimalValue(7) ;
                parameterValue = new ParameterValue("Param0", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("'4 Linux Loop'");
                parameterValue = new ParameterValue("Param1", valueExpression);
                collection.Add (parameterValue);
                
                valueExpression = new DecimalValue(256.00);
                parameterValue = new ParameterValue("Param2", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("true");
                parameterValue = new ParameterValue("Param3", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("DATE'2005-04-18'");
                parameterValue = new ParameterValue("Param4", valueExpression);
                collection.Add (parameterValue);

                GeometryValue geometry1 = (GeometryValue)Expression.Parse("GEOMFROMTEXT ('LINESTRING XY ( 7171.723 8282.99, 6824.82 6545.87, 8920.5 9929.77)')");
                parameterValue = new ParameterValue("Param5", geometry1);
                collection.Add (parameterValue);

                parameters.Add (collection);

                // row 2
                //valueExpression = (LiteralValue)ShpTests.ParseByDataType ("7", DataType.DataType_Decimal);
                valueExpression = new DecimalValue(11) ;
                parameterValue = new ParameterValue("Param0", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("'84 Windows Way'");
                parameterValue = new ParameterValue("Param1", valueExpression);
                collection.Add (parameterValue);
                
                valueExpression = new DecimalValue(128.00);
                parameterValue = new ParameterValue("Param2", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("false");
                parameterValue = new ParameterValue("Param3", valueExpression);
                collection.Add (parameterValue);

                valueExpression = (LiteralValue)Expression.Parse ("DATE'2005-04-16'");
                parameterValue = new ParameterValue("Param4", valueExpression);
                collection.Add (parameterValue);

                //GeometryValue geometry2 = (GeometryValue)Expression.Parse ("GEOMFROMTEXT ('LINESTRING XY ( 6677.99 7820.89, 6272.90 7020.20, 7012.82 7516.91)')");
                //parameterValue = new ParameterValue("Param5", geometry2);
                //collection.Add (parameterValue);

                parameters.Add (collection);

                // row 3
                //valueExpression = (LiteralValue)ShpTests.ParseByDataType ("7", DataType.DataType_Decimal);
                valueExpression = new DecimalValue(11);
                parameterValue = new ParameterValue("Param0", valueExpression);
                collection.Add(parameterValue);

                valueExpression = (LiteralValue)Expression.Parse("'84 Windows Way'");
                parameterValue = new ParameterValue("Param1", valueExpression);
                collection.Add(parameterValue);

                valueExpression = new DecimalValue(128.00);
                parameterValue = new ParameterValue("Param2", valueExpression);
                collection.Add(parameterValue);

                valueExpression = (LiteralValue)Expression.Parse("false");
                parameterValue = new ParameterValue("Param3", valueExpression);
                collection.Add(parameterValue);

                valueExpression = (LiteralValue)Expression.Parse("DATE'2005-04-16'");
                parameterValue = new ParameterValue("Param4", valueExpression);
                collection.Add(parameterValue);

                GeometryValue geometry2 = (GeometryValue)Expression.Parse("GEOMFROMTEXT ('LINESTRING XY ( 6677.99 7820.89, 6272.90 7020.20, 7012.82 7516.91)')");
                parameterValue = new ParameterValue("Param5", geometry2);
                collection.Add(parameterValue);
                 
                parameters.Add(collection);

                // perform the batch insert
                IFeatureReader reader = insert.Execute ();

                // check the id's
                int featid1 = -1;
                int featid2 = -1;
                int featid3 = -1;
                while (reader.ReadNext ())
                {
                    if (-1 != featid1)
                        if (-1 != featid2)
                            if (-1 != featid3)
                                Debug.Fail("too many features inserted");
                            else
                                featid3 = reader.GetInt32("FeatId");
                        else 
                            featid2 = reader.GetInt32 ("FeatId");
                    else
                        featid1 = reader.GetInt32 ("FeatId");
                }
                reader.Close ();
                if (-1 == featid3)
                    Debug.Fail("too few features inserted");

                // check by doing a select
                ISelect select = (ISelect)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_Select);
                select.SetFeatureClassName (className);
                reader = select.Execute ();
                while (reader.ReadNext ())
                {
                    if (featid1 == reader.GetInt32 ("FeatId"))
                    {
                        unit_test_assert ("incorrect id value", 7 == reader.GetDouble ("Id"));
                        unit_test_assert("incorrect street value", 0 == string.Compare("4 Linux Loop", reader.GetString("Street")));
                        unit_test_assert ("incorrect area value", 256.00 == reader.GetDouble ("Area"));
                        unit_test_assert ("incorrect vacant value", reader.GetBoolean ("Vacant"));
                        unit_test_assert ("incorrect birthday value", 2005 == reader.GetDateTime ("Birthday").Year);
                        unit_test_assert ("incorrect birthday value", 4 == reader.GetDateTime ("Birthday").Month);
                        unit_test_assert ("incorrect birthday value", 18 == reader.GetDateTime ("Birthday").Day);
                        unit_test_assert ("geometry is null", !reader.IsNull ("Geometry"));

                        //test geometry data
                        int beforeLen = geometry1.Geometry.Length ;
                        Byte [] before = new Byte[beforeLen] ;
                        before = geometry1.Geometry ;

                        int afterLen = reader.GetGeometry ("Geometry").Length ;
                        Byte [] after = new Byte[afterLen] ;
                        after = reader.GetGeometry ("Geometry") ;

                        unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                        for (int i = 0; i < beforeLen; i++)
                        {
                            Byte left = before[i];
                            Byte right = after[i];
                            unit_test_assert ("incorrect geometry content", left == right);
                        }
                    }
                    else if (featid2 == reader.GetInt32 ("FeatId"))
                    {
                        unit_test_assert ("incorrect id value", 11 == reader.GetDouble ("Id"));
                        unit_test_assert("incorrect street value", 0 == string.Compare("84 Windows Way", reader.GetString("Street")));
                        unit_test_assert ("incorrect area value", 128.00 == reader.GetDouble ("Area"));
                        unit_test_assert ("incorrect vacant value", !reader.GetBoolean ("Vacant"));
                        unit_test_assert ("incorrect birthday value", 2005 == reader.GetDateTime ("Birthday").Year);
                        unit_test_assert ("incorrect birthday value", 4 == reader.GetDateTime ("Birthday").Month);
                        unit_test_assert ("incorrect birthday value", 16 == reader.GetDateTime ("Birthday").Day);
                        unit_test_assert ("geometry is null", reader.IsNull ("Geometry"));
                    }
                    else if (featid3 == reader.GetInt32 ("FeatId"))
                    {
                        unit_test_assert ("incorrect id value", 11 == reader.GetDouble ("Id"));
                        unit_test_assert("incorrect street value", 0 == string.Compare("84 Windows Way", reader.GetString("Street")));
                        unit_test_assert ("incorrect area value", 128.00 == reader.GetDouble ("Area"));
                        unit_test_assert ("incorrect vacant value", !reader.GetBoolean ("Vacant"));
                        unit_test_assert ("incorrect birthday value", 2005 == reader.GetDateTime ("Birthday").Year);
                        unit_test_assert ("incorrect birthday value", 4 == reader.GetDateTime ("Birthday").Month);
                        unit_test_assert ("incorrect birthday value", 16 == reader.GetDateTime ("Birthday").Day);
                        unit_test_assert ("geometry is null", !reader.IsNull ("Geometry"));

                        //test geometry data
                        int beforeLen = geometry2.Geometry.Length ;
                        Byte [] before = new Byte[beforeLen] ;
                        before = geometry2.Geometry ;

                        int afterLen = reader.GetGeometry ("Geometry").Length ;
                        Byte [] after = new Byte[afterLen] ;
                        after = reader.GetGeometry ("Geometry") ;

                        unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                        for (int i = 0; i < beforeLen; i++)
                        {
                            Byte left = before[i];
                            Byte right = after[i];
                            unit_test_assert ("incorrect geometry content", left == right);
                        }
                    }
                }
                reader.Close ();
         
                // close and reopen the connection
                // close and reopen the connection
                base.ConnectionInitialized.Close();
                unit_test_assert("connection state not open", ConnectionState.ConnectionState_Open == base.ConnectionInitialized.Open());

                // check by doing a select
                select = (ISelect)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_Select);
                select.SetFeatureClassName (className);
                reader = select.Execute ();
                while (reader.ReadNext ())
                {
                    if (featid1 == reader.GetInt32("FeatId"))
                    {
                        unit_test_assert("incorrect id value", 7 == reader.GetDouble("Id"));
                        unit_test_assert("incorrect street value", 0 == string.Compare("4 Linux Loop", reader.GetString("Street")));
                        unit_test_assert("incorrect area value", 256.00 == reader.GetDouble("Area"));
                        unit_test_assert("incorrect vacant value", reader.GetBoolean("Vacant"));
                        unit_test_assert("incorrect birthday value", 2005 == reader.GetDateTime("Birthday").Year);
                        unit_test_assert("incorrect birthday value", 4 == reader.GetDateTime("Birthday").Month);
                        unit_test_assert("incorrect birthday value", 18 == reader.GetDateTime("Birthday").Day);
                        unit_test_assert("geometry is null", !reader.IsNull("Geometry"));

                        //test geometry data
                        int beforeLen = geometry1.Geometry.Length;
                        Byte[] before = new Byte[beforeLen];
                        before = geometry1.Geometry;

                        int afterLen = reader.GetGeometry("Geometry").Length;
                        Byte[] after = new Byte[afterLen];
                        after = reader.GetGeometry("Geometry");

                        unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                        for (int i = 0; i < beforeLen; i++)
                        {
                            Byte left = before[i];
                            Byte right = after[i];
                            unit_test_assert("incorrect geometry content", left == right);
                        }
                    }
                    else if (featid2 == reader.GetInt32("FeatId"))
                    {
                        unit_test_assert("incorrect id value", 11 == reader.GetDouble("Id"));
                        unit_test_assert("incorrect street value", 0 == string.Compare("84 Windows Way", reader.GetString("Street")));
                        unit_test_assert("incorrect area value", 128.00 == reader.GetDouble("Area"));
                        unit_test_assert("incorrect vacant value", !reader.GetBoolean("Vacant"));
                        unit_test_assert("incorrect birthday value", 2005 == reader.GetDateTime("Birthday").Year);
                        unit_test_assert("incorrect birthday value", 4 == reader.GetDateTime("Birthday").Month);
                        unit_test_assert("incorrect birthday value", 16 == reader.GetDateTime("Birthday").Day);
                        unit_test_assert("geometry is null", reader.IsNull("Geometry"));
                    }
                    else if (featid3 == reader.GetInt32("FeatId"))
                    {
                        unit_test_assert("incorrect id value", 11 == reader.GetDouble("Id"));
                        unit_test_assert("incorrect street value", 0 == string.Compare("84 Windows Way", reader.GetString("Street")));
                        unit_test_assert("incorrect area value", 128.00 == reader.GetDouble("Area"));
                        unit_test_assert("incorrect vacant value", !reader.GetBoolean("Vacant"));
                        unit_test_assert("incorrect birthday value", 2005 == reader.GetDateTime("Birthday").Year);
                        unit_test_assert("incorrect birthday value", 4 == reader.GetDateTime("Birthday").Month);
                        unit_test_assert("incorrect birthday value", 16 == reader.GetDateTime("Birthday").Day);
                        unit_test_assert("geometry is null", !reader.IsNull("Geometry"));

                        //test geometry data
                        int beforeLen = geometry2.Geometry.Length;
                        Byte[] before = new Byte[beforeLen];
                        before = geometry2.Geometry;

                        int afterLen = reader.GetGeometry("Geometry").Length;
                        Byte[] after = new Byte[afterLen];
                        after = reader.GetGeometry("Geometry");

                        unit_test_assert("incorrect geometry size", beforeLen == afterLen);
                        for (int i = 0; i < beforeLen; i++)
                        {
                            Byte left = before[i];
                            Byte right = after[i];
                            unit_test_assert("incorrect geometry content", left == right);
                        }
                    }
                }
                reader.Close ();         
            }
            catch (OSGeo.Common.Exception e) 
            {
                ShpTests.Fail (e);
            }
        }
    }
}
