/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "Pch.h"
#include "SpatialContextTests.h"

CPPUNIT_TEST_SUITE_REGISTRATION (SpatialContextTests);
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION (SpatialContextTests, "SpatialContextTests");

FdoPtr<FdoIConnection> SpatialContextTests::mConnection;

static const wchar_t*    spatial_context_name = L"SCTest1s";
static const wchar_t*    spatial_context_desc = L"A spatial context";
static const wchar_t*    spatial_context_coordsysID1 = L"20250";
static const double      spatial_context_minx = -1000071.0;
static const double      spatial_context_maxx =  1000071.0;
static const double      spatial_context_miny = -1000071.0;
static const double      spatial_context_maxy =  1000071.0;
static const FdoSpatialContextExtentType spatial_context_extent_type = FdoSpatialContextExtentType_Static;
static const double      spatial_context_ztolerance = 0.01;

static const wchar_t*    spatial_context_name2 = L"SCTest2s";
static const wchar_t*    spatial_context_desc2 =  L"Another SC";
static const wchar_t*    spatial_context_desc2b = L"Another SC!";
static const wchar_t*    spatial_context_coordsysID2 = L"20251";
static const double      spatial_context_minx2 = -5000071.0;
static const double      spatial_context_maxx2 =  5000071.0;
static const double      spatial_context_miny2 = -3000071.0;
static const double      spatial_context_maxy2 =  7000071.0;  // NOTE: arcsde will convert the spatial context to a square area, no matter what you do
static const double      spatial_context_minx2b = -5001071.0;
static const double      spatial_context_maxx2b =  5001071.0;
static const double      spatial_context_miny2b = -3001071.0;
static const double      spatial_context_maxy2b =  7001071.0;
static const FdoSpatialContextExtentType spatial_context_extent_type2 = FdoSpatialContextExtentType_Static;
static const double      spatial_context_ztolerance2 = 0.01;

static const wchar_t* class_description = L"Dwelling address";
static const wchar_t* id_name = L"FeatId";
static const wchar_t* id_description = L"Unique identifier";
static const bool     id_nullable = false;
static const bool     id_autogenerated = true;
static const bool     id_readonly = true;
static const wchar_t* location_name = L"Location2";  // Oracle doesn't like the identifier "Location"
static const wchar_t* location_description = L"Geodetic location";
static const int      location_geometry_types = FdoGeometricType_Point;
static const bool     location_has_elevation = false;
static const bool     location_has_measure = false;



SpatialContextTests::SpatialContextTests(void)
{
}

SpatialContextTests::~SpatialContextTests(void)
{
}

void SpatialContextTests::setUp ()
{
    if (!CreateSchemaOnly() /* || !bSchemaCreated */ )
    {
        mConnection = ArcSDETests::GetConnection ();
        mConnection->SetConnectionString (ArcSDETestConfig::ConnStringAustralia());
        mConnection->Open ();
    }

#if 0
    // Get all ArcSDE coordinate systems, useful for testing:
    int projCSCount = 0;
    int projCsCode = 0;
    PE_TABLE projCSTable = NULL;
    projCSTable = pe_predefined_projcs (&projCSCount);
    for (int i=0; i<projCSCount; i++)
    {
        projCsCode = projCSTable[i].code;
    }
#endif
}

void SpatialContextTests::tearDown ()
{
    if (mConnection)
        mConnection->Close ();
}



void SpatialContextTests::test_create_delete()
{
    FdoPtr<FdoIGetSpatialContexts> cmdGetSpatialContexts;
    FdoPtr<FdoISpatialContextReader> scReader;
    bool bFound = false;

    if (CreateSchemaOnly()) return;

    try
    {
        //////////////////////////////////////////////////////////////////////
        // Clean up from previous tests:
        //////////////////////////////////////////////////////////////////////

        FdoPtr<FdoIDestroySpatialContext> cmdDestroySpatialContext = (FdoIDestroySpatialContext*)mConnection->CreateCommand(FdoCommandType_DestroySpatialContext);
        cmdDestroySpatialContext->SetName(spatial_context_name2);
        try
        {
            cmdDestroySpatialContext->Execute();
        }
        catch (FdoException *e)
        {
            e->Release();
        }


        //////////////////////////////////////////////////////////////////////
        // Create a spatial context:
        //////////////////////////////////////////////////////////////////////

        create_spatial_context2();

        //////////////////////////////////////////////////////////////////////
        // Get all spatial contexts, validating the one we created is there and is set right:
        //////////////////////////////////////////////////////////////////////

        ValidateSC2();

        // Clear cache by disconnecting/reconnecting
        mConnection->Close();
        mConnection->Open();

        ValidateSC2();


        //////////////////////////////////////////////////////////////////////
        // Destroy the spatial context we created:
        //////////////////////////////////////////////////////////////////////

        cmdDestroySpatialContext = (FdoIDestroySpatialContext*)mConnection->CreateCommand(FdoCommandType_DestroySpatialContext);
        cmdDestroySpatialContext->SetName(spatial_context_name2);
        cmdDestroySpatialContext->Execute();


        //////////////////////////////////////////////////////////////////////
        // Get all spatial contexts, validating the one we created is now gone:
        //////////////////////////////////////////////////////////////////////

        cmdGetSpatialContexts = (FdoIGetSpatialContexts*)mConnection->CreateCommand(FdoCommandType_GetSpatialContexts);
        cmdGetSpatialContexts->SetActiveOnly(false);
        scReader = cmdGetSpatialContexts->Execute();
        bFound = false;
        while (scReader->ReadNext())
        {
            if (0==wcscmp(scReader->GetName(), spatial_context_name2))
                bFound = true;
        }
        CPPUNIT_ASSERT_MESSAGE("Spatial context was not deleted properly", !bFound);

        // Clear cache by disconnecting/reconnecting
        mConnection->Close();
        mConnection->Open();

        cmdGetSpatialContexts = (FdoIGetSpatialContexts*)mConnection->CreateCommand(FdoCommandType_GetSpatialContexts);
        cmdGetSpatialContexts->SetActiveOnly(false);
        scReader = cmdGetSpatialContexts->Execute();
        bFound = false;
        while (scReader->ReadNext())
        {
            if (0==wcscmp(scReader->GetName(), spatial_context_name2))
                bFound = true;
        }
        CPPUNIT_ASSERT_MESSAGE("Spatial context was not deleted properly", !bFound);
    }
    catch (FdoException *e)
    {
        fail (e);
    }
}


// Test the active spatial context
void SpatialContextTests::test_active()
{
    if (CreateSchemaOnly()) return;

    try
    {
        //////////////////////////////////////////////////////////////////////
        // Clean up from previous tests:
        //////////////////////////////////////////////////////////////////////

        FdoPtr<FdoIDestroySpatialContext> cmdDestroySpatialContext = (FdoIDestroySpatialContext*)mConnection->CreateCommand(FdoCommandType_DestroySpatialContext);
        cmdDestroySpatialContext->SetName(spatial_context_name2);
        try
        {
            cmdDestroySpatialContext->Execute();
        }
        catch (FdoException *e)
        {
            e->Release();
        }



        //////////////////////////////////////////////////////////////////////
        // Create a spatial context:
        //////////////////////////////////////////////////////////////////////

        create_spatial_context2();


        //////////////////////////////////////////////////////////////////////
        // Activate the new spatial context:
        //////////////////////////////////////////////////////////////////////

        FdoPtr<FdoIActivateSpatialContext> cmdActivateSpatialContext = (FdoIActivateSpatialContext*)mConnection->CreateCommand(FdoCommandType_ActivateSpatialContext);
        cmdActivateSpatialContext->SetName(spatial_context_name2);
        cmdActivateSpatialContext->Execute();


        //////////////////////////////////////////////////////////////////////
        // Get all spatial contexts, validating the one we created is active:
        //////////////////////////////////////////////////////////////////////

        FdoPtr<FdoIGetSpatialContexts> cmdGetSpatialContexts = (FdoIGetSpatialContexts*)mConnection->CreateCommand(FdoCommandType_GetSpatialContexts);
        cmdGetSpatialContexts->SetActiveOnly(false);
        FdoPtr<FdoISpatialContextReader> scReader = cmdGetSpatialContexts->Execute();
        bool bFound = false;
        while (scReader->ReadNext())
        {
            // validate this spatial context, if its the one we created earlier:
            if (0==wcscmp(scReader->GetName(), spatial_context_name2))
            {
                bFound = true;
                CPPUNIT_ASSERT_MESSAGE("Spatial context should be active", scReader->IsActive());
            }
        }
        CPPUNIT_ASSERT_MESSAGE("Spatial Context not found", bFound);


        //////////////////////////////////////////////////////////////////////
        // Get only the active spatial context, validating its the one we created is active:
        //////////////////////////////////////////////////////////////////////

        cmdGetSpatialContexts->SetActiveOnly(true);
        scReader = cmdGetSpatialContexts->Execute();
        CPPUNIT_ASSERT_MESSAGE("Should have exactly one spatial context active", scReader->ReadNext());
        CPPUNIT_ASSERT_MESSAGE("Wrong spatial context returned", (0==wcscmp(scReader->GetName(), spatial_context_name2)));
        CPPUNIT_ASSERT_MESSAGE("Should have no more than one spatial context active", !scReader->ReadNext());


        //////////////////////////////////////////////////////////////////////
        // Update the spatial context we created:
        //////////////////////////////////////////////////////////////////////

        FdoPtr<FdoICreateSpatialContext> cmdCreateSpatialContext = (FdoICreateSpatialContext*)mConnection->CreateCommand(FdoCommandType_CreateSpatialContext);
        cmdCreateSpatialContext->SetUpdateExisting(true);
        cmdCreateSpatialContext->SetName(spatial_context_name2);
        cmdCreateSpatialContext->SetDescription(spatial_context_desc2b);
        set_extent(cmdCreateSpatialContext, spatial_context_minx2b, spatial_context_miny2b, spatial_context_maxx2b, spatial_context_maxy2b);

        // Perform the create operation:
        cmdCreateSpatialContext->Execute();


        //////////////////////////////////////////////////////////////////////
        // Get only the active spatial context, validating its the one we updated:
        //////////////////////////////////////////////////////////////////////

        cmdGetSpatialContexts->SetActiveOnly(true);
        scReader = cmdGetSpatialContexts->Execute();
        CPPUNIT_ASSERT_MESSAGE("Should have exactly one spatial context active", scReader->ReadNext());
        CPPUNIT_ASSERT_MESSAGE("Wrong spatial context returned", (0==wcscmp(scReader->GetName(), spatial_context_name2)));
        // NOTE: due to defect 639233, we truncate descriptions longer than 23 characters to avoid an ArcSDE bug, whcih means descriptions may not be roundtripped properly.
        //CPPUNIT_ASSERT_MESSAGE("Wrong description returned", 0==wcscmp(scReader->GetDescription(), spatial_context_desc2b));
        double minx,miny,maxx,maxy;
        get_extent(scReader, minx, miny, maxx, maxy);
        if (!ArcSDETests::fuzzyEqual(minx, spatial_context_minx2b))
           CPPUNIT_ASSERT_MESSAGE("Wrong extent returned", ArcSDETests::fuzzyEqual(minx, spatial_context_minx2b));
        if (!ArcSDETests::fuzzyEqual(miny, spatial_context_miny2b))
           CPPUNIT_ASSERT_MESSAGE("Wrong extent returned", ArcSDETests::fuzzyEqual(miny, spatial_context_miny2b));
        if (!ArcSDETests::fuzzyEqual(maxx, spatial_context_maxx2b))
           CPPUNIT_ASSERT_MESSAGE("Wrong extent returned", ArcSDETests::fuzzyEqual(maxx, spatial_context_maxx2b));
        if (!ArcSDETests::fuzzyEqual(maxy, spatial_context_maxy2b))
           CPPUNIT_ASSERT_MESSAGE("Wrong extent returned", ArcSDETests::fuzzyEqual(maxy, spatial_context_maxy2b));


        //////////////////////////////////////////////////////////////////////
        // Destroy the spatial context we created:
        //////////////////////////////////////////////////////////////////////

        cmdDestroySpatialContext = (FdoIDestroySpatialContext*)mConnection->CreateCommand(FdoCommandType_DestroySpatialContext);
        cmdDestroySpatialContext->SetName(spatial_context_name2);
        cmdDestroySpatialContext->Execute();


        //////////////////////////////////////////////////////////////////////
        // Get only the active spatial context, validating its NOT the one we created:
        //////////////////////////////////////////////////////////////////////

        cmdGetSpatialContexts->SetActiveOnly(true);
        scReader = cmdGetSpatialContexts->Execute();
        if (scReader->ReadNext())
        {
            CPPUNIT_ASSERT_MESSAGE("Spatial context no longer exists", (0!=wcscmp(scReader->GetName(), spatial_context_name2)));
            CPPUNIT_ASSERT_MESSAGE("Should have no more than one spatial context active", !scReader->ReadNext());
        }
    }
    catch (FdoException *e)
    {
        fail (e);
    }
}


// Test applying the spatial context to tables
void SpatialContextTests::test_tables()
{
#ifdef _DEBUG  // Since this test relies heavily on ApplySchema to achieve proper testing,
               // we can't easily factor the ApplySchema calls out using CreateSchemaOnly();
               // So, we only run this test when ApplySchema is available (e.g. in _DEBUG mode only).

    if (CreateSchemaOnly()) return;

    try
    {
        //////////////////////////////////////////////////////////////////////
        // Clean up from previous tests:
        //////////////////////////////////////////////////////////////////////

        CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaScTestCls1s(), ArcSDETestConfig::ClassNameScTestCls1s(), true);
        CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaScTestCls1s(), ArcSDETestConfig::ClassNameScTestCls1s());

        FdoPtr<FdoIDestroySpatialContext> cmdDestroySpatialContext = (FdoIDestroySpatialContext*)mConnection->CreateCommand(FdoCommandType_DestroySpatialContext);
        cmdDestroySpatialContext->SetName(spatial_context_name);
        try
        {
            cmdDestroySpatialContext->Execute();
        }
        catch (FdoException *e)
        {
            e->Release();
        }
        cmdDestroySpatialContext->SetName(spatial_context_name2);
        try
        {
            cmdDestroySpatialContext->Execute();
        }
        catch (FdoException *e)
        {
            e->Release();
        }



        //////////////////////////////////////////////////////////////////////
        // Create a spatial context:
        //////////////////////////////////////////////////////////////////////

        create_spatial_context();


        //////////////////////////////////////////////////////////////////////
        // Activate the new spatial context:
        //////////////////////////////////////////////////////////////////////

        FdoPtr<FdoIActivateSpatialContext> cmdActivateSpatialContext = (FdoIActivateSpatialContext*)mConnection->CreateCommand(FdoCommandType_ActivateSpatialContext);
        cmdActivateSpatialContext->SetName(spatial_context_name);
        cmdActivateSpatialContext->Execute();


        //////////////////////////////////////////////////////////////////////
        // Create a class (it should pick up the active spatial context for its geometry column):
        //////////////////////////////////////////////////////////////////////

        FdoPtr<FdoFeatureSchema> schema;  // we need to hang on to the schema, so the class doesn't lose its parent pointer
        FdoPtr<FdoClassDefinition> pClassDef = create_class(schema.p);


        //////////////////////////////////////////////////////////////////////
        // Insert geometric data:
        //////////////////////////////////////////////////////////////////////

        // Insert within extents (shouldn't get error):
        FdoPtr<FdoIInsert> insert = (FdoIInsert*)mConnection->CreateCommand (FdoCommandType_Insert);
        insert->SetFeatureClassName (ArcSDETestConfig::QClassNameScTestCls1s());
        FdoPtr<FdoPropertyValueCollection> values = insert->GetPropertyValues ();
        FdoPtr<FdoValueExpression> expression = (FdoValueExpression*)FdoExpression::Parse (L"GEOMFROMTEXT ('POINT XY ( 999000 -999000 )')");
        FdoPtr<FdoPropertyValue> value = FdoPropertyValue::Create (location_name, expression);
        values->Add (value);
        FdoPtr<FdoIFeatureReader> reader = insert->Execute ();
        reader->Close ();
        reader = NULL;

        // Insert outside extents (SHOULD get error):
        insert = (FdoIInsert*)mConnection->CreateCommand (FdoCommandType_Insert);
        insert->SetFeatureClassName (ArcSDETestConfig::QClassNameScTestCls1s());
        values = insert->GetPropertyValues ();
        expression = (FdoValueExpression*)FdoExpression::Parse (L"GEOMFROMTEXT ('POINT XY ( 2000010 -999000 )')");
        value = FdoPropertyValue::Create (location_name, expression);
        values->Add (value);
        try
        {
            reader = insert->Execute ();
            reader->Close();
            CPPUNIT_ASSERT_MESSAGE("Should have failed to insert outside spatial context extents, but didn't", false);
        }
        catch (FdoException *e)
        {
            // We expect to get an exception here
            e->Release();
        }


        //////////////////////////////////////////////////////////////////////
        // Create a second spatial context:
        //////////////////////////////////////////////////////////////////////

        create_spatial_context2();


        //////////////////////////////////////////////////////////////////////
        // Update spatial context of geometry column:
        //////////////////////////////////////////////////////////////////////

/* //NOTE: we don't test changing the spatial context, since that implies supporting ApplySchema and
//      having to do a lot of the coding to support modifying a class.

        FdoPtr<FdoPropertyDefinitionCollection> pProps = pClassDef->GetProperties();
        FdoPtr<FdoPropertyDefinition> pProp = pProps->GetItem(location_name);
        FdoGeometricPropertyDefinition* pGeomProp = dynamic_cast<FdoGeometricPropertyDefinition*>(pProp.p);
        pGeomProp->SetSpatialContextAssociation(spatial_context_name2);
        FdoPtr<FdoIApplySchema> applySchema = (FdoIApplySchema*)mConnection->CreateCommand(FdoCommandType_ApplySchema);
        applySchema->SetFeatureSchema(schema);
        try
        {
            applySchema->Execute();
            CPPUNIT_ASSERT_MESSAGE("Should fail to update spatial context of a class containing data", false);
        }
        catch (FdoException *e)
        {
            // we expect an exception here.
            e->Release();
        }

        // Delete all rows of class:
        CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaSctestcls(), ArcSDETestConfig::ClassNameScTestCls1s(), true);

        // Try again to update spatial context (should succeed now):
        applySchema->Execute();


        //////////////////////////////////////////////////////////////////////
        // Insert geometric data:
        //////////////////////////////////////////////////////////////////////

        // Insert within extents (shouldn't get error):
        insert = (FdoIInsert*)mConnection->CreateCommand (FdoCommandType_Insert);
        insert->SetFeatureClassName (ArcSDETestConfig::QClassNameScTestCls1s());
        values = insert->GetPropertyValues ();
        value = values->GetItem (location_name);
        expression = (FdoValueExpression*)FdoExpression::Parse (L"GEOMFROMTEXT ('POINT XY ( 2000000 -2000000 )')");
        value->SetValue (expression);
        reader = insert->Execute ();
        reader->Close ();

        // Insert outside extents (SHOULD get error):
        insert = (FdoIInsert*)mConnection->CreateCommand (FdoCommandType_Insert);
        insert->SetFeatureClassName (ArcSDETestConfig::QClassNameScTestCls1s());
        values = insert->GetPropertyValues ();
        value = values->GetItem (location_name);
        expression = (FdoValueExpression*)FdoExpression::Parse (L"GEOMFROMTEXT ('POINT XY ( 10000000 0 )')");
        value->SetValue (expression);
        try
        {
            reader = insert->Execute ();
            reader->Close();
            CPPUNIT_ASSERT_MESSAGE("Should have failed to insert outside spatial context extents, but didn't", false);
        }
        catch (FdoException *e)
        {
            // We expect to get an exception here
            e->Release();
        }


        //////////////////////////////////////////////////////////////////////
        // Attempt to destroy the spatial context (should fail):
        //////////////////////////////////////////////////////////////////////

        cmdDestroySpatialContext = (FdoIDestroySpatialContext*)mConnection->CreateCommand(FdoCommandType_DestroySpatialContext);
        cmdDestroySpatialContext->SetName(spatial_context_name2);
        try
        {
            cmdDestroySpatialContext->Execute();
            CPPUNIT_ASSERT_MESSAGE("Should have failed to delete spatial context, since it's in use", false);
        }
        catch (FdoException *e)
        {
            // We expect to get an exception here
            e->Release();
        }
*/


        //////////////////////////////////////////////////////////////////////
        // Delete the class (should succeed):
        //////////////////////////////////////////////////////////////////////

        delete_class();


        //////////////////////////////////////////////////////////////////////
        // Destroy the spatial context (should succeed):
        //////////////////////////////////////////////////////////////////////

        cmdDestroySpatialContext = (FdoIDestroySpatialContext*)mConnection->CreateCommand(FdoCommandType_DestroySpatialContext);
        cmdDestroySpatialContext->SetName(spatial_context_name2);
        cmdDestroySpatialContext->Execute();


        //////////////////////////////////////////////////////////////////////
        // Destroy the other spatial context:
        //////////////////////////////////////////////////////////////////////

        cmdDestroySpatialContext = (FdoIDestroySpatialContext*)mConnection->CreateCommand(FdoCommandType_DestroySpatialContext);
        cmdDestroySpatialContext->SetName(spatial_context_name);
        cmdDestroySpatialContext->Execute();
    }
    catch (FdoException *e)
    {
        fail (e);
    }

#endif // _DEBUG  // Since this test uses ApplySchema
}





// Create a spatial context for testing purposes:
void SpatialContextTests::create_spatial_context (void)
{
    // Set up a spatial context with test data:
    FdoPtr<FdoICreateSpatialContext> cmdCreateSpatialContext = (FdoICreateSpatialContext*)mConnection->CreateCommand(FdoCommandType_CreateSpatialContext);
    cmdCreateSpatialContext->SetCoordinateSystem(spatial_context_coordsysID1);
    cmdCreateSpatialContext->SetDescription(spatial_context_desc);
    cmdCreateSpatialContext->SetName(spatial_context_name);
    cmdCreateSpatialContext->SetExtentType(spatial_context_extent_type);
    cmdCreateSpatialContext->SetZTolerance(spatial_context_ztolerance);
    set_extent(cmdCreateSpatialContext, spatial_context_minx, spatial_context_miny, spatial_context_maxx, spatial_context_maxy);

    // Perform the create operation:
    try
    {
        cmdCreateSpatialContext->Execute();
    }
    catch (FdoException *ex)
    {
        ex->Release();
    }
}



// Create a second spatial context for testing purposes:
void SpatialContextTests::create_spatial_context2 (void)
{
    // Set up a spatial context with test data:
    FdoPtr<FdoICreateSpatialContext> cmdCreateSpatialContext = (FdoICreateSpatialContext*)mConnection->CreateCommand(FdoCommandType_CreateSpatialContext);
    cmdCreateSpatialContext->SetCoordinateSystem(spatial_context_coordsysID2);
    cmdCreateSpatialContext->SetDescription(spatial_context_desc2);
    cmdCreateSpatialContext->SetName(spatial_context_name2);
    cmdCreateSpatialContext->SetExtentType(spatial_context_extent_type2);
    cmdCreateSpatialContext->SetZTolerance(spatial_context_ztolerance2);
    set_extent(cmdCreateSpatialContext, spatial_context_minx2, spatial_context_miny2, spatial_context_maxx2, spatial_context_maxy2);

    // Perform the create operation:
    try
    {
        cmdCreateSpatialContext->Execute();
    }
    catch (FdoException *ex)
    {
        ex->Release();
    }
}


FdoClassDefinition* SpatialContextTests::create_class(FdoFeatureSchema *&schema)
{
    FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
    schema = ArcSDETests::GetDefaultSchema(mConnection);  //FdoFeatureSchema::Create (schema_name, schema_description);
    FdoPtr<FdoClassCollection> classes = schema->GetClasses ();

    // build an id property
    FdoPtr<FdoDataPropertyDefinition> id = FdoDataPropertyDefinition::Create (id_name, id_description);
    id->SetDataType (FdoDataType_Int32);
    id->SetNullable (id_nullable);
    id->SetIsAutoGenerated (id_autogenerated);
    id->SetReadOnly (id_readonly);

    // build a location geometry property
    FdoPtr<FdoGeometricPropertyDefinition> location = FdoGeometricPropertyDefinition::Create (location_name, location_description);
    location->SetGeometryTypes (location_geometry_types);
    location->SetHasElevation (location_has_elevation);
    location->SetHasMeasure (location_has_measure);

    // assemble the feature class
    FdoPtr<FdoFeatureClass> feature = FdoFeatureClass::Create (ArcSDETestConfig::ClassNameScTestCls1s(), class_description);
    FdoPtr<FdoPropertyDefinitionCollection> properties = feature->GetProperties ();
    properties->Add (id);
    properties->Add (location);
    feature->SetGeometryProperty (location);
    FdoPtr<FdoDataPropertyDefinitionCollection> identities = feature->GetIdentityProperties ();
    identities->Add (id);

    // submit the new schema
    classes->Add (feature);
    apply->SetFeatureSchema (schema);
    apply->Execute ();

    return FDO_SAFE_ADDREF(feature.p);
}


void SpatialContextTests::delete_class ()
{
    FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
    FdoPtr<FdoFeatureSchema> schema = ArcSDETests::GetDefaultSchema(mConnection);  //FdoFeatureSchema::Create (schema_name, schema_description);
    FdoPtr<FdoClassCollection> classes = schema->GetClasses ();
    FdoPtr<FdoClassDefinition> feature = classes->GetItem (ArcSDETestConfig::ClassNameScTestCls1s());
    feature->Delete ();
    apply->SetFeatureSchema (schema);
    apply->Execute ();
}


void SpatialContextTests::ValidateSC2()
{
    FdoPtr<FdoIGetSpatialContexts> cmdGetSpatialContexts = (FdoIGetSpatialContexts*)mConnection->CreateCommand(FdoCommandType_GetSpatialContexts);
    cmdGetSpatialContexts->SetActiveOnly(false);
    FdoPtr<FdoISpatialContextReader> scReader = cmdGetSpatialContexts->Execute();
    bool bFound = false;
    while (scReader->ReadNext())
    {
        // Get spatial context's general info:
        FdoString *name  = scReader->GetName();
        FdoString *desc  = scReader->GetDescription();
        FdoString *coordSysName = scReader->GetCoordinateSystem();
        FdoString *coordSysWkt  = scReader->GetCoordinateSystemWkt();
        FdoSpatialContextExtentType contextType = scReader->GetExtentType();
        double zTolerance = scReader->GetZTolerance();
        double zyTolerance = scReader->GetXYTolerance();
        double minx,miny,maxx,maxy;
        get_extent(scReader, minx, miny, maxx, maxy);

        /* // Uncomment this for additional debug info:
        printf("\nCOORDSYS:");
        printf("\n  name=%ls", name);
        printf("\n  desc=%ls", desc);
        printf("\n  coordsys=%ls", coordSysName);
        printf("\n  corodsyswkt=%ls", coordSysWkt);
        printf("\n  extenttype=%ls", contextType==FdoSpatialContextExtentType_Static ? L"Static" : L"Dynamic");
        printf("\n  xTolerance=%f", zTolerance);
        printf("\n  zyTolerance=%f", zyTolerance);
        printf("\n  minx=%f", minx);
        printf("\n  miny=%f", miny);
        printf("\n  maxx=%f", maxx);
        printf("\n  maxy=%f", maxy);
        printf("\n");
        */

        // validate this spatial context, if its the one we created earlier:
        if (0==wcscmp(name, spatial_context_name2))
        {
            bFound = true;
            CPPUNIT_ASSERT_MESSAGE("Spatial Context description mismatch", 0==wcscmp(desc, spatial_context_desc2));

            CPPUNIT_ASSERT_MESSAGE("Spatial context coordsys id/name mismatch", 0==wcscmp(coordSysName, spatial_context_coordsysID2));

            // NOTE: the wkt we retrieve will almost always be formatted slightly differently than what we provided
            //        (e.g. spacing, removal of ".0" on floating point values, etc), so we can't do a straight test comparison.
            //CPPUNIT_ASSERT_MESSAGE("Spatial Context coordsys wkt mismatch", 0==wcscmp(coordSysWkt, spatial_context_coordsysWKT));

            CPPUNIT_ASSERT_MESSAGE("Spatial Context type mismatch", contextType==spatial_context_extent_type2);
            CPPUNIT_ASSERT_MESSAGE("Spatial Context tolerance mismatch", zTolerance==spatial_context_ztolerance2);
            if (!ArcSDETests::fuzzyEqual(minx, spatial_context_minx2))
               CPPUNIT_ASSERT_MESSAGE("Extent minx wrong", ArcSDETests::fuzzyEqual(minx, spatial_context_minx2));
            if (!ArcSDETests::fuzzyEqual(maxx, spatial_context_maxx2))
               CPPUNIT_ASSERT_MESSAGE("Extent maxx wrong", ArcSDETests::fuzzyEqual(maxx, spatial_context_maxx2));
            if (!ArcSDETests::fuzzyEqual(miny, spatial_context_miny2))
               CPPUNIT_ASSERT_MESSAGE("Extent miny wrong", ArcSDETests::fuzzyEqual(miny, spatial_context_miny2));
            if (!ArcSDETests::fuzzyEqual(maxy, spatial_context_maxy2))
               CPPUNIT_ASSERT_MESSAGE("Extent maxy wrong", ArcSDETests::fuzzyEqual(maxy, spatial_context_maxy2));
        }
    }

    CPPUNIT_ASSERT_MESSAGE("Spatial Context not found", bFound);
}
