/*
* Copyright (C) 2007  Haris Kurtagic
* 
* This library is free software; you can redistribute it and/or
* modify it under the terms of version 2.1 of the GNU Lesser
* General Public License as published by the Free Software Foundation.
* 
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "StdAfx.h"
#include "c_KgInfApplySchema.h"
#include "c_KgInfConnection.h"

c_KgInfApplySchema::c_KgInfApplySchema(c_KgInfConnection *Conn)
  : c_KgInfFdoCommand(Conn)
{
  m_IngoreStates = true;
}

c_KgInfApplySchema::~c_KgInfApplySchema(void)
{
}


/// \brief
/// Gets the name of the schema to create.
/// 
/// \return
/// Returns the name of the schema to create
/// 
FdoFeatureSchema* c_KgInfApplySchema::GetFeatureSchema()
{
  return FDO_SAFE_ADDREF(m_FeatureSchema.p);
}

/// \brief
/// Sets the name of the schema to create.
/// 
/// \param value 
/// Input the name of the schema to create
/// 
/// \return
/// Returns nothing
/// 
void c_KgInfApplySchema::SetFeatureSchema(FdoFeatureSchema* Schema)
{
  m_FeatureSchema = Schema;
  FDO_SAFE_ADDREF(m_FeatureSchema.p);
}

/// \brief
///  Gets the FdoPhysicalSchemaMapping used to specify how the schema definition
/// gets mapped into physical storage.
/// 
/// \return
/// Returns FdoPhysicalSchemaMapping
/// 
FdoPhysicalSchemaMapping* c_KgInfApplySchema::GetPhysicalMapping()
{
  return FDO_SAFE_ADDREF(m_PhysMapping.p);
}

/// \brief
/// Sets the FdoPhysicalSchemaMapping used to specify how the schema definition
/// gets mapped into physical storage.
/// 
/// \param value 
/// Input the FdoPhysicalSchemaMapping
/// 
/// \return
/// Returns nothing
/// 
void c_KgInfApplySchema::SetPhysicalMapping(FdoPhysicalSchemaMapping* PhysMap)
{
  m_PhysMapping = PhysMap;
  FDO_SAFE_ADDREF(m_PhysMapping.p);
}

/// \brief
///  Indicates whether Execute() will ignore element states 
/// when applying the feature schema.
/// 
/// \return
/// Returns true if elements states will be ignored, false otherwise. 
/// 
///  
FdoBoolean c_KgInfApplySchema::GetIgnoreStates()
{
  return m_IngoreStates;
}

/// \brief
/// Changes the handling of element states by Execute().
/// 
/// \param ignoreStates 
/// Input True: Execute() will ignore the 
/// element states on the Feature Schema elements. Instead, it 
/// will additively merge the Feature Schema into the current DataStore.
/// False: Execute() will use the element states to determine whether
/// each element is inserted, modified or deleted.
/// 
/// \return
/// Returns nothing
/// 
void c_KgInfApplySchema::SetIgnoreStates( FdoBoolean IgnoreStates )
{
  m_IngoreStates = IgnoreStates;
}



/// \brief
/// Executes the ApplySchema command that creates metadata and physical
/// storage for the schema. An exception is thrown if the schema already
/// exists or if a schema already exists and the feature provider only
/// supports a single schema. If schema changes include deletion of classes
/// or the schema itself, then those classes (or the schema, if it is marked
/// as deleted) must not contain any instance data. If they do, an exception
/// will be thrown.
/// Implementors must call FdoFeatureSchema::AcceptChanges() when Execute() succeeds.
/// It must be called after the schema has been successfully applied, in
/// order to synchronize the states of the schema elements.
/// 
/// \return
/// Returns nothing
/// 
void c_KgInfApplySchema::Execute()
{
  if( !m_IngoreStates ) 
    throw FdoException::Create(L"c_KgInfApplySchema::Execute Supports only IgnoreStates='true'");
    
  if( !m_FeatureSchema.p ) return;
//    
try
{  
      
  FdoPtr<FdoClassCollection> classes = m_FeatureSchema->GetClasses();
  int classcount = classes->GetCount();
  for(int classind=0;classind<classcount;classind++)
  {
    FdoStringP sql_cols;
    FdoStringP sql_create;
    FdoStringP geometry_col;
    FdoPtr<FdoClassDefinition> classdef = classes->GetItem(classind);
 
  // for the class properties need to convert to oracle types
  // als check if there is an AutoGenerated Int field
    FdoPropertyDefinition* autoprop=NULL;
    sql_cols = "";
    FdoPtr<FdoPropertyDefinitionCollection> props = classdef->GetProperties();
    if( props.p && props->GetCount() )
    {
      
      FdoStringP sep;
      FdoStringP proporatype;
      int propcount = props->GetCount();
      for(int propind=0;propind<propcount;propind++)
      {
        FdoPtr<FdoPropertyDefinition> propdef = props->GetItem(propind);
        
        // add all columns exect geometry; geometry is added separately
        if( propdef->GetPropertyType() == FdoPropertyType_DataProperty )
        {
          FdoDataPropertyDefinition * propdata = (FdoDataPropertyDefinition *)propdef.p;
          proporatype = "";
          if( !c_FdoInf_API::FdoPropertyToInfDataType(propdef.p,proporatype) )
          {
            
            throw FdoCommandException::Create( L"c_KgInfApplySchema::Execute: Unknown Property Definition ");  
          }
            
          if( propdata->GetIsAutoGenerated() && (propdata->GetDataType() == FdoDataType_Int32) || (propdata->GetDataType() == FdoDataType_Int64) )
          {
            sql_cols = sql_cols+  sep + " " + propdef->GetName() + " SERIAL PRIMARY KEY ";
            autoprop = propdata;
          }
          else
          {
            sql_cols = sql_cols + sep + " " + propdef->GetName() + " " + proporatype;
          }
          sep = ",";
        }
        else if( propdef->GetPropertyType() == FdoPropertyType_GeometricProperty )
        {
          FdoDataPropertyDefinition * propdata = (FdoDataPropertyDefinition *)propdef.p;
          sql_cols = sql_cols + sep + " " + propdef->GetName() + " ST_Geometry";
          geometry_col = propdef->GetName();
          sep = ",";
        }
      }
      // create sql for creating the table  
      FdoStringP tablename = classdef->GetName();
      //simon
      FdoStringP tablenameTmp;
      //change table name to coorect table name withouth ~
      if(tablename.Contains(L"~"))
      {
        tablenameTmp = tablename.Right(L"~");
        if(tablenameTmp.Contains(L"~"))
        {
          tablenameTmp = tablenameTmp.Left(L"~");
        }
        else if(tablenameTmp.Contains(geometry_col))
        {
          tablenameTmp = tablename.Left(L"~");
        }
        tablename = tablenameTmp;
      }
      
      //if( autoprop )
      {
      // identity is alreday added
        sql_create = sql_create + "CREATE TABLE " + tablename + " ( " + sql_cols + " ) ";
      }
      
      
      string sqlstr = sql_create;
      const char*dbg = sqlstr.c_str();
      
      m_Connection->OTL_ExecuteSql(dbg);
      propcount = props->GetCount();
      for(int propind=0;propind<propcount;propind++)
      {
        FdoPtr<FdoPropertyDefinition> propdef = props->GetItem(propind);
      
        // add all columns exect geometry; geometry is added separately
        if( propdef->GetPropertyType() == FdoPropertyType_GeometricProperty )
        {
          FdoGeometricPropertyDefinition* geomprop = (FdoGeometricPropertyDefinition*)propdef.p;
          if( geomprop )
          {
            InsertSdoGeomMetadata(tablename,geomprop);
          }
        }
      }      
    } 
  }
}
catch(c_KgOtlException& ea)
{
  throw FdoCommandException::Create( ea.what() );    
}  

    
}//endof c_KgInfApplySchema::Execute


void c_KgInfApplySchema::InsertSdoGeomMetadata(FdoStringP TableName,FdoGeometricPropertyDefinition* GeomProp)
{
  try
  {
    FdoString* scname = GeomProp->GetSpatialContextAssociation();
          
    FdoPtr<c_KgInfSpatialContextCollection> sccol = m_Connection->GetSpatialContexts();
    FdoPtr<c_KgInfSpatialContext> spcontext = sccol->FindItem(scname);
    
    bool isgeogcs=false;
    
    c_KgInfSridDesc orasrid;
    m_Connection->GetOracleSridDesc(GeomProp,orasrid);
   
    char buff [2048];
    FdoStringP gname = GeomProp->GetName();
          
    wstring owner;
    c_FdoInf_API::GetTableOwner(m_Connection,TableName,owner);
    wstring whitespaces=L" \t\r\n";
    wstring str(owner);
    FdoStringP owner2 = str.erase(str.find_last_not_of(whitespaces) + 1).c_str();
    int geomType = GeomProp->GetGeometryTypes();
    int geomInfType = 0;
    c_FdoInf_API::FdoGeomTypesToInfGeomType(GeomProp,geomInfType);
                       
    sprintf(buff,"INSERT INTO geometry_columns (f_table_catalog, f_table_schema, f_table_name, f_geometry_column, geometry_type, srid) VALUES('informix','%s','%s','%s',%i,%i)"
    ,(const char*)owner2,(const char*)TableName,(const char*)gname,geomInfType,orasrid.m_OraSrid);
    m_Connection->OTL_ExecuteSql(buff);
  }
  catch(c_KgOtlException& ea)
  {
    throw FdoCommandException::Create( ea.what() );    
  }  
}//end of c_KgInfApplySchema::InsertSdoGeomMetadata

