/*
* Copyright (C) 2004-2006  Autodesk, Inc.
* 
* This library is free software; you can redistribute it and/or
* modify it under the terms of version 2.1 of the GNU Lesser
* General Public License as published by the Free Software Foundation.
* 
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*/

#pragma once

#include "FDO\Commands\Feature\mgIBaseSelectImp.h"
#include "FDO\Commands\Feature\mgISelectAggregates.h"

class FdoISelectAggregates;

BEGIN_NAMESPACE_OSGEO_FDO_COMMANDS
ref class IdentifierCollection;
END_NAMESPACE_OSGEO_FDO_COMMANDS

BEGIN_NAMESPACE_OSGEO_FDO_FILTER
ref class Filter;
END_NAMESPACE_OSGEO_FDO_FILTER

BEGIN_NAMESPACE_OSGEO_FDO_COMMANDS_FEATURE
interface class IDataReader;

/// \ingroup (OSGeoFDOCommandsFeature)
/// \brief
/// The ISelectAggregateImp class is a cocrete implementation of interface ISelectAggregates.
/// The ISelectAggregate interface defines a Select command that handle selection of properties that include 
/// the use of aggregate functions. This command differs from the ISelect command in a number of ways. 
/// It handles selection of properties that are aggregate functions and supports selecting distinct property values. 
/// It also includes grouping criteria. The execute method returns an IDataReader which is not tied to a specific class. 
/// Unlike Select, ISelectAggregate does not include any locking functions.
private ref class ISelectAggregatesImp : public NAMESPACE_OSGEO_FDO_COMMANDS_FEATURE::IBaseSelectImp, 
                                          public NAMESPACE_OSGEO_FDO_COMMANDS_FEATURE::ISelectAggregates
{
public:
    /// \brief
    /// Executes the select command and returns a reference to an IDataReader.
    /// 
    /// \return
    /// Returns the data reader.
    /// 
	virtual NAMESPACE_OSGEO_FDO_COMMANDS_FEATURE::IDataReader^ Execute();

    /// \brief
    /// Set the distinct option of the selection. 
    /// Non-simple properties such as object properties, geometry properties, raster properties, association properties, etc. will not be supported with Distinct.
    /// 
    /// \param value 
    /// true or false; when set to true, only distinct values are returned. Otherwise all values are returned
    /// 
    /// \return
    /// Returns nothing
    /// 
    /// Note: Grouping criteria is not supported with Distinct. 
    /// \brief
    /// Get the distinct option.
    /// 
    /// \return
    /// Returns true if distinct is set, false otherwise.
    /// 
    virtual property System::Boolean Distinct
    {
        System::Boolean get();
        System::Void set(System::Boolean value);
    }

    /// \brief
    /// Gets the IdentifierCollection that holds the list of group by property names. If empty no grouping is used. This list is initially
    /// empty and the caller need to add the property that the command should use as a group by criteria. No LOB or Geometry type properties
    /// can be used for ordering.
    /// 
    /// \return
    /// Returns the list of group by property names.
    /// 
    property NAMESPACE_OSGEO_FDO_COMMANDS::IdentifierCollection^ Grouping
    {
        virtual NAMESPACE_OSGEO_FDO_COMMANDS::IdentifierCollection^ get();
    }

    /// \brief
    /// Set the grouping by filter. Use the grouping filter to restrict the groups of returned properties to those groups for 
    /// which the specified filter is TRUE. For example "order by city" and  "min(lanes) = 2". The Filter have to evalute to a 
    /// binary value(true or false).
    /// 
    /// \param filter 
    /// The grouping filter.
    /// 
    /// \return
    /// Returns nothing
    /// 
    /// \brief
    /// Gets the grouping by filter.
    /// 
    /// \return
    /// Returns the grouping filter.
    /// 
    virtual property NAMESPACE_OSGEO_FDO_FILTER::Filter^ GroupingFilter
    {
        NAMESPACE_OSGEO_FDO_FILTER::Filter^ get();
        System::Void set(NAMESPACE_OSGEO_FDO_FILTER::Filter^ value);
    }

    /// \brief
    /// Add join criteria definitions to define join selects.
    /// This function throws an eception in case is not supported by the data provider
    /// 
    /// \return
    /// Returns a join criteria collection or throws an exception.
    /// 
    property NAMESPACE_OSGEO_FDO_EXPRESSION::JoinCriteriaCollection^ JoinCriteria
    {
        virtual NAMESPACE_OSGEO_FDO_EXPRESSION::JoinCriteriaCollection^ get();
    }

    /// \brief
    /// Gets the alias of the class in join definition.
    /// 
    /// \return
    /// Returns the alias or null value.
    /// 
    /// \brief
    /// Sets the alias of the class in join definition.
    /// 
    /// \param value 
    /// Input the alias value.
    /// 
    /// \return
    /// Returns nothing
    /// 
    virtual property System::String^ Alias
    {
        System::String^ get();
        System::Void set(System::String^ value);
    }

internal:
	ISelectAggregatesImp(System::IntPtr unmanaged, System::Boolean autoDelete) : NAMESPACE_OSGEO_FDO_COMMANDS_FEATURE::IBaseSelectImp(unmanaged, autoDelete)
	{

	}

	inline FdoISelectAggregates* GetImpObj();
public:
    virtual IntPtr GetDisposableObject() override;
};

END_NAMESPACE_OSGEO_FDO_COMMANDS_FEATURE


