#!/usr/bin/env python
#
# Setup script for Berkeley DB XML package

import os, sys
from distutils.core import setup, Extension

#
# Posix:
#
# dbxml_home, db_home, xqilla_home, and xerces_home are
# set based on the arguments  passed to the configure script.
# This can be overridden with the --with-dbxml, --with-berkeleydb, --with-xqilla, 
# and --with-xerces arguments.
#
# dbxml_home is generally "../.." when building from source but in
# order to build against a specific installed version it can be specified
# on the command line (e.g. /usr/local/BerkeleyDBXML.2.4).
#
# Windows: may require further editing to reflect site specifics.
#
if os.name == "posix":
  dbxml_home = "@prefix@"
  db_home = "@DB_DIR@"
  xerces_home = "@XERCES_DIR@"
  xqilla_home = "@XQILLA_DIR@"
  LFLAGS = os.environ.get('LFLAGS', [])
else:
  dbxml_home = "../.."
  xerces_home = "../../../xerces-c-src"
  xqilla_home = "../../../xqilla"
  db_home = "../../../db-4.8.26"

bsddb_home = ""

for arg in sys.argv:
  if arg.startswith('--with-dbxml='):
    dbxml_home = arg.split('=')[1]
  if arg.startswith('--with-berkeleydb='):
    db_home = arg.split('=')[1]
  elif arg.startswith('--with-xerces='):
    xerces_home = arg.split('=')[1]
  elif arg.startswith('--with-xqilla='):
    xqillahome = arg.split('=')[1]
  elif arg.startswith('--lflags='):
    LFLAGS = arg.split('=')[1].split()
    sys.argv.remove(arg)
  elif arg.startswith('--with-bsddb='):
    bsddb_home = arg.split('=')[1]

lflags_arg = []

sys.argv = filter(lambda x: not x.startswith("--with-"), sys.argv)

debug = "--debug" in sys.argv or "-g" in sys.argv

# setup complains when passed debug flags for install
if "install" in sys.argv and not "build" in sys.argv:
  sys.argv = filter(lambda x: (x != "--debug") and (x != "-g"), sys.argv)

# if --with-bsddb was specified tell source to include bsddb.h
# from there
if bsddb_home:
  INCLUDES = [os.path.abspath(bsddb_home)]
  DEFINES = [('HAVE_BSDDB',1)]
else:
  INCLUDES = []
  DEFINES = []

if os.name == "posix":
  INCLUDES.append(os.path.join(dbxml_home,"include"))
  INCLUDES.append(os.path.join(db_home, "include"))

  LIBDIRS =     [os.path.join(dbxml_home,"build_unix/.libs"),
                 os.path.join(dbxml_home, "lib"),
                 os.path.join(db_home, "lib"),
                 os.path.join(xqilla_home, "lib"),
                 os.path.join(xerces_home, "lib")]

  LIBS =        ["dbxml",
                 "db-4",
                 "xqilla", "xerces-c"]

  DATAFILES =   []

  if LFLAGS:
        lflags_arg = LFLAGS + LIBS

  # Now run with whatever settings we've got
  setup(name = "dbxml",
              version = "2.5.16",
              description = "Berkeley DB XML Python API",
              long_description = """\
              This module provides a complete wrapping of the C++ API
              to the Berkeley DB XML (BDB XML) XML database.  BDB
              XML can be used to store, retrieve, manage, and query (using
              XQuery) XML documents.""",
              author = "Oracle",
              author_email = "berkeleydb-info_us@oracle.com",
              url = "http://www.oracle.com",
              py_modules = ["dbxml"],
              ext_modules = [Extension("_dbxml", ["dbxml_python_wrap.cpp"],
                                       include_dirs = INCLUDES,
                                       library_dirs = LIBDIRS,
                                       runtime_library_dirs = LIBDIRS,
                                       define_macros = DEFINES,
                                       libraries = LIBS,
                                       extra_link_args = lflags_arg
                                       )],
              data_files = DATAFILES)
elif os.name == "nt":
  INCLUDES.append(os.path.join(dbxml_home,"include"))
  INCLUDES.append(os.path.join(db_home, "build_windows"))
  INCLUDES.append(os.path.join(db_home, "dbinc"))

  if debug:
    LIBDIRS =   [os.path.join(dbxml_home,"../lib"),
                 os.path.join(dbxml_home,"build_windows/Debug"),
                 os.path.join(db_home, "build_windows/Debug"),
                 os.path.join(xqilla_home, "lib"),
                 os.path.join(xerces_home, "Build/Win32/VC7")]
  
    LIBS =      ["libdbxml25D",
                 "libdb48D", "xqilla22d",
                 "xerces-c_3D"]

    DATAFILES = [("", [os.path.join(dbxml_home,"../bin/debug/libdbxml25D.dll"),
                       os.path.join(dbxml_home,"../bin/debug/libdb48D.dll"),
                       os.path.join(dbxml_home,"../bin/debug/xqilla22d.dll"),
                       os.path.join(dbxml_home,"../bin/debug/xerces-c_3_0D.dll"),
                       os.path.join(dbxml_home,"build_windows/zlib1.dll"),
                       os.path.join(dbxml_home,"build_windows/zlibwapi.dll")])]
  else:
    LIBDIRS =   [os.path.join(dbxml_home,"../lib"),
                 os.path.join(dbxml_home,"build_windows/Release"),
                 os.path.join(db_home, "build_windows/Release"),
                 os.path.join(xqilla_home, "lib"),
                 os.path.join(xerces_home, "Build/Win32/VC7")]
  
    LIBS =      ["libdbxml25",
                 "libdb48", "xqilla22",
                 "xerces-c_3"]

    DATAFILES = [("", [os.path.join(dbxml_home,"../bin/libdbxml25.dll"),
                       os.path.join(dbxml_home,"../bin/libdb48.dll"),
                       os.path.join(dbxml_home,"../bin/xqilla22.dll"),
                       os.path.join(dbxml_home,"../bin/xerces-c_3_0.dll"),
                       os.path.join(dbxml_home,"build_windows/zlib1.dll"),
                       os.path.join(dbxml_home,"build_windows/zlibwapi.dll")])]


    # Now run with whatever settings we've got
  setup(name = "dbxml",
        version = "2.5.16",
        description = "Berkeley DB XML Python API",
        long_description = """\
        This module provides a complete wrapping of the C++ API
        to the Berkeley DB XML (BDB XML) XML database.  BDB
        XML can be used to store, retrieve, manage, and query (using
        XQuery) XML documents.""",
        author = "Oracle",
        author_email = "berkeleydb-info_us@oracle.com",
        url = "http://www.oracle.com",
        py_modules = ["dbxml"],
        ext_modules = [Extension("_dbxml", ["dbxml_python_wrap.cpp"],
                                 include_dirs = INCLUDES,
                                 library_dirs = LIBDIRS,
                                 define_macros = DEFINES,
                                 libraries = LIBS,
                                 extra_compile_args = ["/GR"]
                                 )],
        data_files = DATAFILES)
    
else:
  print "I don't know anything about your platform '%s'." % os.name
  print "Please check the build instructions for more information"
  sys.exit(1)
