# Copyright (c) 2024 The Chromium Embedded Framework Authors. All rights
# reserved. Use of this source code is governed by a BSD-style license that
# can be found in the LICENSE file.

from __future__ import absolute_import
from cef_parser import get_copyright
from file_util import write_file_if_changed
from version_util import read_version_files
import os
import sys


def make_api_versions_header(json):
  result = get_copyright(
      full=True, translator=False) + """//
// ---------------------------------------------------------------------------
//
// This file was generated by the make_api_versions_header.py tool. Versions
// are managed using the version_manager.py tool. For usage details see
// https://bitbucket.org/chromiumembedded/cef/wiki/ApiVersioning.md
//

#ifndef CEF_INCLUDE_CEF_API_VERSIONS_H_
#define CEF_INCLUDE_CEF_API_VERSIONS_H_

#include "include/base/cef_build.h"
"""

  for version, hashes in json['hashes'].items():
    version_part = f"""
// $COMMENT$
#define CEF_API_VERSION_{version} {version}
#if defined(OS_WIN)
#define CEF_API_HASH_{version} "$WINDOWS$"
#elif defined(OS_MAC)
#define CEF_API_HASH_{version} "$MAC$"
#elif defined(OS_LINUX)
#define CEF_API_HASH_{version} "$LINUX$"
#endif
"""

    # Substitute hash values for placeholders.
    for key, value in hashes.items():
      version_part = version_part.replace(f"${key.upper()}$", value)

    result += version_part

  result += f"""
// Oldest supported CEF version.
#define CEF_API_VERSION_MIN CEF_API_VERSION_{json['min']}

// Newest supported CEF version.
#define CEF_API_VERSION_LAST CEF_API_VERSION_{json['last']}

#endif  // CEF_INCLUDE_CEF_API_VERSIONS_H_
"""

  return result


def make_api_versions_inc(json):
  result = get_copyright(
      full=False, translator=False) + """//
// ---------------------------------------------------------------------------
//
// This file was generated by the make_api_versions_header.py tool.
//

namespace {

struct ApiVersionHash {
  int version;
  const char* const hash;
};

const ApiVersionHash kApiVersionHashes[] = {"""

  for version in json['hashes'].keys():
    result += f"\n{{{version}, CEF_API_HASH_{version}}},"

  result += """
};

}  // namespace
"""

  return result


def write_api_versions(out_header_file, out_inc_file, json) -> bool:
  """
  Return True if the files were written, False if no changes were made.
  """
  out_file = os.path.abspath(out_header_file)
  result = make_api_versions_header(json)
  if not result:
    sys.stderr.write(f'Failed to create {out_file}\n')
    sys.exit(1)
  header_write_result = write_file_if_changed(out_file, result)

  out_file = os.path.abspath(out_inc_file)
  result = make_api_versions_inc(json)
  if not result:
    sys.stderr.write(f'Failed to create {out_file}\n')
    sys.exit(1)
  inc_write_result = write_file_if_changed(out_file, result)

  return header_write_result or inc_write_result


def main(argv):
  if len(argv) < 5:
    print(
        f"Usage:\n {argv[0]} <output_header_file> <output_inc_file> <api_versions_file> <api_untracked_file>"
    )
    sys.exit(-1)

  json = read_version_files(argv[3], argv[4], initialize=True, combine=True)[0]

  if not write_api_versions(argv[1], argv[2], json):
    print('Nothing done')


if '__main__' == __name__:
  main(sys.argv)
