// Copyright 2024 The Chromium Embedded Framework Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be found
// in the LICENSE file.

#include "cef/libcef/browser/views/widget.h"

#include "cef/libcef/browser/chrome/views/chrome_browser_frame.h"
#include "cef/libcef/browser/views/view_util.h"
#include "cef/libcef/browser/views/widget_impl.h"
#include "cef/libcef/browser/views/window_impl.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/themes/theme_service.h"
#include "chrome/browser/themes/theme_service_factory.h"

namespace {

ui::ColorProviderKey::SchemeVariant GetSchemeVariant(
    ui::mojom::BrowserColorVariant color_variant) {
  using BCV = ui::mojom::BrowserColorVariant;
  using SV = ui::ColorProviderKey::SchemeVariant;
  static constexpr auto kSchemeVariantMap = base::MakeFixedFlatMap<BCV, SV>({
      {BCV::kTonalSpot, SV::kTonalSpot},
      {BCV::kNeutral, SV::kNeutral},
      {BCV::kVibrant, SV::kVibrant},
      {BCV::kExpressive, SV::kExpressive},
  });
  return kSchemeVariantMap.at(color_variant);
}

}  // namespace

// static
CefWidget* CefWidget::Create(CefWindowView* window_view) {
  if (window_view->IsChromeStyle()) {
    return new ChromeBrowserFrame(window_view);
  }
  return new CefWidgetImpl(window_view);
}

// static
CefWidget* CefWidget::GetForWidget(views::Widget* widget) {
  if (auto window = view_util::GetWindowFor(widget)) {
    if (auto* window_view =
            static_cast<CefWindowImpl*>(window.get())->cef_window_view()) {
      if (auto widget_view = view_util::GetHostView(widget)) {
        widget = widget_view->GetWidget();
      }
      if (window_view->IsChromeStyle()) {
        return static_cast<ChromeBrowserFrame*>(widget);
      }
      return static_cast<CefWidgetImpl*>(widget);
    }
  }
  return nullptr;
}

// static
ui::ColorProviderKey CefWidget::GetColorProviderKey(
    const ui::ColorProviderKey& widget_key,
    Profile* profile) {
  // Based on BrowserFrame::GetColorProviderKey.
  auto key = widget_key;

  const auto* theme_service = ThemeServiceFactory::GetForProfile(profile);
  CHECK(theme_service);

  // color_mode.
  [&key, theme_service]() {
    const auto browser_color_scheme = theme_service->GetBrowserColorScheme();
    if (browser_color_scheme != ThemeService::BrowserColorScheme::kSystem) {
      key.color_mode =
          browser_color_scheme == ThemeService::BrowserColorScheme::kLight
              ? ui::ColorProviderKey::ColorMode::kLight
              : ui::ColorProviderKey::ColorMode::kDark;
    }
  }();

  // user_color.
  // Device theme retains the user_color from `Widget`.
  if (!theme_service->UsingDeviceTheme()) {
    if (theme_service->UsingAutogeneratedTheme()) {
      key.user_color = theme_service->GetAutogeneratedThemeColor();
    } else if (auto user_color = theme_service->GetUserColor()) {
      key.user_color = user_color;
    }
  }

  // user_color_source.
  if (theme_service->UsingDeviceTheme()) {
    key.user_color_source = ui::ColorProviderKey::UserColorSource::kAccent;
  } else if (theme_service->GetIsGrayscale()) {
    key.user_color_source = ui::ColorProviderKey::UserColorSource::kGrayscale;
  } else if (theme_service->GetIsBaseline()) {
    key.user_color_source = ui::ColorProviderKey::UserColorSource::kBaseline;
  } else {
    CHECK(key.user_color.has_value());
    key.user_color_source = ui::ColorProviderKey::UserColorSource::kAccent;
  }

  // scheme_variant.
  ui::mojom::BrowserColorVariant color_variant =
      theme_service->GetBrowserColorVariant();
  if (!theme_service->UsingDeviceTheme() &&
      color_variant != ui::mojom::BrowserColorVariant::kSystem) {
    key.scheme_variant = GetSchemeVariant(color_variant);
  }

  // frame_type.
  key.frame_type = ui::ColorProviderKey::FrameType::kNative;

  return key;
}
