// Copyright (c) 2015 The Chromium Embedded Framework Authors. All rights
// reserved. Use of this source code is governed by a BSD-style license that
// can be found in the LICENSE file.

#ifndef CEF_TESTS_CEFCLIENT_BROWSER_MAIN_CONTEXT_H_
#define CEF_TESTS_CEFCLIENT_BROWSER_MAIN_CONTEXT_H_
#pragma once

#include <string>

#include "include/base/cef_macros.h"
#include "include/base/cef_ref_counted.h"
#include "include/cef_command_line.h"
#include "include/internal/cef_types_wrappers.h"
#include "tests/cefclient/browser/osr_renderer_settings.h"

namespace client {

class RootWindowManager;

// Used to store global context in the browser process. The methods of this
// class are thread-safe unless otherwise indicated.
class MainContext {
 public:
  // Returns the singleton instance of this object.
  static MainContext* Get();

  // Returns the global command-line.
  virtual CefRefPtr<CefCommandLine> GetCommandLine() = 0;

  // Returns the full path to the console log file.
  virtual std::string GetConsoleLogPath() = 0;

  // Returns the full path to |file_name|.
  virtual std::string GetDownloadPath(const std::string& file_name) = 0;

  // Returns the app working directory including trailing path separator.
  virtual std::string GetAppWorkingDirectory() = 0;

  // Returns the main application URL based on |command_line| and global state.
  virtual std::string GetMainURL(CefRefPtr<CefCommandLine> command_line) = 0;

  // Returns the background color.
  virtual cef_color_t GetBackgroundColor() = 0;

  // Returns true if the Views framework will be used as the global default.
  virtual bool UseViewsGlobal() = 0;

  // Returns true if Alloy style will be used as the global default. Alloy style
  // is optional. Windowless rendering requires Alloy style.
  virtual bool UseAlloyStyleGlobal() = 0;

  // Returns true if touch events are enabled.
  virtual bool TouchEventsEnabled() = 0;

  // Returns true if the default popup implementation should be used.
  virtual bool UseDefaultPopup() = 0;

  // Populate |settings| based on command-line arguments.
  virtual void PopulateSettings(CefSettings* settings) = 0;
  virtual void PopulateBrowserSettings(CefBrowserSettings* settings) = 0;
  virtual void PopulateOsrSettings(OsrRendererSettings* settings) = 0;

  // Returns the object used to create/manage RootWindow instances.
  virtual RootWindowManager* GetRootWindowManager() = 0;

 protected:
  MainContext();
  virtual ~MainContext();

 private:
  DISALLOW_COPY_AND_ASSIGN(MainContext);
};

}  // namespace client

#endif  // CEF_TESTS_CEFCLIENT_BROWSER_MAIN_CONTEXT_H_
