/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */ 

#ifndef CPP_UNIT_CAPABILITYTEST_H
#define CPP_UNIT_CAPABILITYTEST_H

/* A test case for the Capability. */

class FdoCapabilityTest : public CppUnit::TestCase
{

    CPPUNIT_TEST_SUITE(FdoCapabilityTest);
    CPPUNIT_TEST(Connection);
    CPPUNIT_TEST(Datastore);
    CPPUNIT_TEST(Schema);
    CPPUNIT_TEST(GeomFilter);
    CPPUNIT_TEST_SUITE_END();


    private:

        // The class ArgObject stores a single signature argument together
        // with a flag indicating whether or not it has been processed during
        // a cross check operation.

        class ArgObject {

            public:

                wchar_t     *name;
                FdoDataType dataType;
                bool        found;

                ArgObject  ();
                ~ArgObject ();

        };  //  class ArgObject

        // The class ClassTypeElement stores a class type the provider supports
        // and a flag indicating whether or not it has been processed during a
        // cross check operation.

        class ClassTypeElement {

            public:

                FdoClassType classType;
                bool         checked;

                ClassTypeElement  ();
                ~ClassTypeElement ();

        };  //  class ClassTypeElement

        // The class DataTypeElement stores a data type the provider supports
        // and a flag indicating whether or not it has been processed during a
        // cross check operation.

        class DataTypeElement {

            public:

                FdoDataType dataType;
                bool        checked;

                DataTypeElement ();
                ~DataTypeElement ();

        };  //  class DataTypeElement

        // The class ExpectedSchemaCapabilities stores all expected schema
        // capabilities. When executing the schema capability APIs the returned
        // data is cross checked against the data provided by this class.

        class ExpectedSchemaCapabilities {

            public:

                // Lists of expected schema capability data.

                std::vector<DataTypeElement*>  autogeneratedDataTypes;
                std::vector<ClassTypeElement*> classTypes;
                std::vector<DataTypeElement*>  dataTypes;
                std::vector<DataTypeElement*>  idPropTypes;

                // Data value length limitations

                FdoInt64 blobDataValueLimit;
                FdoInt64 clobDataValueLimit;
                FdoInt64 decimalDataValueLimit;
                FdoInt64 booleanDataValueLimit;
                FdoInt64 byteDataValueLimit;
                FdoInt64 dateTimeDataValueLimit;
                FdoInt64 doubleDataValueLimit;
                FdoInt64 int16DataValueLimit;
                FdoInt64 int32DataValueLimit;
                FdoInt64 int64DataValueLimit;
                FdoInt64 singleDataValueLimit;
                FdoInt64 stringDataValueLimit;

                // Name size limitations.

                FdoInt32 classNameLimit;
                FdoInt32 datastoreNameLimit;
                FdoInt32 descriptionLimit;
                FdoInt32 propertyNameLimit;
                FdoInt32 schemaNameLimit;

                // Reserved character list.

                FdoStringP reservedCharacters;

                // Schema capability flags.

                bool supportsAssociationProperties;
                bool supportsAutoIdGeneration;
                bool supportsCompositeId;
                bool supportsCompositeUniqueValueConstraints;
                bool supportsDataStoreScopeUniqueIdGeneration;
                bool supportsDefaultValue;
                bool supportsExclusiveValueRangeConstraints;
                bool supportsInclusiveValueRangeConstraints;
                bool supportsInheritance;
                bool supportsMultipleSchemas;
                bool supportsNetworkModel;
                bool supportsNullValueConstraints;
                bool supportsObjectProperties;
                bool supportsSchemaModification;
                bool supportsSchemaOverrides;
                bool supportsUniqueValueConstraints;
                bool supportsValueConstraintsList;


                // Class Functions.

                ExpectedSchemaCapabilities  ();
                ~ExpectedSchemaCapabilities ();

                void Clear     ();
                void SetToZero ();

        };  //  class ExpectedSchemaCapabilities

        // The class FctSignature stores a single signature definition together
        // with a flag indicating whether or not it has been processed during a
        // cross check operation.

        class FctSignature {

            public:

                FdoDataType     returnDataType;
                FdoPropertyType returnPropertyType;
                bool            expectDataType;
                ArgObject       *arguments[10];
                FdoInt32        argCount;
                bool            found;

                FctSignature  ();
                ~FctSignature ();

        };  //  class FctSignature

        // The class FctObject contains a single function definition together
        // with a flag indicating whether or not it has been processed during
        // a cross check operation.

        class FctObject {

            public:

                wchar_t      *name;
                bool         isAggregate;
                FctSignature *signature[20];
                FdoInt32     signatureCount;
                bool         found;

                FctObject  ();
                ~FctObject ();

        };  //  class FctObject


    public:

        FdoCapabilityTest ();
        virtual ~FdoCapabilityTest ();
        void setUp ();

    protected:

        //  CheckFunction:
        //      The function checks whether or not the named function is listed
        //      in the collection of the expected functions. 

        void CheckFunction (FdoString *name,
                            bool      isAggregate);

        //  CheckExpectation:
        //      The function checks whether or not the list of supported
        //      function signatures matches the list of expected ones.

        void CheckExpectation ();

        //  CheckSignature:
        //      The function checks whether or not the identified signature is
        //      listed in the collection of the expected signatures for a
        //      function.

        void CheckSignature (FdoDataType fctReturnType,
                             FdoString   *argName,
                             FdoDataType argDataType);

        //  Connection:
        //      Unit test function to check connection capabilities.

        void Connection ();

        //  Datastore:
        //      Unit test function to check datastore capabilities.

        void Datastore ();

        //  GeomFilter:
        //      Unit test function to check geometry filter caps.
        void GeomFilter();

        //  GetDataValueLimitErrorMsg:
        //      The function returns an error text using the provided data.

        FdoStringP GetDataValueLimitErrorMsg (FdoString *dataTypeId,
                                              FdoInt64  expectedValue,
                                              FdoInt64  actualValue);

        //  GetDataType:
        //      The function returns a string equivalent for the given data
        //      type .

        FdoString *GetDataType (FdoDataType dataType);

        //  GetUncheckedDataTypes:
        //      The function navigates through the provided list of data types
        //      and constructs a string of all the unchecked data types. The
        //      string is returned back to the caller.

        FdoStringP GetUncheckedDataTypes (
                                    std::vector<DataTypeElement*> dataTypes);

        //  GetUnexpectedFlagValueErrorString:
        //      The function returns an error string based on the provided
        //      information.

        FdoStringP GetUnexpectedFlagValueErrorString (FdoString *flagName,
                                                      bool      expected_value,
                                                      bool      actualValue);

        //  GetLockTypeCount:
        //      The function returns the number of lock types supported by the current provider.

        virtual FdoInt32 GetLockTypeCount();

        //  InitArgument:
        //      The function creates a new object of type "ArgObject", 
        //      initializes it using the provided data and returns it back
        //      to the caller.

        virtual ArgObject *InitArgument (FdoString   *argName,
                                         FdoDataType argDataType);

        //  InitClassTypeElement:
        //      The function creates a new object of type "ClassTypeElement",
        //      initializes it using the provided data and returns it back
        //      to the caller.

        virtual ClassTypeElement *InitClassTypeElement (
                                                    FdoClassType classType);

        //  InitDataTypeElement:
        //      The function creates a new object of type "DataTypeElement", 
        //      initializes it using the provided data and returns it back
        //      to the caller.

        virtual DataTypeElement *InitDataTypeElement (FdoDataType dataType);

        //  InitExpectedSchemaCapabilityData:
        //      The function initializes the object with the data expected to
        //      be returned by the schema capability APIs.

        virtual void InitExpectedSchemaCapabilityData ();

        //  InitSchemaCapabilitiesAutogeneratedDataTypes:
        //      The function initializes the list of expected data types the
        //      provider supports for autogenerated properties.

        virtual void InitSchemaCapabilitiesAutogeneratedDataTypes ();

        //  InitSchemaCapabilitiesClassTypes:
        //      The function initializes the list of expected class types the
        //      provider supports.

        virtual void InitSchemaCapabilitiesClassTypes ();

        //  InitSchemaCapabilitiesDataTypes:
        //      The function initializes the list of expected data types the
        //      provider supports.

        virtual void InitSchemaCapabilitiesDataTypes ();

        //  InitSchemaCapabilitiesFlags:
        //      The function initializes the list of expected schema capability
        //      flags for the provider.

        virtual void InitSchemaCapabilitiesFlags ();

        //  InitSchemaCapabilitiesIdPropDataTypes:
        //      The function initializes the list of expected identity property
        //      data types the provider supports.

        virtual void InitSchemaCapabilitiesIdPropDataTypes ();

        //  InitSchemaCapabilitiesMaxDataValueLimits:
        //      The function initializes expected data value limits for the
        //      provider.

        virtual void InitSchemaCapabilitiesMaxDataValueLimits ();

        //  InitSchemaCapabilitiesNameSizeLimits:
        //      The function initializes expected name size limits the provider
        //      supports.

        virtual void InitSchemaCapabilitiesNameSizeLimits ();

        //  InitSchemaCapabilitiesReservedCharacters:
        //      The function initializes the list of reserved characters for
        //      the provider.

        virtual void InitSchemaCapabilitiesReservedCharacters ();

        //  InitSignature:
        //      The function creates a new object of type "FctSignature", 
        //      initializes it using the provided data and returns it back
        //      to the caller.

        virtual FctSignature *InitSignature (FdoDataType     retDataType,
                                             FdoPropertyType retPropertyType,
                                             bool            expDataType);

        //  InitSupportedFunctions:
        //      The function creates the initial list of supported functions
        //      and their supported signatures.

        virtual void InitSupportedFunctions ();

        //  MapClassTypeToString:
        //      The function maps a provided class type to a its string
        //      equivalent.

        FdoStringP MapClassTypeToString (FdoClassType classType);

        //  MapDataTypeToString:
        //      The function maps a provided data type to a its string
        //      equivalent.

        FdoStringP MapDataTypeToString (FdoDataType dataType);

        //  MarkDataType:
        //      The function searched the given data type in the provided list
        //      of data types. If found, the entry is marked as being processed.

        void MarkDataType (std::vector<DataTypeElement*> dataTypes,
                           FdoDataType                   dataType);

        //  Schema:
        //      Unit test function to check schema capabilities.

        void Schema ();

        //  Schema_CheckAutoGeneratedDataTypes:
        //      The function checks whether or not the corresponding schema
        //      capability API returns the expected set of data types that
        //      can be used for autogenerated properties.

        void Schema_CheckAutoGeneratedDataTypes (
                                FdoISchemaCapabilities *schemaCapabilities);

        //  Schema_CheckClassTypes:
        //      The function checks whether or not the corresponding schema
        //      capability API returns the expected set of class types the
        //      provider supports.

        void Schema_CheckClassTypes (
                                FdoISchemaCapabilities *schemaCapabilities);

        //  Schema_CheckDataTypes:
        //      The function checks whether or not the corresponding schema
        //      capability API returns the expected set of supported data.

        void Schema_CheckDataTypes (
                                FdoISchemaCapabilities *schemaCapabilities);

        //  Schema_CheckFlags:
        //      The function checks whether or not the corresponding schema
        //      capability API returns the expected flag value for the 
        //      supported schema capability flags.

        void Schema_CheckFlags (FdoISchemaCapabilities *schemaCapabilities);

        //  Schema_CheckIdPropDataTypes:
        //      The function checks whether or not the corresponding schema
        //      capability API returns the expected list of supported identity
        //      property data types.

        void Schema_CheckIdPropDataTypes (
                                FdoISchemaCapabilities *schemaCapabilities);

        //  Schema_CheckMaxDataValueLimits:
        //      The function checks whether or not the corresponding schema
        //      capability API returns the expected name data value lmits.

        void Schema_CheckMaxDataValueLimits (
                                FdoISchemaCapabilities *schemaCapabilities);

        //  Schema_CheckNameSizeLimits:
        //      The function checks whether or not the corresponding schema
        //      capability API returns the expected name size limits.

        void Schema_CheckNameSizeLimits (
                                FdoISchemaCapabilities *schemaCapabilities);

        //  Schema_CheckReservedCharacters:
        //      The function checks whether or not the corresponding schema
        //      capability API returns the expected set of reserved characters.

        void Schema_CheckReservedCharacters (
                                FdoISchemaCapabilities *schemaCapabilities);

        //  Schema_CrossCheckClassTypes:
        //      The function checks whether or not the corresponding schema
        //      capability API returned all expected class types for the
        //      provider.

        void Schema_CrossCheckClassTypes ();

        //  Schema_MarkClassTypes:
        //      The function searches the list of expected class types for
        //      the provided class type. If found the entry is marked as being
        //      processed.

        void Schema_MarkClassTypes (FdoClassType classType);

        //  SetProvider:
        //      Function to set the provider object for the test.

        virtual void SetProvider () {};   


        //  m_expectedSchemaCapabilities:
        //      Represents the object that holds the expected data the schema
        //      capability APIs should return.

        ExpectedSchemaCapabilities *m_expectedSchemaCapabilities;

        //  m_supportedFunctions:
        //      Represents the list of supported functions.

        FctObject *m_supportedFunctions[20];

        //  m_currFctObject:
        //      Represents a pointer to the current supported function in the
        //      the list of supported functions.

        FctObject *m_currFctObject;

        //  m_numSupportedFunctions:
        //      Contains the number of expected supported functions.

        FdoInt32 m_numSupportedFunctions;

};  //  class FdoCapabilityTest

#endif // CPP_UNIT_CAPABILITYTEST_H
