/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "Pch.h"
#include "RecommitTest.h"

CPPUNIT_TEST_SUITE_REGISTRATION (RecommitTest);
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION (RecommitTest, "RecommitTest");

FdoPtr<FdoIConnection> RecommitTest::mConnection;

int iCityId1 = -1;
int iCityId2 = -1;
int iBeachId1 = -1;
int iBeachId2 = -1;
int iBeachId3 = -1;
int iBeachId4 = -1;


RecommitTest::RecommitTest (void)
{
}

RecommitTest::~RecommitTest (void)
{
}

void RecommitTest::setUp ()
{
    static bool bSchemaCreated = false;

    if (!CreateSchemaOnly() || !bSchemaCreated)
    {
        mConnection = ArcSDETests::GetConnection ();
        mConnection->SetConnectionString (ArcSDETestConfig::ConnStringMetadcov());
        mConnection->Open ();
    }

    if (CreateSchemaOnly())
    {
        if (!bSchemaCreated)
        {
            try
            {
                CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaCity(), ArcSDETestConfig::ClassNameCity());
            }
            catch (FdoException* e)
            {
                e->Release ();
            }
            catch (...)
            {
                throw;
            }

            try
            {
                CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaBeach(), ArcSDETestConfig::ClassNameBeach());
            }
            catch (FdoException* e)
            {
                e->Release ();
            }
            catch (...)
            {
                throw;
            }

            // Create schema
            CreateSchema ();

            bSchemaCreated = true;
        }
    }
    else
    {
        CleanUpLTs();

        CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaCity(), ArcSDETestConfig::ClassNameCity(), true);
        CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaBeach(), ArcSDETestConfig::ClassNameBeach(), true);

        // Add some data
        AddData ();
    }
}

void RecommitTest::tearDown ()
{
    if (CreateSchemaOnly())
    {
        // Leave schema around
    }
    else
    {
        CleanUpLTs();
    }

    if (mConnection)
        mConnection->Close ();
}


void RecommitTest::CleanUpLTs()
{
    try
    {
        if (longTransactionExists (L"versionconflict"))
            rollbackLongTransaction (L"versionconflict");

        if (longTransactionExists (L"LT1"))
            rollbackLongTransaction (L"LT1");
        if (longTransactionExists (L"LT2"))
            rollbackLongTransaction (L"LT2");
    }
    catch (FdoException *e)
    {
        e->Release ();
        fail (e);
    }
    catch (...)
    {
        throw;
    }
}


void RecommitTest::CreateSchema ()
{
    FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
    FdoPtr<FdoFeatureSchema> schema = ArcSDETests::GetDefaultSchema (mConnection);
    FdoPtr<FdoClassCollection> classes = schema->GetClasses ();

    // build a simple identity and a few data property
    FdoPtr<FdoDataPropertyDefinition> id = FdoDataPropertyDefinition::Create (L"Id", L"Unique identifier");
    id->SetDataType (FdoDataType_Int32);
    id->SetNullable (false);
    id->SetIsAutoGenerated (true);
    id->SetReadOnly (true);

    FdoPtr<FdoDataPropertyDefinition> males = FdoDataPropertyDefinition::Create (L"MALE65_", L"males over 65 years");
    males->SetDataType (FdoDataType_Int32);
    males->SetPrecision (0);
    males->SetScale (0);
    males->SetNullable (false);

    FdoPtr<FdoDataPropertyDefinition> houses = FdoDataPropertyDefinition::Create (L"VACANT_HOU", L"vacant rental properties");
    houses->SetDataType (FdoDataType_Int32);
    houses->SetPrecision (0);
    houses->SetScale (0);
    houses->SetNullable (false);

    FdoPtr<FdoDataPropertyDefinition> area = FdoDataPropertyDefinition::Create (L"AREANAME", L"geographical area");
    area->SetDataType (FdoDataType_String);
    area->SetLength (25);

    // assemble the class
    FdoPtr<FdoClass> cls = FdoClass::Create (L"CITY", L"test class");
    FdoPtr<FdoPropertyDefinitionCollection> properties = cls->GetProperties ();
    properties->Add (id);
    properties->Add (males);
    properties->Add (houses);
    properties->Add (area);
    FdoPtr<FdoDataPropertyDefinitionCollection> identities = cls->GetIdentityProperties ();
    identities->Add (id);

    // set class capabilities
    SetClassCapabilities (cls, false, true);

    // add it to the schema
    classes->Add (cls);

    // build a seconde simple identity and a few data property
    id = FdoDataPropertyDefinition::Create (L"Id", L"Unique identifier");
    id->SetDataType (FdoDataType_Int32);
    id->SetNullable (false);
    id->SetIsAutoGenerated (true);
    id->SetReadOnly (true);

    FdoPtr<FdoDataPropertyDefinition> owner = FdoDataPropertyDefinition::Create (L"OWNER", L"property owner");
    owner->SetDataType (FdoDataType_String);
    owner->SetLength (25);

    FdoPtr<FdoDataPropertyDefinition> county = FdoDataPropertyDefinition::Create (L"COUNTY", L"location (county)");
    county->SetDataType (FdoDataType_String);
    county->SetLength (25);

    // assemble the class
    cls = FdoClass::Create (L"BEACH", L"second test class");
    properties = cls->GetProperties ();
    properties->Add (id);
    properties->Add (owner);
    properties->Add (county);
    identities = cls->GetIdentityProperties ();
    identities->Add (id);

    // set class capabilities
    SetClassCapabilities (cls, false, true);

    // add it to the schema
    classes->Add (cls);

    // submit the new schema
    apply->SetFeatureSchema (schema);
    apply->Execute ();
}

void RecommitTest::AddData ()
{
    FdoPtr <FdoITransaction> transaction = mConnection->BeginTransaction ();
    FdoPtr<FdoIInsert> insert = (FdoIInsert*)mConnection->CreateCommand (FdoCommandType_Insert);
    insert->SetFeatureClassName (ArcSDETestConfig::QClassNameCity());

    FdoPtr<FdoPropertyValueCollection> values = insert->GetPropertyValues ();
    FdoPtr<FdoPropertyValue> males = FdoPropertyValue::Create ();
    FdoPtr<FdoPropertyValue> house = FdoPropertyValue::Create ();
    FdoPtr<FdoPropertyValue> area = FdoPropertyValue::Create ();
    males->SetName (L"MALE65_");
    house->SetName (L"VACANT_HOU");
    area->SetName (L"AREANAME");
    males->SetValue (L"156000");
    house->SetValue (L"4289");
    area->SetValue (L"'OTTAWA, ONTARIO'");
    values->Add (males);
    values->Add (house);
    values->Add (area);
    FdoPtr<FdoIFeatureReader> reader = insert->Execute ();
    reader->ReadNext();
    iCityId1 = reader->GetInt32(AdjustRdbmsName(L"Id"));
    reader->Close();

    males->SetValue (L"254000");
    house->SetValue (L"9191");
    area->SetValue (L"'VANCOUVER, BC'");
    reader = insert->Execute ();
    reader->ReadNext();
    iCityId2 = reader->GetInt32(AdjustRdbmsName(L"Id"));
    reader->Close();

    insert->SetFeatureClassName (ArcSDETestConfig::QClassNameBeach());
    values->Clear ();
    FdoPtr<FdoPropertyValue> owner = FdoPropertyValue::Create ();
    FdoPtr<FdoPropertyValue> county = FdoPropertyValue::Create ();
    owner->SetName (L"OWNER");
    county->SetName (L"COUNTY");
    owner->SetValue (L"'CASIDY/MARINA WAY MRGE'");
    county->SetValue (L"'AB'");
    values->Add (owner);
    values->Add (county);
    reader = insert->Execute ();
    reader->ReadNext();
    iBeachId1 = reader->GetInt32(AdjustRdbmsName(L"Id"));
    reader->Close();

    owner->SetValue (L"'LOFT PROPERTIES'");
    county->SetValue (L"'OO'");
    reader = insert->Execute ();
    reader->ReadNext();
    iBeachId2 = reader->GetInt32(AdjustRdbmsName(L"Id"));
    reader->Close();

    transaction->Commit ();
}

void RecommitTest::recommit ()
{
    //_CrtSetDbgFlag (_CRTDBG_CHECK_ALWAYS_DF);

    if (CreateSchemaOnly())  return;

    // activate the spatial context
    activateSpatialContext (L"12");

    // create and activate a long transaction
    createLongTransaction (L"LT1");
    activateLongTransaction (L"LT1");
    const wchar_t* name = getActiveLongTransaction ();
    FdoStringP  expectedLtName = FdoStringP::Format(L"%ls.LT1", (FdoString*)ArcSDETestConfig::UserNameMetadcov());
    CPPUNIT_ASSERT_MESSAGE ("LT1 long transaction not active", 0 == wcscmp (expectedLtName, name));
    delete[] name;

    // edit the data
    updateData (L"99999");

    // create version conflict, and commit the new long transaction
    createVersionConflict ();
    name = getActiveLongTransaction ();
    CPPUNIT_ASSERT_MESSAGE ("DEFAULT long transaction not active", 0 == wcscmp (FdoStringP::Format(L"%ls.DEFAULT", (const wchar_t *) ArcSDETestConfig::SdeUsername()), name));  //TODO
    delete[] name;

    activateLongTransaction (L"LT1");
    name = getActiveLongTransaction ();
    CPPUNIT_ASSERT_MESSAGE ("LT1 long transaction not active", 0 == wcscmp (expectedLtName, name));
    delete[] name;

    // commit the existing long transaction
    FdoPtr<FdoICommitLongTransaction> commitCmd = (FdoICommitLongTransaction*)mConnection->CreateCommand (FdoCommandType_CommitLongTransaction);
    commitCmd->SetName (L"LT1");
    FdoPtr<FdoILongTransactionConflictDirectiveEnumerator> enumerator = commitCmd->Execute ();
    CPPUNIT_ASSERT_MESSAGE ("wrong number of conflicts", 2 == enumerator->GetCount ());

    // resolve the version conflicts
    while (enumerator->ReadNext ())
        enumerator->SetResolution (FdoLongTransactionConflictResolution_Child);

    // recommit
    enumerator = commitCmd->Execute ();

    // check the number of unresolved conflicts.
    CPPUNIT_ASSERT_MESSAGE ("after recommiting, enumerator has conflicts", 0 == enumerator->GetCount ());
}

void RecommitTest::multibranch (void)
{
    if (CreateSchemaOnly())  return;

	// This test involves the following steps:
	// 1. Activate the spatial context
	// 2. Create first long transaction (LT1)	
	// 3. Create second long transaction (LT2)
	// 4. Edit data in LT1
	// 5. Edit data in LT2
	// 6. Commit data in LT2
	// 7. Commit data in LT1
	// 8. Resolve version conflicts

    // activate the spatial context
    activateSpatialContext (L"12");
    
    // create the first long transaction
    createLongTransaction (L"LT1");

    // create the second long transaction
    createLongTransaction (L"LT2");

    // edit data in LT1
    activateLongTransaction (L"LT1");
    const wchar_t* name = getActiveLongTransaction ();
    FdoStringP  expectedLt1Name = FdoStringP::Format(L"%ls.LT1", (FdoString*)ArcSDETestConfig::UserNameMetadcov());
    CPPUNIT_ASSERT_MESSAGE ("LT1 long transaction not active", 0 == wcscmp (expectedLt1Name, name));
    delete[] name;
    updateData1 ();
    insertData1 ();
    deleteData1 ();

    // edit data in LT2
    activateLongTransaction (L"LT2");
    name = getActiveLongTransaction ();
    FdoStringP  expectedLt2Name = FdoStringP::Format(L"%ls.LT2", (FdoString*)ArcSDETestConfig::UserNameMetadcov());
    CPPUNIT_ASSERT_MESSAGE ("LT2 long transaction not active", 0 == wcscmp (expectedLt2Name, name));
    updateData2 (); // update the beach feature that was deleted in LT1
    deleteData2 (); // delete the city feature that was updated in LT1
    insertData2 (); // insert a new beach feature

    // commit data in LT2
	FdoPtr<FdoICommitLongTransaction> commit = (FdoICommitLongTransaction*)mConnection->CreateCommand (FdoCommandType_CommitLongTransaction);
	commit->SetName (L"LT2");
	FdoPtr<FdoILongTransactionConflictDirectiveEnumerator> conflicts = commit->Execute();
    CPPUNIT_ASSERT_MESSAGE ("conflicts detected", 0 == conflicts->GetCount ());

    // try to commit data in LT1
	activateLongTransaction (L"LT1");	
	commit = (FdoICommitLongTransaction*)mConnection->CreateCommand (FdoCommandType_CommitLongTransaction);
	commit->SetName (L"LT1");
	conflicts = commit->Execute ();
    CPPUNIT_ASSERT_MESSAGE ("no conflicts detected", 0 != conflicts->GetCount ());

    // resolve version conflicts
    while (conflicts->ReadNext ())
        conflicts->SetResolution (FdoLongTransactionConflictResolution_Child);

    // recommit
	conflicts = commit->Execute ();
    CPPUNIT_ASSERT_MESSAGE ("conflicts detected", 0 == conflicts->GetCount ());

    // check that it's what we expect
    FdoPtr<FdoISelect> select = (FdoISelect*)mConnection->CreateCommand (FdoCommandType_Select);
    select->SetFeatureClassName (ArcSDETestConfig::QClassNameCity());
    FdoPtr<FdoIFeatureReader> reader = select->Execute ();
    bool found = false;
    while (reader->ReadNext ())
    {
        if (iCityId2 == reader->GetInt32 (AdjustRdbmsName(L"Id")))
        {
            CPPUNIT_ASSERT_MESSAGE ("LT1 edit missing", 0 == wcscmp (L"CALGARY, ALBERTA" , reader->GetString (L"AREANAME")));
            found = true;
        }
    }
    CPPUNIT_ASSERT_MESSAGE ("LT1 feature not found", found);

    select->SetFeatureClassName (ArcSDETestConfig::QClassNameBeach());
    reader = select->Execute ();
    bool found3 = false;
    bool found4 = false;
    while (reader->ReadNext ())
    {
        if (iBeachId1 == reader->GetInt32 (AdjustRdbmsName(L"Id")))
            CPPUNIT_FAIL ("child didn't win the delete");
        else if (iBeachId3 == reader->GetInt32 (AdjustRdbmsName(L"Id")))
            found3 = true;
        else if (iBeachId4 == reader->GetInt32 (AdjustRdbmsName(L"Id")))
            found4 = true;
    }
    CPPUNIT_ASSERT_MESSAGE ("LT1 feature not found", found3);
    CPPUNIT_ASSERT_MESSAGE ("expected non-conflicting LT2 feature not found", found4);
}

void RecommitTest::activateSpatialContext (const FdoString* spatialContextName)
{
    try
    {
        FdoPtr<FdoIActivateSpatialContext> command = (FdoIActivateSpatialContext*)mConnection->CreateCommand (FdoCommandType_ActivateSpatialContext);
        command->SetName (spatialContextName);
        command->Execute ();
    }
    catch(FdoException *ex)
    {
        ex->Release();
    }
}

void RecommitTest::activateLongTransaction (const FdoString *LTName)
{
    FdoPtr<FdoIActivateLongTransaction> command = (FdoIActivateLongTransaction*)mConnection->CreateCommand (FdoCommandType_ActivateLongTransaction);
    command->SetName (LTName);
    command->Execute ();
}

void RecommitTest::deactivateLongTransaction ()
{
    FdoPtr<FdoIDeactivateLongTransaction> command = (FdoIDeactivateLongTransaction*)mConnection->CreateCommand (FdoCommandType_DeactivateLongTransaction);    
    command->Execute ();
}

void RecommitTest::createLongTransaction (const FdoString *LTName)
{
    FdoPtr<FdoICreateLongTransaction> command = (FdoICreateLongTransaction*)mConnection->CreateCommand (FdoCommandType_CreateLongTransaction);
    command->SetName (LTName);
    command->Execute ();
}

void RecommitTest::rollbackLongTransaction (const FdoString *LTName)
{
    FdoPtr<FdoIRollbackLongTransaction> command = (FdoIRollbackLongTransaction*)mConnection->CreateCommand (FdoCommandType_RollbackLongTransaction);
    command->SetName (LTName);
    command->Execute ();
}

bool RecommitTest::longTransactionExists (const FdoString *LTName)
{
    try
    {
        FdoPtr<FdoIGetLongTransactions> command = (FdoIGetLongTransactions*)mConnection->CreateCommand (FdoCommandType_GetLongTransactions);
        command->SetName (LTName);
        FdoPtr<FdoILongTransactionReader> reader = command->Execute ();        
        if (reader->ReadNext ())
            return true;
        else
            return false;
    }
    catch (FdoException *ex)
    {
        ex->Release ();
        return false;
    }
}

const FdoString* RecommitTest::getActiveLongTransaction ()
{
    FdoPtr<FdoIGetLongTransactions> command = (FdoIGetLongTransactions*)mConnection->CreateCommand (FdoCommandType_GetLongTransactions);
    FdoPtr<FdoILongTransactionReader> reader = command->Execute ();
    

    while (reader->ReadNext ())
        if (reader->IsActive ())
            break;

    // Copy the name of the active long transaction
    size_t lBufferSize = wcslen (reader->GetName ()) + 1;
    wchar_t *activeLTName = new wchar_t[lBufferSize];    
    FdoCommonOSUtil::swprintf (activeLTName, lBufferSize, L"%ls", reader->GetName ());
    return activeLTName;
}


void RecommitTest::createVersionConflict () {
    FdoString *LTName = L"versionconflict";

    // Reactivate the ROOT long transaction
    deactivateLongTransaction ();

    // Create and activate the long transaction
    createLongTransaction (LTName);
    activateLongTransaction (LTName);    

    // Update the data
    FdoPtr<FdoITransaction> trans = mConnection->BeginTransaction ();
    updateData (L"111");
    trans->Commit ();    

    // Commit the long transaction
    FdoPtr<FdoICommitLongTransaction> commitCmd = (FdoICommitLongTransaction*)mConnection->CreateCommand (FdoCommandType_CommitLongTransaction);
    commitCmd->SetName (LTName);
    FdoPtr<FdoILongTransactionConflictDirectiveEnumerator> enumerator = commitCmd->Execute ();

    CPPUNIT_ASSERT_MESSAGE ("after creating version conflict, enumerator has conflicts", 0 == enumerator->GetCount ());
}

void RecommitTest::updateData (FdoString *val) {
    // Create the Update command to update CITY features
    FdoPtr<FdoIUpdate> command = (FdoIUpdate*)mConnection->CreateCommand (FdoCommandType_Update);
    command->SetFeatureClassName (ArcSDETestConfig::QClassNameCity());
    command->SetFilter (L"MALE65_ > 10000");

    // set the property values to update
    FdoPtr<FdoPropertyValueCollection> propertyValues = command->GetPropertyValues ();
    FdoPtr<FdoPropertyValue> vacantHouseProp = FdoPropertyValue::Create ();
    vacantHouseProp->SetName (L"VACANT_HOU");
    vacantHouseProp->SetValue (val);
    propertyValues->Add (vacantHouseProp);
    FdoInt32 numFeatures = command->Execute ();
    CPPUNIT_ASSERT_MESSAGE ("wrong number of features updated", 1 == numFeatures);
}

void RecommitTest::updateData1 ()
{
	FdoPtr<FdoITransaction> transaction = mConnection->BeginTransaction();
	FdoPtr<FdoIUpdate> update = (FdoIUpdate*)mConnection->CreateCommand (FdoCommandType_Update);		
	update->SetFeatureClassName (ArcSDETestConfig::QClassNameCity());
	update->SetFilter (L"MALE65_ > 175000");
	FdoPtr<FdoPropertyValueCollection> properties = update->GetPropertyValues ();
    FdoPtr<FdoStringValue> calgary = FdoStringValue::Create (L"CALGARY, ALBERTA");
	FdoPtr<FdoPropertyValue> value = FdoPropertyValue::Create (L"AREANAME", calgary);
	properties->Add (value);
    update->Execute ();
	transaction->Commit();
}

void RecommitTest::insertData1 ()
{
	FdoPtr<FdoITransaction> transaction = mConnection->BeginTransaction ();
	FdoPtr<FdoIInsert> insert = (FdoIInsert*)mConnection->CreateCommand (FdoCommandType_Insert);		
	insert->SetFeatureClassName (ArcSDETestConfig::QClassNameBeach());
    FdoPtr<FdoPropertyValueCollection> properties = insert->GetPropertyValues ();
    FdoPtr<FdoStringValue> vh = FdoStringValue::Create (L"VH");
	FdoPtr<FdoPropertyValue> value = FdoPropertyValue::Create (L"COUNTY", vh);
	properties->Add (value);
	FdoPtr<FdoIFeatureReader> reader = insert->Execute ();
    reader->ReadNext();
    iBeachId4 = reader->GetInt32(AdjustRdbmsName(L"Id"));
    reader->Close();
    transaction->Commit();
}

void RecommitTest::deleteData1 ()
{
	FdoPtr<FdoITransaction> transaction = mConnection->BeginTransaction ();
    FdoPtr<FdoIDelete> del = (FdoIDelete*)mConnection->CreateCommand (FdoCommandType_Delete);
    del->SetFeatureClassName (ArcSDETestConfig::QClassNameBeach());
    del->SetFilter (L"owner = 'CASIDY/MARINA WAY MRGE'");	
	del->Execute ();
	transaction->Commit ();
}

void RecommitTest::updateData2 ()
{
	FdoPtr<FdoITransaction> transaction = mConnection->BeginTransaction();
	FdoPtr<FdoIUpdate> update = (FdoIUpdate*)mConnection->CreateCommand (FdoCommandType_Update);		
	update->SetFeatureClassName (ArcSDETestConfig::QClassNameBeach());
	update->SetFilter (L"owner = 'CASIDY/MARINA WAY MRGE'");
	FdoPtr<FdoPropertyValueCollection> properties = update->GetPropertyValues ();
    FdoPtr<FdoStringValue> owner = FdoStringValue::Create (L"SPECTRE");
	FdoPtr<FdoPropertyValue> value = FdoPropertyValue::Create (L"OWNER", owner);
    FdoPtr<FdoStringValue> aa = FdoStringValue::Create (L"AA");
	value = FdoPropertyValue::Create (L"COUNTY", aa);
	properties->Add (value);
    update->Execute ();
	transaction->Commit();
}

void RecommitTest::deleteData2 ()
{
	FdoPtr<FdoITransaction> transaction = mConnection->BeginTransaction ();
    FdoPtr<FdoIDelete> del = (FdoIDelete*)mConnection->CreateCommand (FdoCommandType_Delete);
    del->SetFeatureClassName (ArcSDETestConfig::QClassNameCity());
    del->SetFilter (L"MALE65_ > 175000");	
	del->Execute ();
	transaction->Commit ();
}

void RecommitTest::insertData2 ()
{
	FdoPtr<FdoITransaction> transaction = mConnection->BeginTransaction ();
	FdoPtr<FdoIInsert> insert = (FdoIInsert*)mConnection->CreateCommand (FdoCommandType_Insert);		
	insert->SetFeatureClassName (ArcSDETestConfig::QClassNameBeach());
    FdoPtr<FdoPropertyValueCollection> properties = insert->GetPropertyValues ();
    FdoPtr<FdoStringValue> owner = FdoStringValue::Create (L"REALESTATE 'R US");
	FdoPtr<FdoPropertyValue> value = FdoPropertyValue::Create (L"OWNER", owner);
	properties->Add (value);
    FdoPtr<FdoStringValue> aa = FdoStringValue::Create (L"AA");
	value = FdoPropertyValue::Create (L"COUNTY", aa);
	properties->Add (value);
	FdoPtr<FdoIFeatureReader> reader = insert->Execute ();
    reader->ReadNext();
    iBeachId3 = reader->GetInt32(AdjustRdbmsName(L"Id"));
    reader->Close();
    transaction->Commit();
}

