
// 
//  
//  Copyright (C) 2004-2011  Autodesk, Inc.
//  
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of version 2.1 of the GNU Lesser
//  General Public License as published by the Free Software Foundation.
//  
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  

// StampVer.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"
#include <windows.h>
#include <vector>

typedef std::pair<std::wstring, std::wstring> RECORD_ELEMENT; // name, value
typedef std::vector<RECORD_ELEMENT> CONTENT_RES;

#define STRINGNOTNULL(str) ((!(str) || !*(str)) ? (PCWSTR)L"\\0" : (str))
#define BUFSIZE 4096

class ResourceReader
{
	PUCHAR m_start;
	PUCHAR m_current;
	size_t m_size;

public:
	ResourceReader (PUCHAR start, size_t size)
		: m_start(start), m_size(size), m_current(start)
	{ 
	}
	
    void Align()
    { 
		PULONG_PTR ptr = (PULONG_PTR)&m_current;
		*ptr += 3;
		*ptr &= ~(ULONG_PTR)3;
	}
	
    void CheckPosition (size_t len)
    { 
		if ((m_current - m_start + len) > m_size)
			throw L"Invalid index to read";
	}
	
    PUCHAR Position()
    {
        return m_current;
    }

    void operator += (size_t len)
    {
        CheckPosition(len);
        m_current += len;
    }
	
    PWORD MarkLength()
    {
        PWORD ptr = (PWORD)m_current;
        m_current += sizeof(WORD);
        return ptr;
    }

	void CheckUShort (WORD val)
    {
		if (*(PWORD)m_current != val)
			throw L"Invalid ushort found!";
		m_current += sizeof(WORD);
	}

	void CheckUInt (DWORD val)
    {
		if (*(PDWORD)m_current != val)
			throw L"Invalid uint found!";
		m_current += sizeof(DWORD);
	}

	void CheckString (LPCWSTR str)
    {
		size_t len = wcslen (str);
		CheckPosition((len + 1) * sizeof(WCHAR) + sizeof(DWORD));
		for (size_t i = 0; i <= len; i++)
        {
			if (*(PWCHAR)m_current != *str && *(PWCHAR)m_current != (*str ^ 0x20))
				throw L"Invalid string found!";
			m_current += sizeof(WCHAR);
			str++;
		}
        Align();
	}

    LPCWSTR GetString()
    {
        PWCHAR buf = (PWCHAR)m_current;
        while(*buf != '\0' && (size_t)(buf - (PWCHAR)m_start) < m_size) buf++;
        if (buf == (PWCHAR)(m_start + m_size))
            throw L"Invalid string found!";
        LPCWSTR ret = (LPCWSTR)m_current;
        m_current = (PUCHAR)(buf+1);
        Align();
        return ret;
    }

	void ReadNameValue (LPCWSTR *name, LPCWSTR *value)
	{
		CheckPosition (5*sizeof(WORD));
		PWORD pStart = MarkLength();
		WORD wLength = *pStart;
		if (wLength > 1024 || wLength < 5*sizeof(WORD))
			throw L"Invalid element value!";

		CheckPosition (5*sizeof(WORD) + wLength);
		WORD wValueLength = *((PWORD)m_current);
		*this += 2;
		CheckUShort (1);

		size_t nLength = wcsnlen((LPWSTR)m_current, wLength/sizeof(WCHAR));
		if (nLength == 0 || nLength == (wLength/sizeof(WCHAR)))
			throw L"Invalid element value!";
		*name = (LPCWSTR)m_current;
		unsigned bLength = (nLength + 1)*sizeof(WCHAR);
		*this += bLength;
		Align();

		if (m_current >= (PUCHAR)pStart + *pStart)
        {
			*value = L"";
			return;
		}

		wLength -= bLength;
		nLength = wcsnlen((LPWSTR)m_current, wLength/sizeof(WCHAR));
		if (nLength == 0 || nLength == (wLength/sizeof(WCHAR)))
			throw L"Invalid element value!";
		
		*value = (LPCWSTR)m_current;
		bLength = (nLength + 1)*sizeof(WCHAR);
		m_current = (PUCHAR)pStart + *pStart;
		Align();
	}
};

class ResourceWriter 
{
	PUCHAR m_start;
	PUCHAR m_current;
	int m_size;

public:
	ResourceWriter (PUCHAR start, unsigned size)
        : m_start(start), m_size(size), m_current(start)
	{
	}

	void Align()
    { 
		PULONG_PTR ptr = (PULONG_PTR)&m_current;
		*ptr += 3;
		*ptr &= ~(ULONG_PTR)3;
	}

	size_t Size()
    {
        return m_current - m_start;
    }

	PUCHAR Position()
    {
        return m_current;
    }

	void EnsureSpace (size_t n)
    { 
		if ((size_t)(Size() + n) > m_size)
			throw L"Allocated buffer too small";
	}

	void WriteUShort (WORD val)
    {
		*(PWORD)m_current = val;
		m_current += sizeof(WORD);
	}

	void WriteUInt (DWORD val)
    {
		*(PDWORD)m_current = val;
		m_current += sizeof(DWORD);
	}

	void WriteString (LPCWSTR str, bool align = true)
    {
		if (!str)
            return;
		WORD len = (wcslen (str) + 1) * sizeof(WCHAR);
		EnsureSpace (len + sizeof(DWORD));
		memcpy (m_current, str, len);
		m_current += len;
		if (align)
			Align();
	}

    void operator += (size_t len)
    {
        EnsureSpace (len);
        m_current += len;
    }

    PWORD MarkLength()
    {
        m_current += sizeof(WORD); 
        return (PWORD)(m_current-sizeof(WORD));
    }

	void UpdateLength (PWORD pLen)
    { 
		*pLen = (WORD)(m_current - (PUCHAR)pLen);
	};

    void WriteNameValue(LPCWSTR name, LPCWSTR value)
    {
		WORD wValueLength = value ? (WORD)wcslen(value) : 0;
		if (wValueLength)
			wValueLength = (wValueLength + 1) * sizeof(WCHAR);
		WORD wNameLength = (WORD)((wcslen(name) + 1) * sizeof(WCHAR));
		EnsureSpace (wValueLength + wNameLength + 5*sizeof(WORD));

		PUCHAR buf = m_current;
		WriteUShort (0);
		WriteUShort (wValueLength);
		WriteUShort (1);
		WriteString(name);
		if (wValueLength)
			WriteString (value, false);
		*(PWORD)buf = (WORD)(m_current - buf);
		Align();
	}
};

class FileResInfo
{
private:
    // Old resFile
    DWORD m_size;
    PUCHAR m_buf;
    VS_FIXEDFILEINFO* m_fxi;
    WORD m_langFirstPart;
    WORD m_langSecondPart;
    CONTENT_RES m_content;
    std::wstring m_sFileVerTail;
    std::wstring m_sProductVerTail;
    //out resFile
    DWORD m_sizeOut;
    PUCHAR m_bufOut;
    DWORD m_dwNewFileVersionMS;
	DWORD m_dwNewFileVersionLS;
	DWORD m_dwNewProductVersionMS;
	DWORD m_dwNewProductVersionLS;
public:
    FileResInfo()
    {
        Init();
    }
    ~FileResInfo()
    {
        Clear();
    }
    void Clear()
    {
        delete[] m_buf;
        delete[] m_bufOut;
    }    
    void Init()
    {
        m_size = m_sizeOut = 0;
        m_buf = m_bufOut = NULL;
        m_fxi = NULL;
        m_langFirstPart = m_langSecondPart = 0;
        m_dwNewFileVersionMS = m_dwNewFileVersionLS = 0;
        m_dwNewProductVersionMS = m_dwNewProductVersionLS = 0;
    }

    bool GetVersionResource(LPCWSTR fileName)
    {
        if (m_size)
        {
            Clear();
            Init();
        }

        // get size of resource file
        m_size = ::GetFileVersionInfoSize (fileName, NULL);
        if (!m_size)
            return false;
        m_buf = new unsigned char[m_size];
        // could we get the resource file
        if (!::GetFileVersionInfo (fileName, NULL, m_size, m_buf))
            return false;
        // test to see if resource file is OK
        VS_FIXEDFILEINFO* pTestInfo = NULL;
        DWORD sz = 0;
        if (!::VerQueryValue (m_buf, L"\\", (LPVOID*)&pTestInfo, (PUINT)&sz))
            return false;

        ResourceReader reader((const PUCHAR)m_buf, m_size);

    // VS_VERSIONINFO
	    reader.MarkLength(); // wLength
	    reader.CheckUShort (sizeof(VS_FIXEDFILEINFO)); // wValueLength
	    reader.CheckUShort (0); // wType
	    reader.CheckString (L"VS_VERSION_INFO"); // szKey
        reader.Align(); // Padding1
        m_fxi = (VS_FIXEDFILEINFO *)reader.Position(); // Value
	    if (m_fxi->dwSignature != 0xfeef04bd || m_fxi->dwStrucVersion > 0x00010000 || m_fxi->dwStrucVersion == 0)
		    throw L"Invalid resource definition!";

	    reader += sizeof(VS_FIXEDFILEINFO);
	    reader.Align(); // Padding2

    // StringFileInfo
	    PWORD stringStart = reader.MarkLength(); // wLength
	    reader.CheckUShort(0); //wValueLength
	    reader.CheckUShort(1); //wType

        PCWSTR nameRes = reader.GetString(); // szKey
        if (0 != _wcsicmp(nameRes, L"StringFileInfo"))
        {
            if (0 != _wcsicmp(nameRes, L"VarFileInfo"))
            {
                // look for StringFileInfo
                PUCHAR stStr = (PUCHAR)memchr(reader.Position(), 'S', 0x30);
                if (!stStr)
                    throw L"Invalid resource file!";
                reader += (size_t)(stStr - reader.Position() - 3*sizeof(WORD));
    		    
                // try again
                stringStart = reader.MarkLength(); // wLength
		        reader.CheckUShort(0); // wValueLength
		        reader.CheckUShort(1); // wType
		        reader.CheckString(L"StringFileInfo"); // szKey
            }
            else
                throw L"Invalid resource file!";
        }

	    reader.MarkLength(); // wLength
	    reader.CheckUShort(0); // wValueLength
	    reader.CheckUShort(1); // wType

        // Lang : szKey
        reader.CheckPosition(9 * sizeof(WCHAR));
        LPCWSTR strLang = reader.GetString();
	    if (wcslen(strLang) != 8)
		    throw L"Invalid language string!";

        WCHAR sigLang[9];
        wcscpy_s(sigLang, 9, strLang);

        DWORD val = 0;
        swscanf_s(&sigLang[4], L"%x", &val);
        m_langSecondPart = (WORD)val;
        sigLang[4] = L'\0';
        val = 0;
        swscanf_s(sigLang, L"%x", &val);
        m_langFirstPart = (WORD)val;
	    reader.Align(); // Padding

	    // Get all resource strings
	    do {
		    PCWSTR name, value;
            reader.ReadNameValue(&name, &value);
            // FileVersion will be rebuilt so do not add it to list
            if (_wcsicmp(name, L"FileVersion") == 0)
            {
                PCWSTR tail = GetTailVersion(value);
                if (tail && *tail)
                    m_sFileVerTail = tail;
            }
            // ProductVersion will be rebuilt so do not add it to list
            else if (_wcsicmp(name, L"ProductVersion") == 0)
            {
                PCWSTR tail = GetTailVersion(value);
                if (tail && *tail)
                    m_sProductVerTail = tail;
            }
            else
                m_content.push_back(std::make_pair(name, STRINGNOTNULL(value)));
	    } while(reader.Position() < ((PUCHAR)stringStart + *stringStart));
    }

    bool UpdateVersionResource (LPCWSTR fname)
    {
        if (!m_size)
            return false;
        BuildNewVersionResource();

        bool retVal = true;
	    HANDLE rhandle = ::BeginUpdateResource (fname, false);
	    if (!rhandle) {
		    printf("Error opening file for update resources, err=%d\n", GetLastError());
		    return false;
	    }
	    if (!::UpdateResource(rhandle, (LPCWSTR)(ULONG_PTR)RT_VERSION, (LPCWSTR)(ULONG_PTR)MAKEINTRESOURCE(VS_VERSION_INFO),
            m_langFirstPart, (LPVOID)m_bufOut, m_sizeOut))
        {
            printf("UpdateResource LastError = %d\n", GetLastError());
            retVal = false;
        }
	    
        if (!::EndUpdateResource(rhandle, !retVal))
        {
		    printf("EndUpdateResource LastError = %d\n", GetLastError());
            retVal = false;
        }
        return retVal;
    }

    void SetFileVersion (LPCWSTR arg)
    {
        if (!arg || !*arg)
            return;
        LPCWSTR tail = NULL;
        ParseVersionString(arg, m_dwNewFileVersionMS, m_dwNewFileVersionLS, &tail);
        if (tail && *tail)
            m_sFileVerTail = tail;
        else
            m_sFileVerTail.clear();
    }

    void SetProductVersion (LPCWSTR arg)
    {
        if (!arg || !*arg)
            return;
        LPCWSTR tail = NULL;
        ParseVersionString(arg, m_dwNewProductVersionMS, m_dwNewProductVersionLS, &tail);
        if (tail && *tail)
            m_sProductVerTail = tail;
        else
            m_sProductVerTail.clear();
    }
    
    void SetLegalCopyright (LPCWSTR arg)
    {
        bool foundItem = false;
        for (CONTENT_RES::iterator it = m_content.begin(); it < m_content.end(); it++)
        {
            if (0 == _wcsicmp (L"LegalCopyright", it->first.c_str()))
            {
                it->second = STRINGNOTNULL(arg);
                foundItem = true;
		    }
	    }
        if (!foundItem)
            m_content.push_back(std::make_pair(L"LegalCopyright", STRINGNOTNULL(arg)));
    }

private:
    static void ParseVersionString(LPCWSTR arg, DWORD& versionMS, DWORD& versionLS, LPCWSTR *tail)
    {
	    unsigned n1=0,n2=0,n3=0,n4=0;
        versionMS = versionLS = 0;
	    // less than 4 numbers are minor
	    switch (swscanf_s( arg, L"%u.%u.%u.%u", &n1, &n2, &n3, &n4))
        {
		    case 1:
                versionLS = MAKELONG (n1, 0);
			    break;
		    case 2:
                versionLS = MAKELONG (n2, n1);
			    break;
		    case 3:
                versionLS = MAKELONG (n3, n2);
                versionMS = MAKELONG (n1, 0);
			    break;
		    case 4:
                versionLS = MAKELONG (n4, n3);
                versionMS = MAKELONG (n2, n1);
			    break;
		    default:
			    throw L"Error parsing version string!";
	    }
	    LPCWSTR ltail = _tcschr(arg, L' ');
	    if (ltail)
        {
		    while( *ltail == L' ') ltail++;
		    *tail = (*ltail) ? ltail : NULL;
	    }
    }

    static LPCWSTR GetTailVersion(LPCWSTR version)
    {
        if (version != NULL && *version != L'\0')
        {
            PCWSTR tail = wcschr(version, L' ');
            if (tail)
            {
                while(*tail == L' ') tail++;
                return (*tail) ? tail : L"";
            }
        }
        return L"";
    }

    void BuildNewVersionResource()
    {
        m_bufOut = new unsigned char[m_size*2];
        m_sizeOut = m_size*2;

	    WCHAR temp[256];
        ResourceWriter writer(m_bufOut, m_sizeOut);

    // VS_VERSIONINFO{
        PWORD pTotalLen = writer.MarkLength(); // wLength
        writer.WriteUShort(sizeof(VS_FIXEDFILEINFO)); // wValueLength
        writer.WriteUShort(0); // wType
        writer.WriteString(L"VS_VERSION_INFO"); // szKey
        writer.Align(); // Padding1

	    VS_FIXEDFILEINFO *fxi = (VS_FIXEDFILEINFO *)writer.Position(); // Value
        memcpy(fxi, m_fxi, sizeof(VS_FIXEDFILEINFO));
    	
        // update FileVersion in case is needed
        if (m_dwNewFileVersionMS != 0 || m_dwNewFileVersionLS != 0)
        {
            fxi->dwFileVersionMS = m_dwNewFileVersionMS;
	        fxi->dwFileVersionLS = m_dwNewFileVersionLS;
        }
        // update ProductVersion in case is needed
        if (m_dwNewProductVersionMS != 0 || m_dwNewProductVersionLS != 0)
        {
	        fxi->dwProductVersionMS = m_dwNewProductVersionMS;
	        fxi->dwProductVersionLS = m_dwNewProductVersionLS;
        }
        writer += sizeof(VS_FIXEDFILEINFO);
        writer.Align(); // Padding2

    //struct StringFileInfo{
	    PWORD stringStart = writer.MarkLength(); // wLength
	    writer.WriteUShort(0); // wValueLength
	    writer.WriteUShort(1); // wType
	    writer.WriteString(L"StringFileInfo"); // szKey
        writer.Align(); // Padding

	    PWORD stringTableStart = writer.MarkLength(); // wLength
	    writer.WriteUShort(0); // wValueLength
	    writer.WriteUShort(1); // wType
        swprintf_s(temp, 256, L"%0.4X%0.4X", m_langFirstPart, m_langSecondPart);
	    writer.WriteString(temp); // szKey
        writer.Align(); // Padding

        if (m_sFileVerTail.size())
        {
            swprintf_s(temp, 256, L"%u.%u.%u.%u %ls", HIWORD(fxi->dwFileVersionMS), LOWORD(fxi->dwFileVersionMS),
                HIWORD(fxi->dwFileVersionLS), LOWORD(fxi->dwFileVersionLS), m_sFileVerTail.c_str());
        }
        else
        {
            swprintf_s(temp, 256, L"%u.%u.%u.%u", HIWORD(fxi->dwFileVersionMS), LOWORD(fxi->dwFileVersionMS),
                HIWORD(fxi->dwFileVersionLS), LOWORD(fxi->dwFileVersionLS));
        }
        writer.WriteNameValue (L"FileVersion", temp);

        if (m_sProductVerTail.size())
        {
            swprintf_s(temp, 256, L"%u.%u.%u.%u %ls", HIWORD(fxi->dwProductVersionMS), LOWORD(fxi->dwProductVersionMS),
                HIWORD(fxi->dwProductVersionLS), LOWORD(fxi->dwProductVersionLS), m_sProductVerTail.c_str());
        }
        else
        {
            swprintf_s(temp, 256, L"%u.%u.%u.%u", HIWORD(fxi->dwProductVersionMS), LOWORD(fxi->dwProductVersionMS),
                HIWORD(fxi->dwProductVersionLS), LOWORD(fxi->dwProductVersionLS));
        }
	    writer.WriteNameValue (L"ProductVersion", temp);

	    // Strings
        for (CONTENT_RES::iterator it = m_content.begin(); it < m_content.end(); it++)
        {
            if (it->first.size())
            {
                writer.WriteNameValue (it->first.c_str(), it->second.c_str());
                if (0 == _wcsicmp (L"SpecialBuild", it->first.c_str()))
				    fxi->dwFileFlags |= VS_FF_SPECIALBUILD;
			    if (0 == _wcsicmp(L"PrivateBuild", it->first.c_str()))
				    fxi->dwFileFlags |= VS_FF_PRIVATEBUILD;
		    }
	    }

        writer.Align(); // Padding
        writer.UpdateLength (stringTableStart);
	    writer.UpdateLength (stringStart);

    // VarFileInfo{
        PWORD varInfoStart = writer.MarkLength(); // wLength
	    writer.WriteUShort(0); // wValueLength
	    writer.WriteUShort(1); // wType
	    writer.WriteString(L"VarFileInfo"); // szKey
        writer.Align(); // Padding

    // Var{
	    PWORD varStart = writer.MarkLength(); // wLength
	    writer.WriteUShort(0x04); // wValueLength
	    writer.WriteUShort(0x00); // wType
	    writer.WriteString(L"Translation"); // szKey;
        writer.Align(); // Padding
	    writer.WriteUShort (m_langFirstPart); // Value
	    writer.WriteUShort (m_langSecondPart);

	    writer.UpdateLength (varStart);
	    writer.UpdateLength (varInfoStart);
	    writer.UpdateLength (pTotalLen);

        m_sizeOut = writer.Size();
    }
};

//-f "3.6.0.44"
//-p "3.6.0.0"
//-c "Copyright (C) 2007-2011"
int _tmain(int argc, _TCHAR* argv[])
{
    LPCWSTR use = L"\nStampVer.exe\n\tHelp:           -h\n\tFileVersion:    -f\"value\" ; e.g. \"3.6.0.44\"\n\tProductVersion: -p\"value\" ; e.g. \"3.6.0.0\"\n\tCopyright:      -c\"value\" ; e.g. \"Copyright (C) 2007-2011\"\n\tFileInfo:       -v\"File name\"; parameter ignored!!\n\tFileName:       \"File path\"\n";
    if (argc <= 2)
    {
        wprintf(use);
        return 0;
    }
    std::wstring fname;
    std::wstring fileVer;
    std::wstring prodVer;
    std::wstring copyright;
    bool err = false;
    for (int idx = 1; idx < argc && !err; idx++)
    {
        if (*argv[idx] == L'-')
        {
            if ((idx+1) >= argc)
            {
                err = true;
                break;
            }
            switch(*(argv[idx]+1))
            {
            case L'v':
                // source file with format not supported yet
                continue;
            case L'f':
                fileVer = argv[idx]+2;
                continue;
            case L'p':
                prodVer = argv[idx]+2;
                continue;
            case L'c':
                copyright = argv[idx]+2;
                continue;
            default:
                wprintf(use);
                return 1;
            }
        }
        if ((idx+1) == argc) // last itme must be the file
        {
            if (*argv[idx] == '\"')
                fname = std::wstring(argv[idx]+1, wcslen(argv[idx])-2);
            else
                fname = argv[idx];
        }
        else
        {
            wprintf(use);
            return 1;
        }
    }
    if (fname.empty() || (fileVer.empty() && prodVer.empty() && copyright.empty()))
    {
        wprintf(use);
        return 1;
    }

    try
    {
        FileResInfo retVal;
        if (retVal.GetVersionResource(fname.c_str()))
        {
            if (fileVer.size())
                retVal.SetFileVersion(fileVer.c_str());
            if (prodVer.size())
                retVal.SetProductVersion(prodVer.c_str());
            if (copyright.size())
                retVal.SetLegalCopyright(copyright.c_str());

            retVal.UpdateVersionResource(fname.c_str());
            wprintf(L"\nPatch successful!\n");
            return 0;
        }
        return 1;
    }
    catch(wchar_t* err)
    {
        wprintf(L"\nError: '%ls'\n", err);
        return 1;
    }
    catch(...)
    {
        wprintf(L"\nError: 'unknown'\n");
        return 1;
    }
}

