/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */


#include "stdafx.h"
#include <Sm/Lp/ObjectPropertyClass.h>
#include <Sm/Lp/Schema.h>
#include <Sm/Lp/PropertyMappingSingle.h>
#include <Sm/Error.h>

FdoSmLpObjectPropertyClass::FdoSmLpObjectPropertyClass(
	FdoSmLpObjectPropertyDefinition* pParent,
	FdoSmLpClassDefinition* pParentType,
	FdoSmLpPropertyMappingDefinition* pMapping,
	FdoSmLpPropertyMappingType mappingType,
    FdoPhysicalClassMapping* pOverrides
) :
	FdoSmLpClass(	MakeName(pParent),
						L"Autogenerated Object Property Class",
						pMapping,
						(FdoSmLpClassDefinition*) NULL,
						FdoSmOvTableMappingType_ConcreteTable,
						pParent->GetContainingDbObjectName(),
						pParent->GetContainingDbObject(),
						pParent->GetElementState()
	),
	FdoSmLpClassBase(	MakeName(pParent),
						L"Autogenerated Object Property Class",
						pMapping,
						(FdoSmLpClassDefinition*) NULL,
						FdoSmOvTableMappingType_ConcreteTable,
						pParent->GetContainingDbObjectName(),
						pParent->GetContainingDbObject(),
						pParent->GetElementState(),
                        pParent->GetIsFromFdo()
	),
	mpDependency(pParent->RefDependency()),
	mpObjectProperty( pParent ),
    mMappingType( mappingType )
{
    SetElementState(pParent->GetElementState());

    mpContainingClass = pParent->RefParentClass();

    mSourceProperties = new FdoSmLpDataPropertyDefinitionCollection();
    mTargetProperties = new FdoSmLpDataPropertyDefinitionCollection();

    InitTable( pParent );
}

FdoSmLpObjectPropertyClass::FdoSmLpObjectPropertyClass(
	FdoSmLpObjectPropertyClassP pBase,
	FdoSmLpObjectPropertyDefinition* pParent,
	FdoSmLpClassDefinition* pParentType,
	FdoSmLpPropertyMappingDefinition* pMapping,
	FdoSmLpPropertyMappingType mappingType,
    FdoPhysicalClassMapping* pOverrides
) :
	FdoSmLpClass(	MakeName(pParent),
						L"Autogenerated Object Property Class",
						pMapping,
						pBase.p->SmartCast<FdoSmLpClassDefinition>(),
						FdoSmOvTableMappingType_ConcreteTable,
						pParent->GetContainingDbObjectName(),
						pParent->GetContainingDbObject(),
						pParent->GetElementState()
	),
	FdoSmLpClassBase(	MakeName(pParent),
						L"Autogenerated Object Property Class",
						pMapping,
						pBase.p->SmartCast<FdoSmLpClassDefinition>(),
						FdoSmOvTableMappingType_ConcreteTable,
						pParent->GetContainingDbObjectName(),
						pParent->GetContainingDbObject(),
						pParent->GetElementState(),
                        pParent->GetIsFromFdo()
	),
	mpDependency(pParent->RefDependency()),
	mpObjectProperty( pParent ),
    mMappingType( mappingType )
{
    SetElementState(pParent->GetElementState());

	mpContainingClass = pParent->RefParentClass();

    mSourceProperties = new FdoSmLpDataPropertyDefinitionCollection();
    mTargetProperties = new FdoSmLpDataPropertyDefinitionCollection();

    InitTable( pParent );
}

FdoSmLpObjectPropertyClass::~FdoSmLpObjectPropertyClass(void)
{
}

const FdoSmLpDataPropertyDefinition* FdoSmLpObjectPropertyClass::RefLocalIdProperty() const
{
	return (FdoSmLpDataPropertyDefinition*) mpLocalIdProperty;
}

FdoSmLpDataPropertyP FdoSmLpObjectPropertyClass::GetLocalIdProperty()
{
	return mpLocalIdProperty;
}

const FdoSmLpDataPropertyDefinitionCollection* FdoSmLpObjectPropertyClass::RefSourceProperties() const
{
	return mSourceProperties;
}

const FdoSmLpDataPropertyDefinitionCollection* FdoSmLpObjectPropertyClass::RefTargetProperties() const
{
	return mTargetProperties;
}

const FdoSmLpObjectPropertyDefinition* FdoSmLpObjectPropertyClass::RefObjectProperty() const
{
	return( mpObjectProperty );
}

FdoSmLpObjectPropertyP FdoSmLpObjectPropertyClass::GetObjectProperty()
{
    return ( FDO_SAFE_ADDREF(mpObjectProperty) );
}

const FdoSmLpClassDefinition* FdoSmLpObjectPropertyClass::RefContainingClass() const
{
	return( mpContainingClass );
}

void FdoSmLpObjectPropertyClass::Commit()
{
	// This is an temporary wrapper class so there is no F_ClassDefinition
	// row to modify. 

	for ( int i = 0; i < RefProperties()->GetCount(); i++ ) {
        FdoSmLpPropertyP pProp = GetProperties()->GetItem(i);

		FdoSmLpObjectPropertyDefinition* pObjProp = 
            dynamic_cast<FdoSmLpObjectPropertyDefinition*>( (FdoSmLpPropertyDefinition*) pProp  );

		FdoSmLpSimplePropertyDefinition* pSimpleProp = 
            dynamic_cast<FdoSmLpSimplePropertyDefinition*>( (FdoSmLpPropertyDefinition*) pProp  );

        // To save clutter, only certain object property sub-properties are persisted
        // in the metaschema:
        //
        //      Object Properties: A nested Object Property needs an 
        //      F_AttributeDefinition row to indicate the table that contains the 
        //      object property instances.
        //
        //      When the object property table mapping is Single, then all sub-properties
        //      need to be written.
        //
        //      Data and Geometric Properties: When isfixedcolumn or iscolumncreator
        ///     do not have the default values.

		if ( pObjProp || 
            (mMappingType == FdoSmLpPropertyMappingType_Single) ||
            ( pSimpleProp && (pSimpleProp->GetIsFixedColumn() || !pSimpleProp->GetIsColumnCreator()) ) 
        )
			pProp->Commit();
	}
}

void FdoSmLpObjectPropertyClass::WriteDb( FdoSmPhPropertyWriterP pWriter ) const
{
	pWriter->SetIsFixedColumn( GetIsFixedDbObject() );
}

FdoStringP FdoSmLpObjectPropertyClass::MakeName( const FdoSmLpObjectPropertyDefinition* pParent )
{
	// Object property class name is a concatenation of the object property's containing class
	// and the object property name.
	return( FdoStringP(pParent->RefParentClass()->GetName()) + L"." + pParent->GetName() );
}

void FdoSmLpObjectPropertyClass::InitNestedProperties (
	const FdoSmLpObjectPropertyDefinition* pParent, 
	const FdoSmLpClassDefinition* pParentType,
	FdoSmLpPropertyMappingType mappingType 
)
{

	if ( !mpContainingClass  ) 
		return;

	// Set up the prefix for finding all nested properties that belong
	// to this object property. 
	// Nested property names are not qualified by the outermost containing
	// class so strip out the outermost containing class name.
	FdoStringP propPrefix = FdoStringP( GetName() ).Right(L"." ) + L".";		

    FdoSmLpPropertiesP pProps =
		GetNestedProperties();

    FdoSmLpPropertiesP pNestedProps = 
		((FdoSmLpClassDefinition*) mpContainingClass)->GetNestedProperties();

	// Find the nested properties for this class in the nested properties
	// for the object property containing class. 
	for ( int i = 0; i < pNestedProps->GetCount(); i++ ) {
		FdoSmLpPropertyP pNestedProp =
			pNestedProps->GetItem(i);

		// Add only nested properties with matching prefixes.
		if ( FdoStringP(pNestedProp->GetName()).Left( propPrefix ).GetLength() == 0 ) {
			pProps->Add(pNestedProp);	
		}
	}
}

void FdoSmLpObjectPropertyClass::InitNestedProperties (
	const FdoSmLpObjectPropertyClass* pBase
)
{
	FdoSmLpPropertiesP pProps =
		GetNestedProperties();

	// Copy nested properties from base class.
	for ( int i = 0; i < pBase->RefProperties()->GetCount(); i++ ) {
        FdoSmLpPropertyP pProp = ((FdoSmLpObjectPropertyClass*)pBase)->GetProperties()->GetItem(i);
		pProps->Add( pProp );
	}
}

void FdoSmLpObjectPropertyClass::InitLocalIdProperty (
	const FdoSmLpObjectPropertyDefinition* pParent, 
	const FdoSmLpClassDefinition* pParentType,
	FdoSmLpPropertyMappingType mappingType 
)
{
	// Ref local id property from the Object Property. It will be set
	// when the Object Property was created from an FDO element.
	FdoStringP localIdProp = pParent->GetIdentityPropertyName();

	if ( localIdProp.GetLength() > 0 ) {
		// Make sure the local id is in the property list.
		mpLocalIdProperty = GetProperties()->FindItem(localIdProp)->SmartCast<FdoSmLpDataPropertyDefinition>(true);

		if ( pParentType && !mpLocalIdProperty && (GetElementState() != FdoSchemaElementState_Deleted) ) 
			AddOrderByMissingError( pParent, pParentType, localIdProp );
	}
	else {

		// If no localId on Object Property then calculated it from the 
		// attribute dependency, if any.
		if ( mpDependency ) {
			// Ref the column name from the dependency
			FdoString* localIdColumn = mpDependency->GetIdentityColumn();

			if ( localIdColumn && (wcslen(localIdColumn) > 0) ) {
				// Find the corresponding property.
				mpLocalIdProperty = FDO_SAFE_ADDREF(
					(FdoSmLpDataPropertyDefinition*) FdoSmLpDataPropertyDefinitionCollection::ColName2Property(
						GetProperties(),	
						localIdColumn
					)
                );

				// Error if this object property has a type and the 
				// order column has no property.
				if ( pParentType && !mpLocalIdProperty && (GetElementState() != FdoSchemaElementState_Deleted) ) 
					AddOrderByMissingError( pParent, pParentType, localIdColumn );
			}
		}
	}
}

void FdoSmLpObjectPropertyClass::InitLocalIdProperty (
	const FdoSmLpObjectPropertyClass* pBase
)
{
	const FdoSmLpDataPropertyDefinition* pBaseProp = pBase->RefLocalIdProperty();

	// Copy local id from base class.
	if ( pBaseProp ) {
		mpLocalIdProperty =
			GetProperties()->GetItem(pBaseProp->GetName())->SmartCast<FdoSmLpDataPropertyDefinition>(true);
	}
}

void FdoSmLpObjectPropertyClass::InitIdProperties (
	const FdoSmLpObjectPropertyDefinition* pParent, 
	const FdoSmLpClassDefinition* pParentType,
	FdoSmLpPropertyMappingDefinition* pMapping,
	FdoSmLpPropertyMappingType mappingType 
)
{

	if ( !pParentType ) 
		return;

	const FdoSmLpDataPropertyDefinitionCollection* pIdProps =
		pParentType->RefIdentityProperties();

	FdoSmLpDataPropertiesP pProps =
		GetIdentityProperties();

	if ( pIdProps->GetCount() > 0 ) {
		// Object Property's class has ID properties so make them
		// the ID properties of this class.
		for ( int i = 0; i < pIdProps->GetCount(); i++ ) {
			FdoSmLpPropertyP pProp =
				GetProperties()->GetItem( pIdProps->RefItem(i)->GetName() );
			FdoSmLpDataPropertyP pDataProp = 
                pProp.p->SmartCast<FdoSmLpDataPropertyDefinition>();

			if ( pDataProp )
				pProps->Add(pDataProp);

			if ( !pDataProp && (GetElementState() != FdoSchemaElementState_Deleted) ) 
				AddIdPropNotFoundError( pParent, pIdProps->RefItem(i) );
		}
	}
	else {
		// No ID properties, so generate them from foreign properties
		// and local id property.

		// First part of ID consists of foreign properties

		for ( int i = 0; i < mTargetProperties->GetCount(); i++ ) {
			pProps->Add( FdoSmLpDataPropertyP(mTargetProperties->GetItem(i)) );
		}

		if ( mpLocalIdProperty ) {
			// Add local id property if present. 
			pProps->Add(mpLocalIdProperty);
		}
		else {
			if ( pParent->GetObjectType() != FdoObjectType_Value ) {

				// A collection object property must have a local id property
				// when the type class has no id properties. Otherwise, individual
				// members cannot be distinguished.
				if ( GetElementState() != FdoSchemaElementState_Deleted )
					AddCollectionIDError( pParent );
			}
		}
	}
}

void FdoSmLpObjectPropertyClass::InitIdProperties (
	const FdoSmLpObjectPropertyClass* pBase
)
{
	// Copy identity properties from base class
	for ( int i = 0; i < pBase->RefIdentityProperties()->GetCount(); i++ ) {
		const FdoSmLpDataPropertyDefinition* pBaseProp = 
			pBase->RefIdentityProperties()->RefItem(i);

		FdoSmLpDataPropertyP pProp =
			GetProperties()->GetItem(pBaseProp->GetName())->SmartCast<FdoSmLpDataPropertyDefinition>(true);

		if ( pProp ) 
			GetIdentityProperties()->Add( pProp );
	}
}

void FdoSmLpObjectPropertyClass::InitTable(
    const FdoSmLpObjectPropertyDefinition* pParent
)
{
    // The fixed table and table creator settings are calculated by the 
    // Object Property.
    SetIsFixedDbObject( pParent->GetIsFixedDbObject() );
    SetIsDbObjectCreator( pParent->GetIsDbObjectCreator() );

    // Table location settings are inherited from contain object
    // property's containing table. A class and its object properties
    // must always be in the same oracle schema and tablespace.
    SetOwner( pParent->RefDefiningClass()->GetOwner() );
    SetDatabase( pParent->RefDefiningClass()->GetDatabase() );
//TODO    SetTablespace( pParent->RefDefiningClass()->RefTablespace() );
}

const FdoSmPhDependency* FdoSmLpObjectPropertyClass::RefDependency() const
{
    return mpDependency;
}

FdoSmLpDataPropertiesP FdoSmLpObjectPropertyClass::GetSourceProperties()
{
    return mSourceProperties;
}

FdoSmLpDataPropertiesP FdoSmLpObjectPropertyClass::GetTargetProperties()
{
    return mTargetProperties;
}

bool FdoSmLpObjectPropertyClass::IsTableCreator() const
{
	const FdoSmPhMgr*       pPhysical = RefLogicalPhysicalSchema()->RefPhysicalSchema();
    bool                                isTableCreator = false;

    // RefIsTableCreator() tells us whether Schema Manager created the object property table.
    // However, when RefIsTableCreator() is true, the table is dropped when
    // the object property is deleted. The following does an extra check to ensure that 
    // the object property table and base property tables are different. If they are the 
    // same, false is always returned to ensure that the table does not get
    // dropped.
	if ( pPhysical->RefDbObject(GetDbObjectName()) && GetIsDbObjectCreator() ) {
		// Find the source.

        FdoStringP baseTableName = RefSrcClass() ? RefSrcClass()->GetDbObjectName() : L"";

		// Delete the associated table only if it is not the table for the
		// base or source class, since the base or source class still needs
		// this table.
        // TODO: create an error when it is the base or source class table.
        // A class should never be the creator of a base class table.
		if ( baseTableName.ICompare(GetDbObjectName()) != 0 ) 
            isTableCreator = true;
	}

    return isTableCreator;
}


void FdoSmLpObjectPropertyClass::PostFinalize()
{
	// Set up the link between this class table and the table for the
	// object property's containing class. Do this only if this class
	// is error free, this class has a table, and the containing class
	// has a table.

	if ( (RefErrors()->GetCount() == 0) && (mSourceProperties->GetCount() > 0) &&
		 ( mSourceProperties->GetCount() == mTargetProperties->GetCount()) ) {

		const FdoSmLpDbObject* pTargetTable = mpContainingClass->RefDbObject();

        if ( pTargetTable ) {
			FdoSmLpDbObjectP pTable = GetDbObject();

			if ( pTable ) {
				// Make the containing class table the target of this table.
				pTable->SetTargetDbObject( FDO_SAFE_ADDREF((FdoSmLpDbObject*) pTargetTable) );

				bool bColumnsPresent = true;

				// Make sure all source and target properties have columns.
				for ( int i = 0; i < mSourceProperties->GetCount(); i++ ) {
					if ( (!mTargetProperties->RefItem(i)->RefColumn()) ||
						(!mSourceProperties->RefItem(i)->RefColumn()) ) {
						bColumnsPresent = false;
						break;
					}
				}

				if ( bColumnsPresent ) {
					// All have columns so proceed.
					for ( int i = 0; i < mSourceProperties->GetCount(); i++ ) {
						// We're going from object property table to containing class table
						// so reverse the source and target columns. 
						pTable->AddSourceColumn( FdoSmLpDataPropertyP(mTargetProperties->GetItem(i))->GetColumn() );
						pTable->AddTargetColumn( FdoSmLpDataPropertyP(mSourceProperties->GetItem(i))->GetColumn() );
					}
				}
			}
		}
	}
}

void FdoSmLpObjectPropertyClass::AddIdPropNotFoundError(
	const FdoSmLpObjectPropertyDefinition* pParent,
	const FdoSmLpDataPropertyDefinition* pProp
)
{
	GetErrors()->Add( FdoSmErrorType_Other, 
        FdoSchemaException::Create(
            FdoSmError::NLSGetMessage(
			    FDO_NLSID(FDOSM_175),
    			(FdoString*) pProp->GetQName(), 
	    		(FdoString*) pParent->GetQName() 
            )
		)
	);
}

void FdoSmLpObjectPropertyClass::AddJoinMismatchError(
	const FdoSmLpObjectPropertyDefinition* pParent
)
{
	GetErrors()->Add( FdoSmErrorType_Other, 
        FdoSchemaException::Create(
            FdoSmError::NLSGetMessage(
			    FDO_NLSID(FDOSM_176),
    			(FdoString*) pParent->GetQName()
            )
		)
	);
}

void FdoSmLpObjectPropertyClass::AddSourcePropertyMissingError
(
	const FdoSmLpObjectPropertyDefinition* pParent,
	FdoStringP sourceColumn
)
{
	GetErrors()->Add( FdoSmErrorType_Other, 
        FdoSchemaException::Create(
            FdoSmError::NLSGetMessage(
			    FDO_NLSID(FDOSM_177),
			    (FdoString*) sourceColumn,
			    pParent->GetName(), 
			    (FdoString*) pParent->RefParentClass()->GetQName()
            )
		)
	);
}

void FdoSmLpObjectPropertyClass::AddTargetPropertyMissingError
(
	const FdoSmLpObjectPropertyDefinition* pParent,
	FdoStringP targetColumn
)
{
	GetErrors()->Add( FdoSmErrorType_BaseClassLoop, 
        FdoSchemaException::Create(
            FdoSmError::NLSGetMessage(
			    FDO_NLSID(FDOSM_178),
			    (FdoString*) targetColumn,
			    pParent->GetName(), 
			    (FdoString*) pParent->RefParentClass()->GetQName()
            )
		)
	);
}

void FdoSmLpObjectPropertyClass::AddTargetColumnMissingError
(
	const FdoSmLpObjectPropertyDefinition* pParent,
	const FdoSmLpDataPropertyDefinition* pProp
)
{
	GetErrors()->Add( FdoSmErrorType_ColumnMissing, 
        FdoSchemaException::Create(
            FdoSmError::NLSGetMessage(
			    FDO_NLSID(FDOSM_179),
			    pParent->GetContainingDbObjectName(),
			    pProp->GetColumnName(),
			    pProp->GetName(),
			    (FdoString*) pParent->GetQName()
            )
		)
	);
}

void FdoSmLpObjectPropertyClass::AddOrderByMissingError
(
	const FdoSmLpObjectPropertyDefinition* pParent,
	const FdoSmLpClassDefinition* pParentType,
	FdoString* orderByColumn
)
{
	GetErrors()->Add( FdoSmErrorType_ColumnMissing, 
        FdoSchemaException::Create(
            FdoSmError::NLSGetMessage(
			    FDO_NLSID(FDOSM_180),
			    orderByColumn,
			    (FdoString*) pParent->GetQName(), 
			    (FdoString*) pParentType->GetQName()
            )
		)
	);
}

void FdoSmLpObjectPropertyClass::AddCollectionIDError
(
	const FdoSmLpObjectPropertyDefinition* pParent
)
{
	GetErrors()->Add( FdoSmErrorType_Other, 
        FdoSchemaException::Create(
            FdoSmError::NLSGetMessage(
			    FDO_NLSID(FDOSM_181),
    			(FdoString*) pParent->GetQName()
            )
		)
	);
}


void FdoSmLpObjectPropertyClass::AddTargetConflictError
(
	const FdoSmLpObjectPropertyDefinition* pParent,
	const FdoSmLpDataPropertyDefinition* pProp,
	const FdoSmLpClassDefinition* pParentType
)
{
	GetErrors()->Add( FdoSmErrorType_Other, 
        FdoSchemaException::Create(
            FdoSmError::NLSGetMessage(
			    FDO_NLSID(FDOSM_258),
			    (FdoString*) pParent->GetQName(),
			    (FdoString*) pProp->GetQName(),
			    (FdoString*) pParentType->GetQName()
            )
		)
	);
}


FdoStringP FdoSmLpObjectPropertyClass::GetQName() const
{
    FdoSmLpObjectPropertyP objProp = ((FdoSmLpObjectPropertyClass*)this)->GetObjectProperty();
    FdoPtr<FdoSmLpPropertyDefinition> topProp = objProp->GetTopProperty();
    const FdoSmSchemaElement* topPropClass = topProp->GetParent();
    const FdoSmSchemaElement* topPropSchema = topPropClass->GetParent();
    
	return( topPropSchema->GetQName() + L":" + GetName() );
}
