#ifndef FDOSMLPOBJECTPROPERTYDEFINITION_H
#define FDOSMLPOBJECTPROPERTYDEFINITION_H		1
/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef _WIN32
#pragma once
#endif

#include <Sm/Ph/Dependency.h>
#include <Sm/Lp/DataPropertyDefinition.h>
#include <Sm/Lp/PropertyMappingDefinition.h>

class FdoSmLpObjectPropertyClass;

// ObjectPropertyDefinition derives from PropertyDefinition and represents
// containment of a named instance of a class within another class. The
// contained class must already be defined in the feature schema and cannot be
// abstract.
class FdoSmLpObjectPropertyDefinition : public FdoSmLpPropertyDefinition
{
  public:

    /// Get the type of property ( object, data, geometric, etc. ) that
    /// this object represents.
	virtual FdoPropertyType GetPropertyType() const;

    /// Gets the object type ( value, collection, ordered collection ).
	FdoObjectType GetObjectType() const;

    /// Gets the name of the class that defines the type of this
    /// property.
	FdoString* GetFeatureClassName() const;

    /// Gets a reference to the ClassObject that defines the type of this
    /// property. NULL if no class is defined or the class is not in the
    /// current database. Call GetFeatureClassName() to see if the class is 
    /// defined.
	const FdoSmLpClassDefinition* RefClass() const;

    /// Gets the ordering ( ascending or descending ) when the 
    /// ObjectType is set to OrderedCollection
    FdoOrderType GetOrderType() const;

    /// Gets the name of the Local Id property. Call 
    /// GetIdentityProperty() to get the property.
	FdoStringP GetIdentityPropertyName() const;

    /// Gets a reference to a DataProperty to use for identifying individual
    /// members of a collection type object property. The
    /// DataProperty must belong to the ClassObject that defines the type of
    /// this property.
    /// NULL if this property is not defined or doesn't exist in the
    /// database. Call GetIdentityPropertyName() to see if it is defined.
    const FdoSmLpDataPropertyDefinition* RefIdentityProperty() const;

    /// Get the Mapping Definition that defines how this Object
    /// Property is stored in the database.
    /// Return value is always valid (never NULL).
	const FdoSmLpPropertyMappingDefinition* RefMappingDefinition() const;

    /// Get the dependency for this object property as defined by MetaData.
    /// This function is not for general use since it doesn't return the autogenerated
    /// dependency when there isn't an explicit one in the database.
	const FdoSmPhDependency* RefDependency() const
	{
		return mpDependency;
	}

    /// Get the target class for this relation mapping.
	const FdoSmLpClassDefinition* RefTargetClass() const;

    /// returns the class that contains the primary key properties that reference
    /// this object property's foreign key properties.
    const FdoSmLpClassDefinition* RefPkClass() const;
    FdoPtr<FdoSmLpClassDefinition> GetPkClass();
	
    /// When property mapping is Concrete, this function returns the 
    /// default name for the containing table, before it was adjusted.
    FdoStringP GetDefaultDbObjectName() const;

    /// When property mapping is Concrete and the object property
    /// table is a foreign table, this returns the foreign table name.
    FdoStringP GetRootDbObjectName() const;

    /// Returns true if the object property table name was fixed by Schema Overrides
    bool GetIsFixedDbObject() const
    {
        return mbFixedDbObject;
    }

    /// Returns true if the object property table name was created by the Schema Manager
    bool GetIsDbObjectCreator() const
    {
        return mbDbObjectCreator;
    }
    
    /// Makes this property an inherited property.
	//
    /// Parameters:
    /// 	pBaseProp: the property to inherit from.
	virtual void SetInherited( const FdoSmLpPropertyDefinition* pBaseProp );

    /// Convenience function for casting a property to an object property.
    /// Returns NULL if the property is not an object property.
	static const FdoSmLpObjectPropertyDefinition* Cast( const FdoSmLpPropertyDefinition* src )
	{
		return( src ? ( src->GetPropertyType() == FdoPropertyType_ObjectProperty ? 
						(const FdoSmLpObjectPropertyDefinition*) src  : NULL
					  ) : NULL );
	}

    /// Create an Object Property Class.
	//
    /// Parameters:
    /// 	pParent: the object property
    /// 	pParentType: the class definition for the object property's type.
    /// 	pMapping: the table mapping definition for the object property.
    /// 	mappingType: the table mapping type for the object property.
    ///      pOverrides: Fdo to physical mapping overrides.
	virtual FdoPtr<FdoSmLpObjectPropertyClass> NewClass(
		FdoSmLpObjectPropertyDefinition* pParent, 
		FdoSmLpClassDefinition* pParentType,
		FdoSmLpPropertyMappingDefinition* pMapping,
		FdoSmLpPropertyMappingType mappingType,
        FdoPhysicalClassMapping* pOverrides
	)
    {
        return (FdoSmLpObjectPropertyClass*) NULL;
    }

    /// Create an Object Property Class from a base class. This is called when 
    /// an object property is inherited.
	//
    /// Parameters:
    /// 	pBase: the base class.
    /// 	pParent: the object property
    /// 	pParentType: the class definition for the object property's type.
    /// 	pMapping: the table mapping definition for the object property.
    /// 	mappingType: the table mapping type for the object property.
    ///      pOverrides: Fdo to physical mapping overrides.
	virtual FdoPtr<FdoSmLpObjectPropertyClass> NewClass(
    	FdoPtr<FdoSmLpObjectPropertyClass> pBase,
		FdoSmLpObjectPropertyDefinition* pParent, 
		FdoSmLpClassDefinition* pParentType,
		FdoSmLpPropertyMappingDefinition* pMapping,
		FdoSmLpPropertyMappingType mappingType, 
        FdoPhysicalClassMapping* pOverrides
	)
    {
        return (FdoSmLpObjectPropertyClass*) NULL;
    }

    /// Copy updates from corresponding FDO property. 
    //
    /// Parameters
    ///      pFdoProp: the FDO element.
    ///      elementState: the modification state for this object property
    ///      pPropOverrides: Fdo to physical mapping overrides.
    ///      bIgnoreStates: true if the element state on the FDO element is 
    ///          to be ignored.
	virtual void Update(
        FdoPropertyDefinition* pFdoProp,
        FdoSchemaElementState elementState,
        FdoPhysicalPropertyMapping* pPropOverrides,
        bool bIgnoreStates
    ); 

    /// Synchronize this object property and its table. The table is synchronized
    /// to match the object property.
    //
    /// Parameters:
    //
    ///      bRollbackOnly - 
    ///          true - synchronize only if a change to the class for this object property
    ///              has been rolled back since the previous synchronization.
    ///          false - always synchronize this object property.
	virtual void SynchPhysical(bool bRollbackOnly = true);

    /// Post updates to the database.
	virtual void Commit( bool fromParent );

	virtual FdoSchemaExceptionP Errors2Exception( FdoSchemaException* pFirstException = NULL ) const;

    /// Serialize this property to an XML file.
    /// Mainly for unit testing.
	virtual void XMLSerialize( FILE* xmlFp, int ref ) const;

protected:
    /// unused constructor needed only to build on Linux
    FdoSmLpObjectPropertyDefinition() {}

    FdoSmLpObjectPropertyDefinition(FdoSmPhClassPropertyReaderP propReader, FdoSmLpClassDefinition* parent);
    
    FdoSmLpObjectPropertyDefinition(
        FdoObjectPropertyDefinition* pFdoProp, 
        bool bIgnoreStates,
        FdoSmLpClassDefinition* parent
    );

    /// Constructs an instance of an ObjectPropertyDefinition from a 
    /// base property.
    /// Used mainly by CreateInherited() and CreateCopy().
    //
    /// Parameters:
    /// 	pBaseProperty: initialize the new property from this base property.
    /// 	pTargetClass: the class that will contain the new property.
    /// 	logicalName: The name for the new property.
    /// 		If L"" then the new property has the same name as this
    /// 		property.
    /// 	physicalName: not applicable to object properties
    /// 	bInherit:
    /// 		True: the new property inherits from the base property.
    /// 		False: the new property is just a copy of the base property.
    ///      pPropOverrides: Fdo to physical mapping overrides for created property.
    FdoSmLpObjectPropertyDefinition(
        FdoPtr<FdoSmLpObjectPropertyDefinition> pBaseProperty, 
        FdoSmLpClassDefinition* pTargetClass, 
        FdoStringP logicalName, 
        FdoStringP physicalName, 
        bool bInherit,
        FdoPhysicalPropertyMapping* pPropOverrides = NULL
    );

	~FdoSmLpObjectPropertyDefinition(void);

    /// Get Identity property in modifiable form.
    /// Warning: Finalize() is not tripped.
    FdoSmLpDataPropertyP GetIdentityProperty();

    /// Get the overridden table name, L"" if not overridden
    virtual FdoStringP GetOvTableName() = 0;

    /// Get object property class in modifiable form.
    /// Warning: Finalize() is not tripped.
	FdoPtr<FdoSmLpClassDefinition> GetClass();

    /// Set Identity property 
    void SetIdentityProperty( FdoSmLpDataPropertyP idProp );

    /// Set Identity property name
    void SetIdentityPropertyName( FdoString* idPropName );

    /// Set table mapping definition
    void SetMappingDefinition( FdoSmLpPropertyMappingP mappingDefinition );

    /// Set table mapping type
    void SetTableMapping( FdoSmLpPropertyMappingType tableMapping );

    /// Finalize this object property. Finds the class object defining
    /// the property type.
    /// Resolving the type class is not done in the constructor in case
    /// two schemas reference classes in each other. Class objects are
    /// constructed during schema loading so we don't want to resolve 
    /// these classes until at least one of the schemas has finished 
    /// loading.
	void Finalize();

    /// Creates a new object property table
    virtual FdoSmPhDbObjectP NewTable( 
        FdoSmPhOwnerP owner,
        FdoString* tableName
    );

    /// Creates a new view on a foreign object property table
    virtual FdoSmPhDbObjectP NewView( 
        FdoSmPhOwnerP viewOwner,
        FdoString* viewName, 
        FdoString* database,
        FdoString* owner,
        FdoString* tableName
    );

private:
    /// Check the MetaData for a dependency from the table for the containing class
    /// to the containing table for this property.
	void FindDependency(const FdoSmLpClassDefinition* pParent);

    /// Set the dependency row for this object property.
	void SetDependency( const FdoSmPhDependency* pDependency );

    /// Returns true if the relationship to the primary key table is defined by a base property.
    bool IsPkTableInherited( 
        const FdoSmLpObjectPropertyDefinition* pProp, 
        FdoStringP pkTableName 
    );

    void FinalizeTable( const FdoSmLpClassDefinition*	pParentClass );

    bool ReferenceLoop();

    /// Set the table mapping type to Concrete.
	virtual void SetConcreteMapping() = 0;
    virtual void SetSingleMapping() = 0;

    /// Various helpers for logging schema errors.

	void AddClassMappingError();
	void AddSingleMappingError();
	virtual void AddReferenceLoopError();	
	void AddRefClassMissingError(FdoString* refClassName);
	void AddRefClassDeleteError(const FdoSmLpClassDefinition* pRefClass);
	void AddRefClassAbstractError(const FdoSmLpClassDefinition* pRefClass);
	void AddRefClassFeatureError(const FdoSmLpClassDefinition* pRefClass);
	void AddTypeClassChangeError( FdoStringP fdoTypeClassName );
	void AddIdPropChangeError( FdoString* fdoIdPropName );
	void AddObjTypeChangeError( FdoObjectType fdoObjectType  );
	void AddOrderChangeError( FdoOrderType fdoOrderType  );
	void AddValueHasIdError();
	void AddTypeNoSchemaError( FdoString* typeClassName );
	void AddSingleCollectionError();
    void AddSubPropNotNullError(
        const FdoSmLpClassDefinition* pContainingClass,
        const FdoSmLpDataPropertyDefinition* pDataProp
    );
    void AddCreateNoMetaError( FdoSmPhOwnerP owner );

	FdoObjectType mObjectType;

	FdoStringP mClassName;

	FdoSmLpClassDefinition* mpClass;

	FdoStringP mIdentityPropertyName;
	FdoStringP mFdoIdPropName; // New id property name from FDO class.
    /// Set only when updated from FDO.

    FdoObjectPropertyP mFdoProp;

    FdoSmLpDataPropertyP mpIdentityProperty;

    FdoOrderType mOrderType;

	FdoSmLpPropertyMappingP mpMappingDefinition;
    FdoSmLpPropertyMappingType mTableMapping;

    /// Physical overrides for this object property.
    FdoPhysicalPropertyMappingP mOverrides;
    FdoStringP mPrefix;
    bool mbFixedDbObject;
    bool mbDbObjectCreator;
    FdoStringP mDefaultDbObjectName;
    FdoStringP mRootDbObjectName;

	const FdoSmPhDependency* mpDependency;

    /// This one is set when this object owns the dependency object instead of 
    /// just having a copy. Setting mpMyDependency ensures that the dependency
    /// gets deleted when this object is deleted.
	FdoSmPhDependencyP mpMyDependency;

};

typedef FdoPtr<FdoSmLpObjectPropertyDefinition> FdoSmLpObjectPropertyP;

#endif


