// Copyright (C) 2004-2006  Autodesk, Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of version 2.1 of the GNU Lesser
// General Public License as published by the Free Software Foundation.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "NlsTest.h"
#include "UnitTestUtil.h"
#include <ctime>
#include <cppunit/extensions/HelperMacros.h>
#include "SDF/SdfCommandType.h"
#include "SDF/ICreateSDFFile.h"
#ifdef _WIN32
static const wchar_t* NLS_TEST_FILE = L"..\\..\\TestData\\NlsTest.SDX";
#else
#include <unistd.h>
static const wchar_t* NLS_TEST_FILE = L"../../TestData/NlsTest.SDX";
#endif


CPPUNIT_TEST_SUITE_REGISTRATION( NlsTest );
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION( NlsTest, "NlsTest");

static FdoPropertyValue* AddNewProperty( FdoPropertyValueCollection* propertyValues, const wchar_t *name )
{
    FdoPropertyValue*  propertyValue = NULL;
    propertyValue = propertyValues->FindItem( name );

    if ( propertyValue == NULL )
    {
        propertyValue =  FdoPropertyValue::Create();
        propertyValue->SetName( name );
        propertyValues->Add( propertyValue );
    }

    return propertyValue;
}

NlsTest::NlsTest(void)
{
    
}

NlsTest::~NlsTest(void)
{
    
}


void NlsTest::setUp ()
{

}

void NlsTest::tearDown ()
{
}

void NlsTest::Test1 ()
{
    DoTest(false);
}

void NlsTest::Test2 ()
{
    DoTest(true);
}

void NlsTest::DoTest ( bool reconnect )
{
    FdoPtr<FdoIConnection> connection;

    try
    {
        connection = UnitTestUtil::OpenConnection( NLS_TEST_FILE, true );

        FdoFeatureSchemaP pTestSchema = InitSchema( connection );

        FdoPtr<FdoIApplySchema>pCmd = (FdoIApplySchema*) connection->CreateCommand(FdoCommandType_ApplySchema);
        pCmd->SetFeatureSchema( pTestSchema );
        pCmd->Execute();
        pTestSchema = NULL;

        FdoPtr<FdoIDescribeSchema>pDescSchemaCmd = (FdoIDescribeSchema*) connection->CreateCommand(FdoCommandType_DescribeSchema);
        FdoPtr<FdoFeatureSchemaCollection>pfsc = pDescSchemaCmd->Execute();
 
        if ( reconnect ) {
            connection->Close();
            connection = UnitTestUtil::OpenConnection( NLS_TEST_FILE, false );
        }

        AddFeatures( connection, pfsc, GetClass1Name() );
        AddFeatures( connection, pfsc, GetClass2Name() );

        if ( reconnect ) {
            connection->Close();
            connection = UnitTestUtil::OpenConnection( NLS_TEST_FILE, false );
        }

        SelectFeatures( connection, GetClass1Name() );
        SelectFeatures( connection, GetClass2Name() );
        SelectFeatures( connection, GetClass2Name(), true );
    }
    catch ( FdoException* e )
    {
        printf("Test1 Error: %ls\n", e->GetExceptionMessage() );
        e->Release();
        throw "Association schema TestLoad exception";
    }
}

FdoFeatureSchemaP NlsTest::InitSchema ( FdoIConnection* connection )
{
    FdoFeatureSchemaP pTestSchema = FdoFeatureSchema::Create( L"NlsSchema", L"Test non-ASCII7 class names" );

    FdoFeatureClassP   pfeatureclass = InitFeatureClass( GetClass1Name() );
    FdoClassesP(pTestSchema->GetClasses())->Add( pfeatureclass );
    pfeatureclass = InitFeatureClass( GetClass2Name() );
    FdoClassesP(pTestSchema->GetClasses())->Add( pfeatureclass );
    pfeatureclass = InitFeatureClass( FdoStringP::Format(L"%lc%lc%lc%lc%lc", 0x80, 0xe1, 0xe2, 0xa3, 0xa4) );
    FdoClassesP(pTestSchema->GetClasses())->Add( pfeatureclass );
    pfeatureclass = InitFeatureClass( FdoStringP::Format(L"%lc%lc%lc%lc%lcA", 0xff, 0x3901, 0x3903, 0x4000, 0x5000) );
    FdoClassesP(pTestSchema->GetClasses())->Add( pfeatureclass );
    pfeatureclass = InitFeatureClass( FdoStringP::Format(L"Schema1%lc", 0x3503) );
    FdoClassesP(pTestSchema->GetClasses())->Add( pfeatureclass );

    return pTestSchema;
}

FdoFeatureClassP NlsTest::InitFeatureClass( FdoStringP className ) 
{
    FdoFeatureClassP pfeatureclass = FdoFeatureClass::Create( className, L"FeatureClass Desc" );

    // Add identity property
    FdoDataPropertyP pProp = FdoDataPropertyDefinition::Create( L"FeatId", L"FeatId Prop" );
    pProp->SetDataType( FdoDataType_Int32 );
    pProp->SetNullable(false);
    pProp->SetIsAutoGenerated(true);
    FdoPtr<FdoPropertyDefinitionCollection>(pfeatureclass->GetProperties())->Add( pProp );
    FdoPtr<FdoDataPropertyDefinitionCollection>(pfeatureclass->GetIdentityProperties())->Add( pProp );

    pProp = FdoDataPropertyDefinition::Create( L"Name", L"Name" );
    pProp->SetDataType( FdoDataType_String );
    pProp->SetLength(32);
    pProp->SetNullable(false);
    FdoPtr<FdoPropertyDefinitionCollection>(pfeatureclass->GetProperties())->Add( pProp );

    FdoGeometricPropertyP pGeomProp = FdoGeometricPropertyDefinition::Create( L"Geometry", L"location" );
    pGeomProp->SetGeometryTypes( FdoGeometricType_Point | FdoGeometricType_Curve );
    pGeomProp->SetHasElevation(true);
    FdoPtr<FdoPropertyDefinitionCollection>(pfeatureclass->GetProperties())->Add( pGeomProp );
    pfeatureclass->SetGeometryProperty( pGeomProp );

    return pfeatureclass;
}

void NlsTest::AddFeatures( FdoIConnection* connection, FdoFeatureSchemaCollection* schemas, FdoStringP className )
{
    FdoFeatureSchemaP insSchema = schemas->FindItem( L"NlsSchema" );

    FdoClassDefinitionP insClass = FdoClassesP(insSchema->GetClasses())->GetItem( className );

    FdoPtr<FdoIInsert> insertCommand = (FdoIInsert *) connection->CreateCommand(FdoCommandType_Insert);
    insertCommand->SetFeatureClassName(insClass->GetName());

    AddFeature( insertCommand, 1 );
    AddFeature( insertCommand, 2 );
    AddFeature( insertCommand, 3 );
}
 
void NlsTest::AddFeature( FdoIInsert* insertCommand, FdoInt32 idx )
{
    FdoPtr<FdoPropertyValueCollection> propertyValues = insertCommand->GetPropertyValues();
    FdoPtr<FdoDataValue> dataValue;
    FdoPtr<FdoPropertyValue> propertyValue;

    dataValue = FdoDataValue::Create((FdoInt32) idx);
    propertyValue = AddNewProperty( propertyValues, L"FeatId");
    propertyValue->SetValue(dataValue);

    dataValue = FdoDataValue::Create((FdoString*)FdoStringP::Format(L"Name%d",idx));
    propertyValue = AddNewProperty( propertyValues, L"Name");
    propertyValue->SetValue(dataValue);

    double       coordsBuffer[3];
    int          segCount = 1;

    coordsBuffer[0] = 5 * idx;
    coordsBuffer[1] = 10 * idx;
    coordsBuffer[2] = idx;

    propertyValue = AddNewProperty( propertyValues, L"Geometry");
    FdoPtr<FdoFgfGeometryFactory> gf = FdoFgfGeometryFactory::GetInstance();
    FdoPtr<FdoILineString> line1 = gf->CreateLineString(FdoDimensionality_XY|FdoDimensionality_Z, segCount*3, coordsBuffer);
    FdoPtr<FdoByteArray> byteArray = gf->GetFgf(line1);
    FdoPtr<FdoGeometryValue> geometryValue = FdoGeometryValue::Create(byteArray);
    propertyValue->SetValue(geometryValue);

    FdoPtr<FdoIFeatureReader> reader = insertCommand->Execute();
}

void NlsTest::SelectFeatures( FdoIConnection* connection, FdoStringP className, bool isSpatial )
{
    FdoPtr<FdoISelect> selectCommand = (FdoISelect *) connection->CreateCommand(FdoCommandType_Select);
    selectCommand->SetFeatureClassName( className );

    if ( isSpatial ) {
        FdoPtr<FdoFgfGeometryFactory> gf = FdoFgfGeometryFactory::GetInstance();

        double ordsXYExt[15];
        ordsXYExt[0] = 7.5; ordsXYExt[1] = 17.5; ordsXYExt[2] = 0.0;
        ordsXYExt[3] = 7.5; ordsXYExt[4] = 22.5; ordsXYExt[5] = 0.0;
        ordsXYExt[6] = 12.5; ordsXYExt[7] = 22.5; ordsXYExt[8] = 0.0;
        ordsXYExt[9] = 12.5; ordsXYExt[10] = 17.5; ordsXYExt[11] = 0.0;
        ordsXYExt[12] = 7.5; ordsXYExt[13] = 17.5;  ordsXYExt[14] = 0.0;

        FdoPtr<FdoILinearRing> extRing = gf->CreateLinearRing(FdoDimensionality_XY|FdoDimensionality_Z, 15, ordsXYExt);
        FdoPtr<FdoIPolygon> poly = gf->CreatePolygon(extRing, NULL );
        FdoPtr<FdoGeometryValue> geomValue = FdoPtr<FdoGeometryValue>(FdoGeometryValue::Create(FdoPtr<FdoByteArray>(gf->GetFgf(poly))));
        FdoPtr<FdoSpatialCondition> spatialFilter = FdoPtr<FdoSpatialCondition>(FdoSpatialCondition::Create(L"Geometry",
                                                                          FdoSpatialOperations_Intersects,
                                                                          geomValue));
        selectCommand->SetFilter(spatialFilter);
    }

    FdoPtr<FdoIFeatureReader> rdr = selectCommand->Execute();

    FdoPtr<FdoIdentifierCollection> props = selectCommand->GetPropertyNames();

    int rowCount = 0;

    while ( rdr->ReadNext() )
    {

        rowCount++;

        FdoInt32 featId = rdr->GetInt32( L"FeatId" );
        if ( isSpatial ) 
            CPPUNIT_ASSERT( featId == 2 );
        else
            CPPUNIT_ASSERT( featId > 0 && featId < 4 );

        FdoStringP expectedName = FdoStringP::Format( L"Name%d", featId );
        double expectedX = 5 * featId;
        double expectedY = 10 * featId;
        double expectedZ= featId;

        CPPUNIT_ASSERT( expectedName == rdr->GetString(L"Name") );

        FdoPtr<FdoFgfGeometryFactory> gf = FdoFgfGeometryFactory::GetInstance();
        FdoPtr<FdoByteArray> sGeom = rdr->GetGeometry( L"Geometry" );
        CPPUNIT_ASSERT( sGeom );

        FdoPtr<FdoILineString> line1 = (FdoILineString*) gf->CreateGeometryFromFgf( sGeom );
        FdoPtr<FdoIDirectPosition> pos = line1->GetItem(0);

        CPPUNIT_ASSERT( expectedX == pos->GetX() );
        CPPUNIT_ASSERT( expectedY == pos->GetY() );
        CPPUNIT_ASSERT( expectedZ == pos->GetZ() );
    }

    if ( isSpatial )
        CPPUNIT_ASSERT( rowCount == 1 );
    else
        CPPUNIT_ASSERT( rowCount == 3 );

    rdr->Close();
}

FdoStringP NlsTest::GetClass1Name()
{
    return FdoStringP::Format(L"Test%lc%lc", 0x30b2, 0x311b);
}

FdoStringP NlsTest::GetClass2Name()
{
    return FdoStringP::Format(L"%lc%lc%lc%lc", 0x30b2, 0x311b, 0x06a3, 0x1246);
}


