/*
 * Copyright (C) 2004-2006  Autodesk, Inc.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser
 * General Public License as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "Pch.h"
#include "BasicSchemaTests.h"

CPPUNIT_TEST_SUITE_REGISTRATION (BasicSchemaTests);
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION (BasicSchemaTests, "BasicSchemaTests");

// NOTES:
//   - In order to properly test defect 659586.01 "ArcSDE: Corrupted views cause describeschema to fail linked ",
//   a corrupt ArcSDE view or table needs to be created in the database.  This can be done by create a table
//   using command-line tool "sdetable.exe", then dropping the table using sqlplus's "drop table" command
//   (the table remains registered under ArcSDE even though it no longer exists).
//   On Oracle database r9204 on server otwe1, there is such a view called "defect659586view".

static const wchar_t* class_description = L"Dwelling address";
static const wchar_t* id_name = L"FeatId";
static const wchar_t* id_description = L"Unique identifier";
static const bool     id_nullable = false;
static const bool     id_autogenerated = true;
static const bool     id_readonly = true;
static const wchar_t* street_name = L"Street";
static const wchar_t* street_description = L"Street address";
static const int      street_length = 1000;
static const bool     street_nullable = false;
static const wchar_t* streetnumber_name = L"StreetNumber";
static const wchar_t* streetnumber_description = L"Street number";
static const int      streetnumber_length = 1000;
static const bool     streetnumber_nullable = false;
static const wchar_t* location_name = L"Location2";  //NOTE: oracle doesn't like "Location" as an identifier!
static const wchar_t* location_description = L"Geodetic location";
static const int      location_geometry_types = FdoGeometricType_Point;
static const bool     location_has_elevation = false;
static const bool     location_has_measure = false;

//static wchar_t*       class_name_diffuser = L"TestDiffUser";


FdoPtr<FdoIConnection> BasicSchemaTests::mConnection;

BasicSchemaTests::BasicSchemaTests(void)
{
}

BasicSchemaTests::~BasicSchemaTests(void)
{
}

void BasicSchemaTests::setUp ()
{
    if (!CreateSchemaOnly() /* || !bSchemaCreated */ )
    {
        mConnection = ArcSDETests::GetConnection ();
        mConnection->SetConnectionString (ArcSDETestConfig::ConnStringMetadcov());
        mConnection->Open ();
    }
}

void BasicSchemaTests::tearDown ()
{
    if (mConnection)
        mConnection->Close ();
}

void BasicSchemaTests::get_schema_names()
{
	if (CreateSchemaOnly()) 
		return;

	FdoPtr<ArcSDEGetSchemaNamesCommand> command = (ArcSDEGetSchemaNamesCommand*)mConnection->CreateCommand(FdoCommandType_GetSchemaNames);
	FdoPtr<FdoStringCollection> schemas = command->Execute();
	for (FdoInt32 i = 0; i < schemas->GetCount(); ++i)
		printf("%ls:\n", schemas->GetString(i));

	printf("Schema number: %d\n", schemas->GetCount());
}

void BasicSchemaTests::get_feature_class_names()
{
	if (CreateSchemaOnly()) 
		return;

	FdoPtr<ArcSDEGetSchemaNamesCommand> command = (ArcSDEGetSchemaNamesCommand*)mConnection->CreateCommand(FdoCommandType_GetSchemaNames);
	FdoPtr<FdoStringCollection> schemas = command->Execute();
	FdoInt32 num = 0;
	for (FdoInt32 i = 0; i < schemas->GetCount(); ++i)
	{
		printf("%ls:\n", schemas->GetString(i));
		FdoPtr<ArcSDEGetClassNamesCommand> cmd = (ArcSDEGetClassNamesCommand*)mConnection->CreateCommand(FdoCommandType_GetClassNames);
		cmd->SetSchemaName(schemas->GetString(i));
		FdoPtr<FdoStringCollection> classNames = cmd->Execute();
		for (FdoInt32 j = 0; j < classNames->GetCount(); ++j)
		{
			FdoStringP className = classNames->GetString(j);
			CPPUNIT_ASSERT_MESSAGE("The class name is not qualified.", className.Contains(L":"));
			FdoStringP schemaName = className.Left(L":");
			CPPUNIT_ASSERT_MESSAGE("The class isn't found in the schema.", schemaName == schemas->GetString(i));
			printf("   %ls\n", classNames->GetString(j));
			++num;
		}
	}

	printf("Schema number %d\n", schemas->GetCount());
	printf("Feature class number %d\n", num);
}

void BasicSchemaTests::describe_one_specified_class()
{
	if (CreateSchemaOnly()) 
		return;

	try
	{
		// Fully describe the schema
		FdoPtr<ArcSDEDescribeSchemaCommand> command = (ArcSDEDescribeSchemaCommand*)mConnection->CreateCommand(FdoCommandType_DescribeSchema);
		FdoPtr<FdoFeatureSchemaCollection> schemas = command->Execute();

		// Get the first feature class in the first schema.
		FdoPtr<FdoFeatureSchema> featureSchema = schemas->GetItem(0);
		FdoStringP schemaName = featureSchema->GetName();
		FdoPtr<FdoClassCollection> classes = featureSchema->GetClasses();
		FdoPtr<FdoClassDefinition> classDef = classes->GetItem(0);
		FdoStringP className = classDef->GetName();

		// Try to describe the specified class.
		command->SetSchemaName(schemaName);
		FdoStringsP classNames = FdoStringCollection::Create();
		classNames->Add(className);
		command->SetClassNames(classNames);
		schemas = command->Execute();

		CPPUNIT_ASSERT_MESSAGE("The number of the returned schemas is not 1.", schemas->GetCount() == 1);
		FdoPtr<FdoFeatureSchema> featureSchema2 = schemas->GetItem(0);
		CPPUNIT_ASSERT_MESSAGE("Both schema names are not identical.", FdoStringP(featureSchema->GetName()) == FdoStringP(featureSchema2->GetName()));
		classes = featureSchema2->GetClasses();
		CPPUNIT_ASSERT_MESSAGE("The number of the returned classes is not 1.", classes->GetCount() == 1);
		FdoPtr<FdoClassDefinition> classDef2 = classes->GetItem(0);

		// Compare both class definitions.
		CPPUNIT_ASSERT_MESSAGE("Both class names are not same.", FdoStringP(classDef->GetName()) == FdoStringP(classDef2->GetName()));
		FdoPtr<FdoPropertyDefinitionCollection> properties = classDef->GetProperties();
		FdoPtr<FdoPropertyDefinitionCollection> properties2 = classDef2->GetProperties();
		CPPUNIT_ASSERT_MESSAGE("The property number is not equal.", properties->GetCount() == properties2->GetCount());
		for (FdoInt32 i = 0; i < properties->GetCount(); ++i)
		{
			FdoPtr<FdoPropertyDefinition> propDef = properties->GetItem(i);
			FdoPtr<FdoPropertyDefinition> propDef2 = properties2->GetItem(propDef->GetName());
			CPPUNIT_ASSERT_MESSAGE("The property is not found.", propDef2 != NULL);
			CPPUNIT_ASSERT_MESSAGE("The property type is not same.", propDef->GetPropertyType() == propDef2->GetPropertyType());
		}
	}
	catch (FdoException* e)
	{
		fail(e);
	}
}

void show_schema (FdoFeatureSchemaCollection* schemas, bool bVerifyClassWriteCapability = false, bool bClassesShouldSupportWrite = false)
{
    for (int i = 0; i < schemas->GetCount (); i++)
    {
        FdoFeatureSchema* schema = schemas->GetItem (i);
        printf ("Schema: %ls\n", schema->GetName());
        if ((schema->GetDescription () != NULL) && (0 != wcscmp (schema->GetDescription (), L"")))
            printf ("    Description: %ls\n", schema->GetDescription ());
        FdoClassCollection* classes = schema->GetClasses ();
        for (int j = 0; j < classes->GetCount (); j++)
        {
            FdoClassDefinition* cls = classes->GetItem (j);

            // Output basic class info:
            if (FdoClassType_FeatureClass == cls->GetClassType ())
                printf ("    Feature Class: %ls\n", cls->GetName ());
            else
                printf ("    Class: %ls\n", cls->GetName ());
            if ((cls->GetDescription () != NULL) && (0 != wcscmp (cls->GetDescription (), L"")))
                printf ("        Description: %ls\n", cls->GetDescription ());

            FdoPtr<FdoClassCapabilities> classCapabilities = cls->GetCapabilities();
            printf ("        Class Capabilities:\n");
            if (classCapabilities == NULL)
                printf ("            (Not available).\n");
            else
            {
                printf ("            Supports locking: %s\n", classCapabilities->SupportsLocking() ? "yes" : "no");
                printf ("            Supports long transactions: %s\n", classCapabilities->SupportsLongTransactions() ? "yes" : "no");
                printf ("            Supports write: %s\n", classCapabilities->SupportsWrite() ? "yes" : "no");

                if (bVerifyClassWriteCapability)
                    CPPUNIT_ASSERT_MESSAGE("Class should be read-only", bClassesShouldSupportWrite == classCapabilities->SupportsWrite());
            }


            // Output identity properties:
            FdoDataPropertyDefinitionCollection* identity = cls->GetIdentityProperties ();
            for (int k = 0; k < identity->GetCount (); k++)
            {
                FdoDataPropertyDefinition* definition = identity->GetItem (k);
                printf ("        Id: %ls\n", definition->GetName ());
                if ((definition->GetDescription () != NULL) && (0 != wcscmp (definition->GetDescription (), L"")))
                    printf ("            Description: %ls\n", definition->GetDescription ());
                printf ("            Type: %ls Length: %d Precision: %d %ls\n",
                    ArcSDETests::GetDataTypeString (definition->GetDataType ()),
                    definition->GetLength (),
                    definition->GetPrecision (),
                    definition->GetNullable () ? L"Nullable" : L"NotNull");
                definition->Release ();
            }

            // Output regular properties:
            FdoPropertyDefinitionCollection* properties = cls->GetProperties ();
            for (int k = 0; k < properties->GetCount (); k++)
            {
                FdoPropertyDefinition* definition = properties->GetItem (k);
                if (definition->GetPropertyType () == FdoPropertyType_DataProperty)
                {
                    FdoDataPropertyDefinition* data_definition = (FdoDataPropertyDefinition*)definition;
                    if (!identity->Contains (data_definition))
                    {
                        printf ("        Property: %ls\n", definition->GetName ());
                        if ((data_definition->GetDescription () != NULL) && (0 != wcscmp (data_definition->GetDescription (), L"")))
                            printf ("            Description: %ls\n", data_definition->GetDescription ());
                        printf ("            Type: %ls Length: %d Precision: %d %ls\n",
                            ArcSDETests::GetDataTypeString (data_definition->GetDataType ()),
                            data_definition->GetLength (),
                            data_definition->GetPrecision (),
                            data_definition->GetNullable () ? L"Nullable" : L"NotNull");
                    }
                }
                else if (definition->GetPropertyType () == FdoPropertyType_ObjectProperty)
                {
                    printf ("       Object Property: %ls\n", definition->GetName ());
                    if ((definition->GetDescription () != NULL) && (0 != wcscmp (definition->GetDescription (), L"")))
                        printf ("            Description: %ls\n", definition->GetDescription ());
                }
                else if (definition->GetPropertyType () == FdoPropertyType_GeometricProperty)
                {
                    FdoGeometricPropertyDefinition* geometry_definition = (FdoGeometricPropertyDefinition*)definition;
                    printf ("        Geometric Property: %ls\n", geometry_definition->GetName ());
                    if ((geometry_definition->GetDescription () != NULL) && (0 != wcscmp (geometry_definition->GetDescription (), L"")))
                        printf ("            Description: %ls\n", geometry_definition->GetDescription ());
                    int types = geometry_definition->GetGeometryTypes ();
                    if (0 != (types & FdoGeometricType_Point))
                        printf ("            FdoGeometricType_Point types allowed\n");
                    if (0 != (types & FdoGeometricType_Curve))
                        printf ("            FdoGeometricType_Curve types allowed\n");
                    if (0 != (types & FdoGeometricType_Surface))
                        printf ("            FdoGeometricType_Surface types allowed\n");
                    if (0 != (types & FdoGeometricType_Solid))
                        printf ("            FdoGeometricType_Solid types allowed\n");
                }
                definition->Release ();
            }
            identity->Release ();
            properties->Release ();

            // Output schema attribute dictionary:
            FdoSchemaAttributeDictionary* dictionary = cls->GetAttributes ();
            int count;
            const wchar_t **names = dictionary->GetAttributeNames (count);
            if ((0 != count) && (NULL != names))
            {
                printf ("        MetaData:");
                const wchar_t *name = *names;
                for (int i = 0; i < count; i++)
                {
                    if (0 != i)
                        printf (",");
                    const wchar_t* value = dictionary->GetAttributeValue (name);
                    printf (" %ls=%ls", name, value);
                    name++;
                }
                printf ("\n");
            }
            dictionary->Release ();

            cls->Release ();
        }
        classes->Release ();
        schema->Release ();
    }
}

/* Test basic describe operation. */
void BasicSchemaTests::describe ()
{
    if (CreateSchemaOnly())  return;

    try
    {
        double startSeconds = (double)(long)clock()/1000.0;
        printf("\nBasicSchemaTests::describe(): describeschema begin clock=%f", startSeconds);

        FdoIDescribeSchema*  describe = (FdoIDescribeSchema*)mConnection->CreateCommand (FdoCommandType_DescribeSchema);
        FdoFeatureSchemaCollection* schemas = describe->Execute ();

        double endSeconds = (double)(long)clock()/1000.0;
        printf("\nBasicSchemaTests::describe(): describeschema end clock=%f, elapsed time=%f", endSeconds, endSeconds-startSeconds);

        show_schema (schemas);
        schemas->Release ();
        describe->Release ();
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }
}

/* Test apply schema operation. */
void BasicSchemaTests::apply ()
{
#ifdef _DEBUG  // Since this test relies heavily on ApplySchema to achieve proper testing,
               // we can't easily factor the ApplySchema calls out using CreateSchemaOnly();
               // So, we only run this test when ApplySchema is available (e.g. in _DEBUG mode only).

    if (CreateSchemaOnly())  return;

    try
    {
		// Clean up previous tests:
        CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaAddress(), ArcSDETestConfig::ClassNameAddress());

        FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
        FdoPtr<FdoFeatureSchema> schema = ArcSDETests::GetDefaultSchema(mConnection);  //FdoFeatureSchema::Create (schema_name, schema_description);
		FdoPtr<FdoClassCollection> classes = schema->GetClasses ();

        // build an id property
        FdoPtr<FdoDataPropertyDefinition> id = FdoDataPropertyDefinition::Create (id_name, id_description);
        id->SetDataType (FdoDataType_Int32);
        id->SetNullable (id_nullable);
        id->SetIsAutoGenerated (id_autogenerated);
        id->SetReadOnly (id_readonly);

        // build a street data property
        FdoPtr<FdoDataPropertyDefinition> street = FdoDataPropertyDefinition::Create (street_name, street_description);
        street->SetDataType (FdoDataType_String);
        street->SetLength (street_length);
        street->SetNullable (street_nullable);

        // build a location geometry property
        FdoPtr<FdoGeometricPropertyDefinition> location = FdoGeometricPropertyDefinition::Create (location_name, location_description);
        location->SetGeometryTypes (location_geometry_types);
        location->SetHasElevation (location_has_elevation);
        location->SetHasMeasure (location_has_measure);

        // build an owner object property
        //FdoPtr<FdoClass> owner = FdoClass::Create (L"Owner", L"Dwelling owner");
        //FdoPtr<FdoObjectPropertyDefinition> owned_by = FdoObjectPropertyDefinition::Create (L"Owner", L"Reference to owner");
        //FdoPtr<FdoPropertyDefinitionCollection> owner_properties = owner->GetProperties ();
        //owner_properties->Add (owned_by);
        //classes->Add (owner);

        // assemble the feature class
        FdoPtr<FdoFeatureClass> feature = FdoFeatureClass::Create (ArcSDETestConfig::ClassNameAddress(), class_description);
        FdoPtr<FdoPropertyDefinitionCollection> properties = feature->GetProperties ();
        properties->Add (id);
        properties->Add (street);
        properties->Add (location);
        feature->SetGeometryProperty (location);
        FdoPtr<FdoDataPropertyDefinitionCollection> identities = feature->GetIdentityProperties ();
        identities->Add (id);

        // submit the new schema
        classes->Add (feature);
        apply->SetFeatureSchema (schema);
        apply->Execute ();
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }

#endif // _DEBUG  // Since this test uses ApplySchema
}

/* Test describe one schema operation. */
void BasicSchemaTests::describe_one ()
{
#ifdef _DEBUG  // Since this test relies heavily on ApplySchema to achieve proper testing,
               // we can't easily factor the ApplySchema calls out using CreateSchemaOnly();
               // So, we only run this test when ApplySchema is available (e.g. in _DEBUG mode only).

    if (CreateSchemaOnly())  return;

    try
    {
        FdoPtr<FdoFeatureSchema> schema = ArcSDETests::GetDefaultSchema(mConnection);  //FdoFeatureSchema::Create (schema_name, schema_description);
        CPPUNIT_ASSERT_MESSAGE ("expected schema was not found", schema != NULL);
        FdoPtr<FdoClassCollection> classes = schema->GetClasses ();
        // MULTIPLE SCHEMAS NOT SUPPORTED: CPPUNIT_ASSERT_MESSAGE ("wrong number of classes", 1 == classes->GetCount ());
        FdoPtr<FdoClassDefinition> feature = classes->GetItem (ArcSDETestConfig::ClassNameAddress());
        CPPUNIT_ASSERT_MESSAGE ("class has wrong name", feature != NULL);
        FdoPtr<FdoPropertyDefinitionCollection> properties = feature->GetProperties ();
        CPPUNIT_ASSERT_MESSAGE ("wrong number of properties", 3 == properties->GetCount ());
        FdoPtr<FdoDataPropertyDefinition> id = static_cast<FdoDataPropertyDefinition*> (properties->GetItem (id_name));
        CPPUNIT_ASSERT_MESSAGE ("class has no id property", id != NULL);
        CPPUNIT_ASSERT_MESSAGE ("property id has wrong description", 0 == wcscmp (id->GetDescription (), id_description));
        CPPUNIT_ASSERT_MESSAGE ("property id has wrong data type", FdoDataType_Int32 == id->GetDataType ());
        CPPUNIT_ASSERT_MESSAGE ("property id has wrong nullability", id_nullable == id->GetNullable ());
        CPPUNIT_ASSERT_MESSAGE ("property id is not autogenerated", id_autogenerated == id->GetIsAutoGenerated ());
        CPPUNIT_ASSERT_MESSAGE ("property id has wrong protection", id_readonly == id->GetReadOnly ());
        FdoPtr<FdoDataPropertyDefinition> street = static_cast<FdoDataPropertyDefinition*> (properties->GetItem (street_name));
        CPPUNIT_ASSERT_MESSAGE ("class has no street name property", street != NULL);
        CPPUNIT_ASSERT_MESSAGE ("property street has wrong description", 0 == wcscmp (street->GetDescription (), street_description));
        CPPUNIT_ASSERT_MESSAGE ("property street has wrong data type", FdoDataType_String == street->GetDataType ());
        CPPUNIT_ASSERT_MESSAGE ("property street has wrong length", street_length == street->GetLength ());
        CPPUNIT_ASSERT_MESSAGE ("property street has wrong nullability", street_nullable == street->GetNullable ());
        FdoPtr<FdoGeometricPropertyDefinition> location = static_cast<FdoGeometricPropertyDefinition*> (properties->GetItem (location_name));
        CPPUNIT_ASSERT_MESSAGE ("property location has wrong description", 0 == wcscmp (location->GetDescription (), location_description));
        CPPUNIT_ASSERT_MESSAGE ("property location has wrong geometry types", location_geometry_types == location->GetGeometryTypes ());
        CPPUNIT_ASSERT_MESSAGE ("property location has wrong has_elevation", location_has_elevation == location->GetHasElevation ());
        CPPUNIT_ASSERT_MESSAGE ("property location has wrong has_measure", location_has_measure == location->GetHasMeasure ());
        FdoPtr<FdoDataPropertyDefinitionCollection> identities = feature->GetIdentityProperties ();
        id = static_cast<FdoDataPropertyDefinition*> (identities->GetItem (0));
        CPPUNIT_ASSERT_MESSAGE ("class has no identity property", id != NULL);
        CPPUNIT_ASSERT_MESSAGE ("property id has wrong description", 0 == wcscmp (id->GetDescription (), id_description));
        CPPUNIT_ASSERT_MESSAGE ("property id has wrong data type", FdoDataType_Int32 == id->GetDataType ());
        CPPUNIT_ASSERT_MESSAGE ("property id has wrong nullability", id_nullable == id->GetNullable ());
        CPPUNIT_ASSERT_MESSAGE ("property id has wrong protection", id_readonly == id->GetReadOnly ());

        // Make sure the identity property doesn't come back as a constraint:
        FdoPtr<FdoUniqueConstraintCollection> uniqueConstraintsRead = feature->GetUniqueConstraints();
        CPPUNIT_ASSERT_MESSAGE("Wrong unique constraint count", uniqueConstraintsRead->GetCount() == 0);
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }
#endif // _DEBUG  // since this tests relies indirectly on ApplySchema, which is only exposed in debug builds
}

/* Test delete a class. */
void BasicSchemaTests::delete_class ()
{
#ifdef _DEBUG  // Since this test relies heavily on ApplySchema to achieve proper testing,
               // we can't easily factor the ApplySchema calls out using CreateSchemaOnly();
               // So, we only run this test when ApplySchema is available (e.g. in _DEBUG mode only).

    if (CreateSchemaOnly())  return;

    try
    {
        FdoPtr<FdoFeatureSchema> schema = ArcSDETests::GetDefaultSchema(mConnection);  //FdoFeatureSchema::Create (schema_name, schema_description);
        FdoPtr<FdoClassCollection> classes = schema->GetClasses ();
        FdoPtr<FdoClassDefinition> feature = classes->GetItem (ArcSDETestConfig::ClassNameAddress());
        feature->Delete ();
        FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
        apply->SetFeatureSchema (schema);
        apply->Execute ();
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }

#endif // _DEBUG  // Since this test uses ApplySchema
}

void BasicSchemaTests::destroy_schema ()
{
#ifdef _DEBUG  // Since this test relies heavily on ApplySchema to achieve proper testing,
               // we can't easily factor the ApplySchema calls out using CreateSchemaOnly();
               // So, we only run this test when ApplySchema is available (e.g. in _DEBUG mode only).

/* // DESTROYSCHEMA ONLY DESTROYS CLASSES; NEED TO UPDATE THIS TEST

    if (CreateSchemaOnly())  return;

    try
    {
		// Clean up previous tests:
		CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaAddress(), ArcSDETestConfig::ClassNameAddress());

        FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
        FdoPtr<FdoFeatureSchema> schema = ArcSDETests::GetDefaultSchema(mConnection);  //FdoFeatureSchema::Create (schema_name, schema_description);
        FdoPtr<FdoClassCollection> classes = schema->GetClasses ();

        // build an id property
        FdoPtr<FdoDataPropertyDefinition> id = FdoDataPropertyDefinition::Create (id_name, id_description);
        id->SetDataType (FdoDataType_Int32);
        id->SetNullable (id_nullable);
        id->SetIsAutoGenerated (id_autogenerated);
        id->SetReadOnly (id_readonly);

        // build a street data property
        FdoPtr<FdoDataPropertyDefinition> street = FdoDataPropertyDefinition::Create (street_name, street_description);
        street->SetDataType (FdoDataType_String);
        street->SetLength (street_length);
        street->SetNullable (street_nullable);

        // build a location geometry property
        FdoPtr<FdoGeometricPropertyDefinition> location = FdoGeometricPropertyDefinition::Create (location_name, location_description);
        location->SetGeometryTypes (location_geometry_types);
        location->SetHasElevation (location_has_elevation);
        location->SetHasMeasure (location_has_measure);

        // build an owner object property
        //FdoPtr<FdoClass> owner = FdoClass::Create (L"Owner", L"Dwelling owner");
        //FdoPtr<FdoObjectPropertyDefinition> owned_by = FdoObjectPropertyDefinition::Create (L"Owner", L"Reference to owner");
        //FdoPtr<FdoPropertyDefinitionCollection> owner_properties = owner->GetProperties ();
        //owner_properties->Add (owned_by);
        //classes->Add (owner);

        // assemble the feature class
        FdoPtr<FdoFeatureClass> feature = FdoFeatureClass::Create (ArcSDETestConfig::ClassNameAddress(), class_description);
        FdoPtr<FdoPropertyDefinitionCollection> properties = feature->GetProperties ();
        properties->Add (id);
        properties->Add (street);
        properties->Add (location);
        feature->SetGeometryProperty (location);
        FdoPtr<FdoDataPropertyDefinitionCollection> identities = feature->GetIdentityProperties ();
        identities->Add (id);

        // submit the new schema
        classes->Add (feature);
        apply->SetFeatureSchema (schema);
        apply->Execute ();

        // now delete the schema
        FdoPtr<FdoIDestroySchema> destroy = (FdoIDestroySchema*)mConnection->CreateCommand (FdoCommandType_DestroySchema);
        destroy->SetSchemaName (schema_name);
        destroy->Execute ();

        // make sure it's gone
        FdoPtr<FdoIDescribeSchema> describe = (FdoIDescribeSchema*)mConnection->CreateCommand (FdoCommandType_DescribeSchema);
        describe->SetSchemaName (schema_name);
        FdoPtr<FdoFeatureSchemaCollection> schemas = describe->Execute ();
        CPPUNIT_ASSERT_MESSAGE ("schema still alive", 0 == schemas->GetCount ());
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }

	*/ // MULTIPLE SCHEMAS NOT SUPPORTED, HENCE DESTROYSCHEMA COMMAND NOT SUPPORTED.

#endif // _DEBUG  // Since this test uses DestroySchema
}



/* Test apply schema operation with odd class/property names. */
void BasicSchemaTests::apply_odd_names ()
{
#ifdef _DEBUG  // Since this test relies heavily on ApplySchema to achieve proper testing,
               // we can't easily factor the ApplySchema calls out using CreateSchemaOnly();
               // So, we only run this test when ApplySchema is available (e.g. in _DEBUG mode only).

    if (CreateSchemaOnly())  return;

    try
    {
		FdoString* odd_prop_id_name = L"2_FeatId";
		FdoString* odd_prop_street_name = L"StreetWayWayWayWayWayWayWayWayWayWayWayWayTooLong";  // too long
		FdoString* odd_prop_location_name = L"STREETWayWayWayWayWayWayWayWayWayWayWayWayTooLong";  // same name, different case

		// Clean up previous tests:
        CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaAddressWayTooLong(), ArcSDETestConfig::ClassNameAddressWayTooLong());

        FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
        FdoPtr<FdoFeatureSchema> schema = ArcSDETests::GetDefaultSchema(mConnection);  //FdoFeatureSchema::Create (schema_name, schema_description);
        FdoPtr<FdoClassCollection> classes = schema->GetClasses ();

        // build an id property
        FdoPtr<FdoDataPropertyDefinition> id = FdoDataPropertyDefinition::Create (odd_prop_id_name, id_description);
        id->SetDataType (FdoDataType_Int32);
        id->SetNullable (id_nullable);
        id->SetIsAutoGenerated (id_autogenerated);
        id->SetReadOnly (id_readonly);

        // build a street data property
        FdoPtr<FdoDataPropertyDefinition> street = FdoDataPropertyDefinition::Create (odd_prop_street_name, street_description);
        street->SetDataType (FdoDataType_String);
        street->SetLength (street_length);
        street->SetNullable (street_nullable);

        // build a location geometry property
        FdoPtr<FdoGeometricPropertyDefinition> location = FdoGeometricPropertyDefinition::Create (odd_prop_location_name, location_description);
        location->SetGeometryTypes (location_geometry_types);
        location->SetHasElevation (location_has_elevation);
        location->SetHasMeasure (location_has_measure);

        // build an owner object property
        //FdoPtr<FdoClass> owner = FdoClass::Create (L"Owner", L"Dwelling owner");
        //FdoPtr<FdoObjectPropertyDefinition> owned_by = FdoObjectPropertyDefinition::Create (L"Owner", L"Reference to owner");
        //FdoPtr<FdoPropertyDefinitionCollection> owner_properties = owner->GetProperties ();
        //owner_properties->Add (owned_by);
        //classes->Add (owner);

        // assemble the feature class
        FdoPtr<FdoFeatureClass> feature = FdoFeatureClass::Create (ArcSDETestConfig::ClassNameAddressWayTooLong(), class_description);
        FdoPtr<FdoPropertyDefinitionCollection> properties = feature->GetProperties ();
        properties->Add (id);
        properties->Add (street);
        properties->Add (location);
        feature->SetGeometryProperty (location);
        FdoPtr<FdoDataPropertyDefinitionCollection> identities = feature->GetIdentityProperties ();
        identities->Add (id);

        // submit the new schema
        classes->Add (feature);
        apply->SetFeatureSchema (schema);
        apply->Execute ();

        // Delete schema:
        feature->Delete ();
        apply->SetFeatureSchema (schema);
        apply->Execute ();
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }

#endif // _DEBUG  // Since this test uses ApplySchema
}



/* Test apply schema operation, against a different user than the connected user. */
void BasicSchemaTests::apply_different_user ()
{
#ifdef _DEBUG  // Since this test relies heavily on ApplySchema to achieve proper testing,
               // we can't easily factor the ApplySchema calls out using CreateSchemaOnly();
               // So, we only run this test when ApplySchema is available (e.g. in _DEBUG mode only).

/* //NOTE: This test is commented out for now because when creating an ArcSDE table in a different Oracle user,
   // there are issues with having the right priviledges.   Attempting to do "GRANT ALL PRIVILEDGES" to the
   // connected user did NOT solve the problem.

    if (CreateSchemaOnly())  return;

    try
    {
		// Clean up previous tests:
		CleanUpClass(mConnection, CLASS_TESTDIFFUSER_SCHEMA, class_name_diffuser);

        // Get a different user's schema (AUSTRALIA) than the user we are connected as (METADCOV):
        FdoPtr<FdoIDescribeSchema> describe = (FdoIDescribeSchema*)mConnection->CreateCommand (FdoCommandType_DescribeSchema);
        describe->SetSchemaName (ArcSDETestConfig::UserNameAustralia());
        FdoPtr<FdoFeatureSchemaCollection> describedschemas = describe->Execute ();

        FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
        FdoPtr<FdoFeatureSchema> schema = describedschemas->GetItem(0);
        FdoPtr<FdoClassCollection> classes = schema->GetClasses ();

        // build an id property
        FdoPtr<FdoDataPropertyDefinition> id = FdoDataPropertyDefinition::Create (id_name, id_description);
        id->SetDataType (FdoDataType_Int32);
        id->SetNullable (id_nullable);
        id->SetIsAutoGenerated (id_autogenerated);
        id->SetReadOnly (id_readonly);

        // build a street data property
        FdoPtr<FdoDataPropertyDefinition> street = FdoDataPropertyDefinition::Create (street_name, street_description);
        street->SetDataType (FdoDataType_String);
        street->SetLength (street_length);
        street->SetNullable (street_nullable);

        // build a location geometry property
        FdoPtr<FdoGeometricPropertyDefinition> location = FdoGeometricPropertyDefinition::Create (location_name, location_description);
        location->SetGeometryTypes (location_geometry_types);
        location->SetHasElevation (location_has_elevation);
        location->SetHasMeasure (location_has_measure);

        // assemble the feature class
        FdoPtr<FdoFeatureClass> feature = FdoFeatureClass::Create (class_name_diffuser, class_description);
        FdoPtr<FdoPropertyDefinitionCollection> properties = feature->GetProperties ();
        properties->Add (id);
        properties->Add (street);
        properties->Add (location);
        feature->SetGeometryProperty (location);
        FdoPtr<FdoDataPropertyDefinitionCollection> identities = feature->GetIdentityProperties ();
        identities->Add (id);

        // submit the new schema
        classes->Add (feature);
        apply->SetFeatureSchema (schema);
        apply->Execute ();

        // validate the class is in the right schema:
        describe = (FdoIDescribeSchema*)mConnection->CreateCommand (FdoCommandType_DescribeSchema);
        describe->SetSchemaName (ArcSDETestConfig::UserNameAustralia());
        describedschemas = describe->Execute ();
        FdoPtr<FdoIDisposableCollection> foundClasses = describedschemas->FindClass(class_name_diffuser);
        CPPUNIT_ASSERT_MESSAGE("Didn't find created class", foundClasses->GetCount() == 1);

		// Clean up after test:
		CleanUpClass(mConnection, CLASS_TESTDIFFUSER_SCHEMA, class_name_diffuser);
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }
*/

#endif // _DEBUG  // Since this test uses ApplySchema
}

/* Test describe operation with many release operations. */
void BasicSchemaTests::describe_many ()
{
//In a loop like a dialog creation
//1>	Call describe schema
//2>	Iterate the schema collection, fill an array with each schema in the collection
//3>	Release the collection and the describe
//4>	Iterate the schema array, calling release on each schema

    if (CreateSchemaOnly())  return;

    try
    {
        for (int i = 0; i < 100; i++)
        {
            FdoIDescribeSchema*  describe = (FdoIDescribeSchema*)mConnection->CreateCommand (FdoCommandType_DescribeSchema);
            FdoFeatureSchemaCollection* schemas = describe->Execute ();
            int n = schemas->GetCount ();
            FdoFeatureSchema** array = new FdoFeatureSchema*[n];
            for (int j = 0; j < n; j++)
                array[j] = schemas->GetItem (j);
            describe->Release ();
            schemas->Release ();
            for (int j = 0; j < n; j++)
                array[j]->Release ();
            delete[] array;
        }
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }
}


void BasicSchemaTests::create_unique_constraint()
{
#ifdef _DEBUG  // Since this test relies heavily on ApplySchema to achieve proper testing,
               // we can't easily factor the ApplySchema calls out using CreateSchemaOnly();
               // So, we only run this test when ApplySchema is available (e.g. in _DEBUG mode only).

    if (CreateSchemaOnly())  return;

    try
    {
		// Clean up previous tests:
		CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaAddress(), ArcSDETestConfig::ClassNameAddress2());

        FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
        FdoPtr<FdoFeatureSchema> schema = ArcSDETests::GetDefaultSchema(mConnection);
        FdoPtr<FdoClassCollection> classes = schema->GetClasses ();

        // build an id property
        FdoPtr<FdoDataPropertyDefinition> id = FdoDataPropertyDefinition::Create (id_name, id_description);
        id->SetDataType (FdoDataType_Int32);
        id->SetNullable (id_nullable);
        id->SetIsAutoGenerated (id_autogenerated);
        id->SetReadOnly (id_readonly);

        // build a street data property
        FdoPtr<FdoDataPropertyDefinition> street = FdoDataPropertyDefinition::Create (street_name, street_description);
        street->SetDataType (FdoDataType_String);
        street->SetLength (street_length);
        street->SetNullable (street_nullable);

        // build a street data property
        FdoPtr<FdoDataPropertyDefinition> streetNumber = FdoDataPropertyDefinition::Create (streetnumber_name, streetnumber_description);
        streetNumber->SetDataType (FdoDataType_Int32);
        streetNumber->SetNullable (streetnumber_nullable);

        // build a location geometry property
        FdoPtr<FdoGeometricPropertyDefinition> location = FdoGeometricPropertyDefinition::Create (location_name, location_description);
        location->SetGeometryTypes (location_geometry_types);
        location->SetHasElevation (location_has_elevation);
        location->SetHasMeasure (location_has_measure);

        // assemble the feature class
        FdoPtr<FdoFeatureClass> feature = FdoFeatureClass::Create (ArcSDETestConfig::ClassNameAddress2(), class_description);
        FdoPtr<FdoPropertyDefinitionCollection> properties = feature->GetProperties ();
        properties->Add (id);
        properties->Add (street);
        properties->Add (location);
        properties->Add (streetNumber);
        feature->SetGeometryProperty (location);
        FdoPtr<FdoDataPropertyDefinitionCollection> identities = feature->GetIdentityProperties ();
        identities->Add (id);

        // Add a 2-property unique constraint:
        FdoPtr<FdoUniqueConstraintCollection> uniqueConstraints = feature->GetUniqueConstraints();
        FdoPtr<FdoUniqueConstraint> uniqueConstraint = FdoUniqueConstraint::Create();
        FdoPtr<FdoDataPropertyDefinitionCollection> uniqueConstraintDataProps = uniqueConstraint->GetProperties();
        uniqueConstraintDataProps->Add(street);
        uniqueConstraintDataProps->Add(streetNumber);
        uniqueConstraints->Add(uniqueConstraint);

        // submit the new schema
        classes->Add (feature);
        apply->SetFeatureSchema (schema);
        apply->Execute ();

        // Re-read the class we just created:
        FdoPtr<FdoIDescribeSchema> descSchema = (FdoIDescribeSchema*)mConnection->CreateCommand(FdoCommandType_DescribeSchema);
        FdoPtr<FdoFeatureSchemaCollection> schemas = descSchema->Execute();
        FdoPtr<FdoFeatureSchema> schemaRead = schemas->GetItem(ArcSDETestConfig::ClassSchemaAddress());
        FdoPtr<FdoClassCollection> classDefs = schemaRead->GetClasses();
        FdoPtr<FdoClassDefinition> classDef = classDefs->GetItem(ArcSDETestConfig::ClassNameAddress2());

        // Compare the original constraint to the one we read back:
        FdoPtr<FdoUniqueConstraintCollection> uniqueConstraintsRead = classDef->GetUniqueConstraints();
        int lUniqueConstraintsReadCount = uniqueConstraintsRead->GetCount();
        bool bUniqueConstraintFound = false;
        CPPUNIT_ASSERT_MESSAGE("Wrong unique constraint count", lUniqueConstraintsReadCount == 1);
        for (int j=0; j<lUniqueConstraintsReadCount && !bUniqueConstraintFound; j++)
        {
            FdoPtr<FdoUniqueConstraint> uniqueConstraintRead = uniqueConstraintsRead->GetItem(j);
            FdoPtr<FdoDataPropertyDefinitionCollection> uniqueConstraintDataPropsRead = uniqueConstraintRead->GetProperties();

            const int CONSTRAINT_PROP_COUNT = 2;
            bool bFoundConstraintDataProp[CONSTRAINT_PROP_COUNT];

            if (uniqueConstraintDataPropsRead->GetCount() == uniqueConstraintDataProps->GetCount())
            {
                for (int i=0; i<CONSTRAINT_PROP_COUNT; i++)
                    bFoundConstraintDataProp[i] = false;

                for (int i=0; i<uniqueConstraintDataPropsRead->GetCount(); i++)
                {
                    FdoPtr<FdoDataPropertyDefinition> dataPropRead = uniqueConstraintDataPropsRead->GetItem(i);
                    for (int k=0; k<uniqueConstraintDataProps->GetCount(); k++)
                    {
                        FdoPtr<FdoDataPropertyDefinition> dataPropOriginal = uniqueConstraintDataProps->GetItem(k);
                        if (0==wcscmp(dataPropOriginal->GetName(), dataPropRead->GetName()))
                            bFoundConstraintDataProp[k] = true;
                    }
                }

                bUniqueConstraintFound = true;
                for (int i=0; i<CONSTRAINT_PROP_COUNT; i++)
                    if (!bFoundConstraintDataProp[i])
                        bUniqueConstraintFound = false;
            }
        }

        CPPUNIT_ASSERT_MESSAGE("Unique constraint not found", bUniqueConstraintFound);

        // Try to insert features with duplicate column values that violate the unique constraint we created:

        FdoPtr<FdoIInsert> insert = (FdoIInsert*)mConnection->CreateCommand(FdoCommandType_Insert);
        insert->SetFeatureClassName(ArcSDETestConfig::QClassNameAddress2());
        FdoPtr<FdoPropertyValueCollection> propVals = insert->GetPropertyValues();
        FdoPtr<FdoPropertyValue> propVal;
        propVal = FdoPropertyValue::Create(streetnumber_name, NULL);
        propVal->SetValue(L"123");
        propVals->Add(propVal);
        propVal = FdoPropertyValue::Create(street_name, NULL);
        propVal->SetValue(L"'Rockefeller Ave'");
        propVals->Add(propVal);
        FdoPtr<FdoIFeatureReader> reader = insert->Execute();
        reader->Close();

        try
        {
            reader = insert->Execute();
            reader->Close();
            CPPUNIT_FAIL("Didn't get an exception inserting 2 rows with duplicate values in unique constraint columns");
        }
        catch (FdoException *e)
        {
            // We want to get an exception here, so dont assert that the test failed.
            e->Release();
        }

    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }

#endif // _DEBUG  // Since this test uses ApplySchema
}



/* Test basic describe operation. */
void BasicSchemaTests::describe_read_only ()
{
    if (CreateSchemaOnly())  return;

    try
    {
        // connect as a user with read-only privileges:
        FdoPtr<FdoIConnection> conn = ArcSDETests::GetConnection ();
        conn->SetConnectionString (ArcSDETestConfig::ConnStringReadOnly());
        conn->Open ();

        // describe with read-only privileges:
        FdoPtr<FdoIDescribeSchema> describe = (FdoIDescribeSchema*)conn->CreateCommand (FdoCommandType_DescribeSchema);
        FdoPtr<FdoFeatureSchemaCollection> schemas = describe->Execute ();
        show_schema (schemas, true, false);
        conn->Close ();

        // describe with read-write pribileges:
        describe = (FdoIDescribeSchema*)mConnection->CreateCommand (FdoCommandType_DescribeSchema);
        schemas = describe->Execute ();
        show_schema (schemas, true, true);
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }
}



void BasicSchemaTests::apply_specific_geometry_types ()
{
#ifdef _DEBUG  // Since this test relies heavily on ApplySchema to achieve proper testing,
               // we can't easily factor the ApplySchema calls out using CreateSchemaOnly();
               // So, we only run this test when ApplySchema is available (e.g. in _DEBUG mode only).

    if (CreateSchemaOnly())  return;

    try
    {
		// Clean up previous tests:
        CleanUpClass(mConnection, ArcSDETestConfig::ClassSchemaAddress(), ArcSDETestConfig::ClassNameAddress());

        FdoPtr<FdoIApplySchema> apply = (FdoIApplySchema*)mConnection->CreateCommand (FdoCommandType_ApplySchema);
        FdoPtr<FdoFeatureSchema> schema = ArcSDETests::GetDefaultSchema(mConnection);  //FdoFeatureSchema::Create (schema_name, schema_description);
        FdoPtr<FdoClassCollection> classes = schema->GetClasses ();

        // build an id property
        FdoPtr<FdoDataPropertyDefinition> id = FdoDataPropertyDefinition::Create (id_name, id_description);
        id->SetDataType (FdoDataType_Int32);
        id->SetNullable (id_nullable);
        id->SetIsAutoGenerated (id_autogenerated);
        id->SetReadOnly (id_readonly);

        // build a street data property
        FdoPtr<FdoDataPropertyDefinition> street = FdoDataPropertyDefinition::Create (street_name, street_description);
        street->SetDataType (FdoDataType_String);
        street->SetLength (street_length);
        street->SetNullable (street_nullable);

        // build a location geometry property
        FdoPtr<FdoGeometricPropertyDefinition> location = FdoGeometricPropertyDefinition::Create (location_name, location_description);
        FdoGeometryType geomTypes[] = {FdoGeometryType_Point, FdoGeometryType_LineString};
        location->SetSpecificGeometryTypes (geomTypes, 2);
        location->SetHasElevation (location_has_elevation);
        location->SetHasMeasure (location_has_measure);

        // assemble the feature class
        FdoPtr<FdoFeatureClass> feature = FdoFeatureClass::Create (ArcSDETestConfig::ClassNameAddress(), class_description);
        FdoPtr<FdoPropertyDefinitionCollection> properties = feature->GetProperties ();
        properties->Add (id);
        properties->Add (street);
        properties->Add (location);
        feature->SetGeometryProperty (location);
        FdoPtr<FdoDataPropertyDefinitionCollection> identities = feature->GetIdentityProperties ();
        identities->Add (id);

        // submit the new schema
        classes->Add (feature);
        apply->SetFeatureSchema (schema);
        apply->Execute ();

        // describe schema to make sure it comes back the same:
        FdoPtr<FdoIDescribeSchema> describe = (FdoIDescribeSchema*)mConnection->CreateCommand (FdoCommandType_DescribeSchema);
        FdoPtr<FdoFeatureSchemaCollection> schemas = describe->Execute ();
        schema = schemas->GetItem(ArcSDETestConfig::ClassSchemaAddress());
        classes = schema->GetClasses();
        FdoPtr<FdoClassDefinition> classDef = classes->GetItem(ArcSDETestConfig::ClassNameAddress());
        FdoPtr<FdoPropertyDefinitionCollection> propDefs = classDef->GetProperties();
        FdoPtr<FdoPropertyDefinition> propDef = propDefs->GetItem(location_name);
        FdoGeometricPropertyDefinition* geomProp = static_cast<FdoGeometricPropertyDefinition*>(propDef.p);
        int geomTypeDescCount = -1;
        FdoGeometryType* geomTypesDesc = geomProp->GetSpecificGeometryTypes(geomTypeDescCount);
        CPPUNIT_ASSERT_MESSAGE("Wrong specific geom type count", geomTypeDescCount==2);
        CPPUNIT_ASSERT_MESSAGE("wrong specific geom types", geomTypesDesc[0]==FdoGeometryType_Point || geomTypesDesc[1]==FdoGeometryType_Point);
        CPPUNIT_ASSERT_MESSAGE("wrong specific geom types", geomTypesDesc[0]==FdoGeometryType_LineString || geomTypesDesc[1]==FdoGeometryType_LineString);
    }
    catch (FdoException* ge) 
    {
        fail (ge);
    }

#endif // _DEBUG  // Since this test uses ApplySchema
}

