// Copyright (C) 2004-2006  Autodesk, Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of version 2.1 of the GNU Lesser
// General Public License as published by the Free Software Foundation.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.IO;

using OSGeo.FDO.Connections;
using OSGeo.FDO.Commands;
using OSGeo.FDO.Commands.Schema;
using OSGeo.FDO.Commands.Feature;
using OSGeo.FDO.Schema;
using OSGeo.FDO.Expression;
using OSGeo.FDO.Geometry;

using unit_test.Framework;

namespace unit_test.ProviderTests
{
    class ReadOnlyTests : BaseTestWithConnection
    {
        public ReadOnlyTests(ShpTestProvider connectionProvider)
            : base(connectionProvider)
        { }

        public override void SetUp()
        {
            base.SetUp();
        }

        public override void TearDown()
        {
            base.TearDown();
        }
        public enum GeometricType
        {
            GeometricType_Point = 0x01,
            GeometricType_Curve = 0x02,
            GeometricType_Surface = 0x04,
            GeometricType_Solid = 0x08
        };

        const string schemaName = "TheSchema";
        const string className = "Test";

        private void create_schema(GeometricType type, bool elevation, bool measure)
        {
            // Delete old class, if its still there:
            ShpTests.CleanUpClass(base.ConnectionInitialized, null, className);


            DataPropertyDefinition FeatId = new DataPropertyDefinition("FeatId", "integer");
            FeatId.DataType = DataType.DataType_Int32;
            FeatId.IsAutoGenerated = true;
            FeatId.Nullable = false;

            DataPropertyDefinition Id = new DataPropertyDefinition("Id", "decimal");
            Id.DataType = DataType.DataType_Decimal;
            Id.Precision = 10;
            Id.Scale = 0;

            DataPropertyDefinition street = new DataPropertyDefinition("Street", "text");
            street.DataType = DataType.DataType_String;
            street.Length = 64;

            DataPropertyDefinition area = new DataPropertyDefinition("Area", "double");
            area.DataType = DataType.DataType_Decimal;
            area.Precision = 20;
            area.Scale = 8;

            DataPropertyDefinition vacant = new DataPropertyDefinition("Vacant", "boolean");
            vacant.DataType = DataType.DataType_Boolean;

            DataPropertyDefinition birthday = new DataPropertyDefinition("Birthday", "date");
            birthday.DataType = DataType.DataType_DateTime;

            // build a location geometry property
            GeometricPropertyDefinition location = new GeometricPropertyDefinition("Geometry", "geometry");
            location.GeometryTypes = (int)type;
            location.HasElevation = true;
            location.HasMeasure = true;

            // assemble the feature class
            FeatureClass feature = new FeatureClass(className, "test class created with apply schema");
            PropertyDefinitionCollection properties = feature.Properties;
            properties.Add(FeatId);
            properties.Add(Id);
            properties.Add(street);
            properties.Add(area);
            properties.Add(vacant);
            properties.Add(birthday);
            properties.Add(location);
            feature.GeometryProperty = location;
            DataPropertyDefinitionCollection identities = feature.IdentityProperties;
            identities.Add(FeatId);

            FeatureSchema schema = new FeatureSchema(schemaName, "");
            ClassCollection classes = schema.Classes;
            classes.Add(feature);

            // submit the new schema

            IApplySchema apply = (IApplySchema)base.ConnectionInitialized.CreateCommand(CommandType.CommandType_ApplySchema);
            apply.FeatureSchema = schema;
            apply.Execute();
        }

        public void Test_select ()
        {
            double  area;
            double  length;
            int count;
            try
            {
                ISelect select = (ISelect)base.ConnectionInitialized.CreateCommand (CommandType.CommandType_Select);
                select.SetFeatureClassName ("ontario");
                IFeatureReader reader = select.Execute ();
                count = 0;
                while (reader.ReadNext ())
                {
                    count++;
                    reader.GetInt32 ("FeatId");
                    area = reader.GetDouble ("AREA");
                    length = reader.GetDouble ("PERIMETER");
                    reader.GetDouble ("ONTARIO_");
                    reader.GetDouble ("ONTARIO_ID");

                    int len = reader.GetGeometry("Geometry").Length;
                    Byte[] geometry = new Byte[len];
                    geometry = reader.GetGeometry("Geometry");
                    
                    ShpTests.AnalyzeGeometry( "ontario", count, geometry, length, area);
                }
                unit_test_assert ("no ontario features selected", 0 != count);
            }
            catch (OSGeo.FDO.Common.Exception ge)
            {
                ShpTests.Fail(ge);
            }
            catch(Exception)
            {
                Debug.Fail ("ReadOnlyTests.select() failed");
            }

        }

        public void Test_apply_schema ()
        {
            try
            {
                create_schema(GeometricType.GeometricType_Point, true, true);
                Debug.Fail ("apply schema worked on a read-only volume");
            }
            catch (OSGeo.FDO.Common.Exception ge)
            {
                // check for 'read-only' somwehere in the message
                bool ok = ge.Message.Contains("read-only");
                unit_test_assert ("no read-only message from ApplySchema", ok);
            }
        }

        public void Test_insert ()
        {
            try
            {

                IInsert insert = (IInsert)base.ConnectionInitialized.CreateCommand (CommandType.CommandType_Insert);
                insert.SetFeatureClassName ("ontario");
                PropertyValueCollection values = insert.PropertyValues;

                ValueExpression expression = (ValueExpression)ShpTests.ParseByDataType("2400129.29", DataType.DataType_Decimal);
                PropertyValue value = new PropertyValue("AREA", expression);
                values.Add (value);

                expression = (ValueExpression)ShpTests.ParseByDataType("72821.56", DataType.DataType_Decimal);
                value = new PropertyValue("PERIMETER", expression);
                values.Add (value);

                expression = (ValueExpression)ShpTests.ParseByDataType("2", DataType.DataType_Decimal);
                value = new PropertyValue("ONTARIO_", expression);
                values.Add (value);

                expression = (ValueExpression)ShpTests.ParseByDataType("2", DataType.DataType_Decimal);
                value = new PropertyValue("ONTARIO_ID", expression);
                values.Add (value);

                // add real geometry value:
                GeometryValue geometry = (GeometryValue)Expression.Parse ("GeomFromText('POLYGON XY ((5108.8 5104.7, 5109 5104, 5109 5105, 5108.8 5104.7))')");
                value = new PropertyValue("Geometry", geometry);
                values.Add (value);

                IFeatureReader reader = insert.Execute ();
                if (reader.ReadNext ())
                    Debug.Fail ("insert really worked on a read-only volume");
                reader.Close ();
                Debug.Fail ("insert worked on a read-only volume");
            }
            catch (OSGeo.FDO.Common.Exception ge)
            {
                // check for 'read-only' somwehere in the message
                bool ok = ge.Message.Contains("read-only");
                unit_test_assert("no read-only message from ApplySchema", ok);
            }
        }

        public void Test_update ()
        {
            try
            {
                IUpdate update = (IUpdate)base.ConnectionInitialized.CreateCommand (CommandType.CommandType_Update);
                update.SetFeatureClassName ("ontario");
	            PropertyValueCollection values = update.PropertyValues;
                ValueExpression expression = new DecimalValue(99.9999);
                PropertyValue value = new PropertyValue("ONTARIO_ID", expression);
                values.Add (value);
                if (0 != update.Execute ())
                    Debug.Fail ("update really worked on a read-only volume");
                Debug.Fail ("update worked on a read-only volume");
            }
            catch (OSGeo.FDO.Common.Exception ge)
            {
                // check for 'read-only' somwehere in the message
                bool ok = ge.Message.Contains("read-only");
                unit_test_assert("no read-only message from ApplySchema", ok);
            }
        }

        public void Test_del ()
        {
            try
            {
                IDelete del = (IDelete)base.ConnectionInitialized.CreateCommand (CommandType.CommandType_Delete);
                del.SetFeatureClassName ("ontario");
                int n = del.Execute ();
                if (0 != del.Execute ())
                    Debug.Fail ("delete really worked on a read-only volume");
                Debug.Fail ("delete worked on a read-only volume");
            }
            catch (OSGeo.FDO.Common.Exception ge)
            {
                // check for 'read-only' somwehere in the message
                bool ok = ge.Message.Contains("read-only");
                unit_test_assert("no read-only message from ApplySchema", ok);
            }
        }

    }
}
