#ifndef _ISCHEMACAPABILITIES_H_
#define _ISCHEMACAPABILITIES_H_
// 

//
// Copyright (C) 2004-2006  Autodesk, Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of version 2.1 of the GNU Lesser
// General Public License as published by the Free Software Foundation.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifdef _WIN32
#pragma once
#endif

#include <FdoStd.h>
#include <Fdo/Schema/ClassType.h>
#include <Fdo/Schema/DataType.h>
#include <Fdo/Connections/Capabilities/SchemaElementNameType.h>

/// \brief
/// The FdoISchemaCapbilities interface declares the feature provider's level 
/// of support for the Feature Schema.
class FdoISchemaCapabilities : public FdoIDisposable
{
public:
    /// \brief
    /// Returns an array of the FdoClassType objects the feature provider supports.
    /// 
    /// \param length 
    /// Output the number of class types.
    /// 
    /// \return
    /// Returns the list of class types.
    /// 
    FDO_API virtual FdoClassType* GetClassTypes(FdoInt32& length) = 0;

    /// \brief
    /// Returns an array of the FdoDataType objects the feature provider supports.
    /// 
    /// \param length 
    /// Output the number of data types.
    /// 
    /// \return
    /// Returns the list of data types
    /// 
    FDO_API virtual FdoDataType* GetDataTypes(FdoInt32& length) = 0;

    /// \brief
    /// Returns the maximum supported length of String, BLOB, or CLOB data
    /// properties. For decimal, it is typically the combination of the maximum
    /// scale and precision. For other data types that are not variable in size,
    /// the value returned is the byte length.
    ///
    /// \param dataType
    /// The data type for which the information is to retrieved.
    ///
    /// \return
    /// Returns the maximum data value length for the identified data type.
    ///
    FDO_API virtual FdoInt64 GetMaximumDataValueLength(FdoDataType dataType) = 0;

    /// \brief
    /// Returns the maximum supported precision for a decimal data property.
    ///
    /// \return
    /// Returns the maximum supported precision for a decimal data property.
    ///
    FDO_API virtual FdoInt32 GetMaximumDecimalPrecision() = 0;

    /// \brief
    /// Returns the maximum supported scale for a decimal data property.
    ///
    /// \return
    /// Returns the maximum supported scale for a decimal data property.
    ///
    FDO_API virtual FdoInt32 GetMaximumDecimalScale() = 0;

    /// \brief
    /// Returns the maximum size of a value of the given type. It includes
    /// limits for the data store name, shema name, class name, property name
    /// and description.
    ///
    /// \param name
    /// The schema element name type identifier for which the information is to
    /// be retrieved. Can be any of the following options: FdoSchemaElementNameType_Datastore,
    /// FdoSchemaElementNameType_Schema, FdoSchemaElementNameType_Class, FdoSchemaElementNameType_Property or 
    /// FdoSchemaElementNameType_Description
    ///
    /// \return
    /// Returns the size limitation for the identified schema element.
    ///
    FDO_API virtual FdoInt32 GetNameSizeLimit(FdoSchemaElementNameType name) = 0;

    /// \brief
    /// Returns a string that includes all the reserved characters that cannot be
    /// used for the various schema element names for the provider.
    ///
    /// \return
    /// Returns a string with the reserved characters for the identified schema element.
    ///
    FDO_API virtual FdoString* GetReservedCharactersForName() = 0;

    /// \brief
    /// Returns an array of the FdoDataType objects the feature provider supports for auto-generation.
    /// 
    /// \param length 
    /// Output the number of data types.
    /// 
    /// \return
    /// Returns the list of data types
    /// 
    FDO_API virtual FdoDataType* GetSupportedAutoGeneratedTypes(FdoInt32& length) = 0;

    /// \brief
    /// Returns a list of property types that can be used for identity properties.
    ///
    /// \param length
    /// Output the number of data types.
    ///
    /// \return
    /// Returns a string with the reserved characters for the identified schema element.
    ///
    FDO_API virtual FdoDataType* GetSupportedIdentityPropertyTypes(FdoInt32& length) = 0;

    /// \brief
    /// Determines if the provider supports association property.
    /// 
    /// \return
    /// Returns true if the provider support the association property.
    /// 
    FDO_API virtual bool SupportsAssociationProperties() = 0;

    /// \brief
    /// Determines if the provider can auto-generate unique ID values for a class.
    /// 
    /// \return
    /// Returns true if the provider supports id autogeneration.
    /// 
    FDO_API virtual bool SupportsAutoIdGeneration() = 0;

    /// \brief
    /// Determines if the provider supports multiple identity properties per class.
    /// 
    /// \return
    /// Returns true if the provider supports multiple identity properties per class.
    /// 
    FDO_API virtual bool SupportsCompositeId() = 0;

    /// \brief
    /// Determines if the provider supports composite unique value constraint.
    /// 
    /// \return
    /// Returns true if the provider supports composite unique value constraint.
    /// 
    FDO_API virtual bool SupportsCompositeUniqueValueConstraints() = 0;
    
    /// \brief
    /// Determines if the provider can auto-generate unique ID values that are unique for the entire datastore, rather than just for a particular class.
    /// 
    /// \return
    /// Returns true if the provider supports id autogeneration unique across an datastore.
    /// 
    FDO_API virtual bool SupportsDataStoreScopeUniqueIdGeneration() = 0;

    /// \brief
    /// Returns TRUE if default values can be specified for a data property
    /// definition, FALSE otherwise.
    ///
    /// \return
    /// Returns TRUE if default values can be specified for a data property
    /// definition, FALSE otherwise.
    ///
    FDO_API virtual bool SupportsDefaultValue() = 0;

    /// \brief
    /// Determines if the provider supports exclusive value range constraints.
    /// 
    /// \return
    /// Returns true if the provider supports exclusive value range constraints.
    /// 
	FDO_API virtual bool SupportsExclusiveValueRangeConstraints() = 0;

    /// \brief
    /// Determines if the provider supports inclusive value range constraints.
    /// 
    /// \return
    /// Returns true if the provider supports inclusive value range constraints.
    /// 
	FDO_API virtual bool SupportsInclusiveValueRangeConstraints() = 0;

/// \cond DOXYGEN-IGNORE

    /// \brief
    /// DEPRECATED: Use SupportsInheritance() instead.
    ///          Determines if the feature provider supports inheritance.
    /// 
    /// \return
    /// Returns true if the feature provider supports inheritance
    /// 
    FDO_API FDO_DEPRECATED virtual bool SupportsInheritence();
/// \endcond

    /// \brief
    /// Determines if the feature provider supports inheritance.
    /// 
    /// \return
    /// Returns true if the feature provider supports inheritance
    /// 
    FDO_API virtual bool SupportsInheritance() = 0;

    /// \brief
    /// Determines if the provider supports multiple schemas. Returns false if the provider supports only a single schema.
    /// 
    /// \return
    /// Returns true if the provider supports multiple schemas.
    /// 
    FDO_API virtual bool SupportsMultipleSchemas() = 0;

    /// \brief
    /// Determines if the provider supports the network model.
    /// 
    /// \return
    /// Returns true if the provider supports the network model.
    /// 
    FDO_API virtual bool SupportsNetworkModel() = 0;

    /// \brief
    /// Determines if the provider supports nullable value constraint.
    /// 
    /// \return
    /// Returns true if the provider supports nullable value constraint.
    /// 
    FDO_API virtual bool SupportsNullValueConstraints() = 0;

    /// \brief
    /// Determines if the provider supports object properties. Returns false if the provider supports only data and geometric properties.
    /// 
    /// \return
    /// Returns true if the provider supports object properties.
    /// 
    FDO_API virtual bool SupportsObjectProperties() = 0;
    
    /// \brief
    /// Determines if the provider support update a schema through the ApplySchema command.
    /// If False, then an application can only create a schema once for the provider. They can use the ApplySchema 
    /// command once for a given datastore. Once the schema is defined, it cannot be changed. If True, then the provider 
    /// supports the ability to modify or add to the schema after it is defined initially.
    /// 
    /// \return
    /// Returns true if the provider supports the update of the schema.
    /// 
    FDO_API virtual bool  SupportsSchemaModification() = 0;

    /// \brief
    /// Determines if the provider supports Schema Mapping Overrides 
    /// (overrides to the default rules for mapping Feature Schemas to provider-specific
    /// physical schemas)
    /// 
    /// \return
    /// Returns true if the provider supports Schema Overrides
    /// 
   	FDO_API virtual bool SupportsSchemaOverrides() = 0;

    /// \brief
    /// Determines if the provider supports unique value constraint.
    /// 
    /// \return
    /// Returns true if the provider supports unique value constraint.
    /// 
    FDO_API virtual bool SupportsUniqueValueConstraints() = 0;

    /// \brief
    /// Determines if the provider supports value constraints list.
    /// 
    /// \return
    /// Returns true if the provider supports value constraints list.
    /// 
    FDO_API virtual bool SupportsValueConstraintsList() = 0;
};
#endif


