//  Copyright (C) 2004-2006  Autodesk, Inc.
//  
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of version 2.1 of the GNU Lesser
//  General Public License as published by the Free Software Foundation.
//  
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
class SdfSchemaCapabilities : public FdoISchemaCapabilities
{
    //-------------------------------------------------------
    // Constructor / destructor
    //-------------------------------------------------------

    public:
        // default constructor
        SdfSchemaCapabilities();

    protected:
        // default destructor
        virtual ~SdfSchemaCapabilities();

    //-------------------------------------------------------
    // FdoIDisposable implementation
    //-------------------------------------------------------

    protected:
        // dispose this object
        SDF_API virtual void Dispose();

    //-------------------------------------------------------
    // FdoISchemaCapabilities implementation
    //-------------------------------------------------------

    public:
        // Returns an array of the ClassTypes the feature provider supports.
        // The length parameter gives the number of class types in the array.
        SDF_API virtual FdoClassType* GetClassTypes(FdoInt32& length);

        // Returns an array of the DataTypes the feature provider supports.
        // The length parameter gives the number of data types in the array.
        SDF_API virtual FdoDataType* GetDataTypes(FdoInt32& length);

        // Returns true if the feature provider supports inheritance.
        SDF_API virtual bool SupportsInheritance();

        // Returns true if the provider supports multiple schemas, and false if
        // if only supports a single schema.
        SDF_API virtual bool SupportsMultipleSchemas();

        // Returns true if the provider supports object properties, and false if
        // it only supports data and geometric properties.
        SDF_API virtual bool SupportsObjectProperties();

        /// <summary>Determines if the provider supports association property.</summary>
        /// <returns>Returns true if the provider support the association property.</returns> 
        SDF_API virtual bool SupportsAssociationProperties();

        /// <summary>Determines if the provider supports Schema Mapping Overrides 
        /// (overrides to the default rules for mapping Feature Schemas to provider-specific
        /// physical schemas)</summary>
        /// <returns>Returns true if the provider supports Schema Overrides</returns> 
        SDF_API virtual bool SupportsSchemaOverrides();

        /// <summary>Determines if the provider supports the network model.</summary>
        /// <returns>Returns true if the provider supports the network model.</returns> 
        SDF_API virtual bool SupportsNetworkModel();

        /// <summary>Determines if the provider can auto-generate unique ID values for a class.</summary>
        /// <returns>Returns true if the provider supports id autogeneration.</returns> 
        SDF_API virtual bool SupportsAutoIdGeneration();
        
        /// <summary>Determines if the provider can auto-generate unique ID values that are unique for the entire datastore, rather than just for a particular class.</summary>
        /// <returns>Returns true if the provider supports id autogeneration unique across an datastore.</returns> 
        SDF_API virtual bool SupportsDataStoreScopeUniqueIdGeneration();

        /// <summary>Returns an array of the FdoDataType objects the feature provider supports for auto-generation.</summary>
        /// <param name="length">Output the number of data types.</param> 
        /// <returns>Returns the list of data types</returns> 
        SDF_API virtual FdoDataType* GetSupportedAutoGeneratedTypes(FdoInt32& length);

        /// <summary>Determines if the provider support update a schema through the ApplySchema command.
        /// If False, then an application can only create a schema once for the provider. They can use the ApplySchema 
        /// command once for a given datastore. Once the schema is defined, it cannot be changed. If True, then the provider 
        /// supports the ability to modify or add to the schema after it is defined initially.</summary>
        /// <returns>Returns true if the provider supports the update of the schema.</returns> 
        SDF_API virtual bool  SupportsSchemaModification();

		/// <summary>Determines if the provider supports inclusive value range constraints.</summary>
		/// <returns>Returns true if the provider supports inclusive value range constraints.</returns> 
		SDF_API virtual bool SupportsInclusiveValueRangeConstraints();

		/// <summary>Determines if the provider supports exclusive value range constraints.</summary>
		/// <returns>Returns true if the provider supports exclusive value range constraints.</returns>
		SDF_API virtual bool SupportsExclusiveValueRangeConstraints();

		/// <summary>Determines if the provider supports value constraints list.</summary>
		/// <returns>Returns true if the provider supports value constraints list.</returns>
		SDF_API virtual bool SupportsValueConstraintsList();

		/// <summary>Determines if the provider supports nullable value constraint.</summary>
		/// <returns>Returns true if the provider supports nullable value constraint.</returns>
		SDF_API virtual bool SupportsNullValueConstraints();

		/// <summary>Determines if the provider supports unique value constraint.</summary>
		/// <returns>Returns true if the provider supports unique value constraint.</returns>
		SDF_API virtual bool SupportsUniqueValueConstraints();

		/// <summary>Determines if the provider supports composite unique value constraint.</summary>
		/// <returns>Returns true if the provider supports composite unique value constraint.</returns>
		SDF_API virtual bool SupportsCompositeUniqueValueConstraints();

        // Returns true if the provider supports multiple identity properties per class.
        SDF_API virtual bool SupportsCompositeId();

        /// \brief
        /// Returns the maximum supported length of String, BLOB, or CLOB data
        /// properties. For decimal, it is typically the combination of the maximum
        /// scale and precision. For other data types that are not variable in size,
        /// the value returned is the byte length.
        ///
        /// \param dataType
        /// The data type for which the information is to retrieved.
        ///
        /// \return
        /// Returns the maximum data value length for the identified data type.
        ///
        SDF_API virtual FdoInt64 GetMaximumDataValueLength(FdoDataType dataType);

        /// \brief
        /// Returns the maximum supported precision for a decimal data property.
        ///
        /// \param
        /// None.
        ///
        /// \return
        /// Returns the maximum supported precision for a decimal data property.
        ///
        SDF_API virtual FdoInt32 GetMaximumDecimalPrecision();

        /// \brief
        /// Returns the maximum supported scale for a decimal data property.
        ///
        /// \param
        /// None.
        ///
        /// \return
        /// Returns the maximum supported scale for a decimal data property.
        ///
        SDF_API virtual FdoInt32 GetMaximumDecimalScale();

        /// \brief
        /// Returns the maximum size of a value of the given type. It includes
        /// limits for the data store name, shema name, class name, property name
        /// and description.
        ///
        /// \param name
        /// The schema element name type identifier for which the information is to
        /// be retrieved. Can be any of the following options: FdoSchemaElementNameType_Datastore,
        /// FdoSchemaElementNameType_Schema, FdoSchemaElementNameType_Class, FdoSchemaElementNameType_Property or 
        /// FdoSchemaElementNameType_Description
        ///
        /// \return
        /// Returns the size limitation for the identified schema element.
        ///
        SDF_API virtual FdoInt32 GetNameSizeLimit(FdoSchemaElementNameType name);

        /// \brief
        /// Returns a string that includes all the reserved characters that cannot be
        /// used for the various schema element names for the provider.
        ///
        /// \param
        /// None
        ///
        /// \return
        /// Returns a string with the reserved characters for the identified schema element.
        ///
        SDF_API virtual FdoString* GetReservedCharactersForName();

        /// \brief
        /// Returns a list of property types that can be used for identity properties.
        ///
        /// \param length
        /// Output the number of data types.
        ///
        /// \return
        /// Returns a string with the reserved characters for the identified schema element.
        ///
        SDF_API virtual FdoDataType* GetSupportedIdentityPropertyTypes(FdoInt32& length);

        /// \brief
        /// Returns TRUE if default values can be specified for a data property
        /// definition, FALSE otherwise.
        ///
        /// \param
        /// None
        ///
        /// \return
        /// Returns TRUE if default values can be specified for a data property
        /// definition, FALSE otherwise.
        ///
        SDF_API virtual bool SupportsDefaultValue();
};
